/* globals
 * GH.tpl
 */
/**
 * @module jira-agile/rapid/ui/plan/plan-rank-controller
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-statistics
 * @requires module:jira-agile/rapid/ui/plan/sprint-view
 * @requires module:jira-agile/rapid/events
 */
define('jira-agile/rapid/ui/plan/plan-onboarding', ['jquery', 'underscore', 'jira-agile/rapid/ui/plan/backlog-model', 'jira-agile/rapid/ui/plan/backlog-statistics', 'jira-agile/rapid/ui/plan/sprint-view', 'jira-agile/rapid/events'], function ($, _, BacklogModel, BacklogStatistics, SprintView, Events) {
    'use strict';

    var PlanOnboarding = {};
    Events(PlanOnboarding);

    PlanOnboarding.renderOnboarding = function () {
        var planModeData = this.getPlanModeData();
        this.updateFillYourBacklogMessage(planModeData);
        this.updatePlanYourSprintMessage(planModeData);
        this.updateCreateSprint(planModeData);
        this.updateStartSprint(planModeData);
        this.refresh();
    };

    PlanOnboarding.getPlanModeData = function () {
        // put together all required data
        var backlogModel2 = BacklogModel.getBacklogModel2();
        var backlogIssueList = backlogModel2.getIssueList();
        var planModeData = {
            issueListStats: BacklogStatistics.getIssueListStats(backlogIssueList, false),
            sprintModels: BacklogModel.getSprintModels(),
            canCreateIssue: BacklogModel.canCreateIssue(),
            canManageSprints: BacklogModel.canManageSprints(),
            sprintIssueListStats: [],
            numActiveSprints: 0,
            numFutureSprints: 0
        };
        _.each(planModeData.sprintModels, function (sprintModel) {
            planModeData.sprintIssueListStats.push(SprintView.calculateSprintIssueListStats(sprintModel, sprintModel.getIssueList()));
            if (sprintModel.sprintData.state === 'ACTIVE') {
                planModeData.numActiveSprints++;
            }
            if (sprintModel.sprintData.state === 'FUTURE') {
                planModeData.numFutureSprints++;
            }
        });
        return planModeData;
    };

    /**
     * Update FillYourBacklog onboarding message
     */
    PlanOnboarding.updateFillYourBacklogMessage = function (planModeData) {
        var $container = $('.ghx-backlog-group .ghx-onboarding-fillyourbacklog-group');
        var $backlogMessage = $('.ghx-backlog-group .ghx-helper');
        if (planModeData.issueListStats.totalCount < 3 && planModeData.sprintModels.length === 0 && planModeData.canCreateIssue) {
            // render fill your backlog onboarding
            $container.html(GH.tpl.planonboarding.renderFillYourBacklogMessage());
            $backlogMessage.addClass('hidden');
        } else {
            $container.empty();
            $backlogMessage.removeClass('hidden');
        }
    };

    PlanOnboarding.updatePlanYourSprintMessage = function (planModeData) {
        if (planModeData.canManageSprints && planModeData.numActiveSprints === 0) {
            var sprintModel = _.first(planModeData.sprintModels);
            var sprintStats = _.first(planModeData.sprintIssueListStats);
            if (sprintModel && sprintStats) {
                if (sprintModel.isFutureSprint() && sprintStats.totalCount === 0) {
                    var $containers = $('.js-sprint-container');
                    var $description = $containers.filter('[data-sprint-id="' + sprintModel.getId() + '"]').find('.ghx-helper');
                    $description.html(GH.tpl.planonboarding.renderPlanYourSprintMessage());
                }
            }
        }
    };

    /**
     * Update StartSprint button and chaperone
     */
    PlanOnboarding.updateStartSprint = function (planModeData) {
        // ensure none of the start sprint buttons are primary
        $('.js-sprint-start').removeClass('aui-button-primary');
        if (planModeData.canManageSprints && planModeData.numActiveSprints === 0) {
            // find first sprint eligible to be started
            var sprintModel = _.find(planModeData.sprintModels, function (sprintModel, index) {
                return sprintModel.sprintData.state === 'FUTURE' && planModeData.sprintIssueListStats[index].totalCount > 0;
            });
            if (sprintModel) {
                // make the associated start sprint button primary
                var sprintStartButtonSelector = '.js-sprint-container[data-sprint-id="' + sprintModel.sprintData.id + '"] .js-sprint-start';
                $(sprintStartButtonSelector).addClass('aui-button-primary');
            }
        }
    };

    /**
     * Update CreateSprint button and chaperone
     */
    PlanOnboarding.updateCreateSprint = function (planModeData) {
        var $button = $('.js-add-sprint');
        if (this.createSprintShouldBePrimary(planModeData)) {
            $button.addClass('aui-button-primary');
        } else {
            $button.removeClass('aui-button-primary');
        }
    };

    PlanOnboarding.createSprintShouldBePrimary = function (planModeData) {
        return planModeData.canManageSprints && planModeData.numFutureSprints === 0 && planModeData.numActiveSprints === 0 && planModeData.issueListStats.totalCount >= 3;
    };

    /**
     * Refresh elements (usually due to change in layout).
     */
    PlanOnboarding.refresh = function () {
        this.trigger('refresh');
    };

    return PlanOnboarding;
});

AJS.namespace('GH.planOnboarding', null, require('jira-agile/rapid/ui/plan/plan-onboarding'));