/**
 * @module jira-agile/rapid/ui/plan/plan-issue-list-view
 * @requires module:jira-agile/rapid/ui/plan/plan-drag-and-drop
 */
define('jira-agile/rapid/ui/plan/plan-issue-list-view', ['require'], function (require) {
    'use strict';

    var $ = require('jquery');
    var _ = require('underscore');

    // Data attribute keys
    var ISSUE_KEY = 'issue-key';
    var ISSUE_ID = 'issue-id';
    var FAKE_PARENT_ISSUE_KEY = 'fake-parent-issue-key';

    // Selectors
    var ISSUE_SELECTOR = '.js-issue';
    var MARKER_SELECTOR = '.ghx-marker';
    var PARENT_GROUP_SELECTOR = '.ghx-parent-group';

    // Class names
    var SUBTASK_CLASS = 'ghx-subtask';

    /**
     * Utility methods to do with plan issue lists
     */
    var PlanIssueListView = {};

    /**
     * Given an element in a list, finds the previous visible issue element in the list
     * @param {jQuery} elem the element to search from
     */
    PlanIssueListView.findPreviousVisibleIssueElementInList = function ($elem) {
        var $prevElem = $elem.prevAll(ISSUE_SELECTOR + ":visible");

        return $prevElem.length ? $prevElem : null;
    };

    /**
     * Given an element in the backlog, finds the previous issue element
     * @param {jQuery} elem the element to search from
     * @returns {jQuery} previous valid rankable element
     */
    PlanIssueListView.findPrevRankableElement = function ($elem) {
        var $prevElem = $elem.prevAll().filter(cssOrSelector(MARKER_SELECTOR, ISSUE_SELECTOR, PARENT_GROUP_SELECTOR)).not(':hidden');

        if ($prevElem.length) {
            // If the rank co-ordinate is the same as the rank of the object to be ranked,
            // we are ranking beside a fake-parent group, so skip over it. (happens when re-joining parent with fake parents group).
            if (this.getRankableId($elem) === this.getRankableId($prevElem)) {
                $prevElem = $prevElem.prev();
            }
        }

        return $prevElem;
    };

    /**
     * Given an element in the backlog, finds the next issue element
     * @param {jQuery} elem the element to search from
     * @returns {jQuery} next valid rankable element
     */
    PlanIssueListView.findNextRankableElement = function ($elem) {
        var $nextElem = $elem.nextAll().filter(cssOrSelector(MARKER_SELECTOR, ISSUE_SELECTOR, PARENT_GROUP_SELECTOR)).not(':hidden');

        if ($nextElem.length) {
            // If the rank co-ordinate is the same as the rank of the object to be ranked,
            // we are ranking beside a fake-parent group, so skip over it. (happens when re-joining parent with fake parents group).
            if (this.getRankableId($elem) === this.getRankableId($nextElem)) {
                $nextElem = $nextElem.next();
            }
        }

        return $nextElem;
    };

    /**
     * Get the key of an issue given the rendered issue element
     */
    PlanIssueListView.getIssueKey = function (elem) {
        if (elem.data(ISSUE_KEY)) {
            return elem.data(ISSUE_KEY);
        }

        return PlanIssueListView.getClosestIssueElement(elem).data(ISSUE_KEY);
    };

    /**
     * Get the id of an issue given the rendered issue element
     */
    PlanIssueListView.getIssueId = function (elem) {
        if (elem.data(ISSUE_ID)) {
            return elem.data(ISSUE_ID);
        }

        return PlanIssueListView.getClosestIssueElement(elem).data(ISSUE_ID);
    };

    /**
     * Is the rendered issue element a subtask?
     */
    PlanIssueListView.isSubtask = function (elem) {
        return PlanIssueListView.getClosestIssueElement(elem).hasClass(SUBTASK_CLASS);
    };

    /**
     * Get the key of an issue given the rendered issue element
     */
    PlanIssueListView.getRankableId = function (elem) {
        if (!elem) {
            return false;
        }

        var issueKey = elem.data(FAKE_PARENT_ISSUE_KEY) || elem.data(ISSUE_KEY);
        if (!issueKey) {
            var issue = PlanIssueListView.getClosestIssueElement(elem);
            if (issue.length > 0) {
                return issue.data(ISSUE_KEY);
            }
        }
        return issueKey;
    };

    /**
     *
     * @param {String} issueKey
     * @returns {jQuery}
     */
    PlanIssueListView.getIssueByKey = function (issueKey) {
        return $('.js-issue[data-issue-key=' + issueKey + ']');
    };

    /**
     *
     * @param {jQuery} $element
     * @returns {jQuery}
     */
    PlanIssueListView.getClosestIssueElement = function ($element) {
        return $element.closest(ISSUE_SELECTOR);
    };

    PlanIssueListView.getClosestParent = function ($element) {
        return $element.closest(PARENT_GROUP_SELECTOR);
    };

    function cssOrSelector() {
        return Array.prototype.join.call(arguments, ', ');
    }

    return PlanIssueListView;
});