AJS.test.require('com.pyxis.greenhopper.jira:gh-rapid', function() {
    var PlanContextMenuController = require('jira-agile/rapid/ui/plan/plan-context-menu-controller');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var _ = require('underscore');

    module("Plan Context Menu Controller tests", {
        setup: function() {
            sinon.stub(BacklogModel, 'isRankable').returns(true);
            this.addToContext = function(obj) {
                _.extend(this.ctx, obj);
            };

            this.ctx = {
                type: "scrum",
                selectedIssues: ['ISS-1'],
                singleIssue: true,
                backlogModel2: {
                    getId: function() {
                        return 'backlog';
                    }
                },
                currentModel: {
                    getId: function() {
                        return 'backlog';
                    }
                },
                hasBulkChangePermission: false,
                issueArchivingEnabled: false
            };
        },
        teardown: function() {
            BacklogModel.isRankable.restore();
        }
    } );

    test("module 'jira-agile/rapid/ui/plan/plan-context-menu-controller' exists", function() {
        ok(PlanContextMenuController);
        ok(GH.PlanContextMenuController);
    });

    test("Resolve context for no selected issues fails gracefully", function () {
        this.addToContext({
            selectedIssues: []
        });

        var allActions = getAllContextActions(this.ctx);

        equal(allActions.length, 0, "Should be no actions");
    });

    test("Context Menu for issue doesn't show bulk change option when user doesn't have bulk change permission", function () {
        this.addToContext({
            hasBulkChangePermission: false
        });

        var allActions = getAllContextActions(this.ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-bulk-change" }), "Bulk change option not found in context menu");
    });

    test("Context Menu for issue shows bulk change option when user has bulk change permission", function () {
        this.addToContext({
            hasBulkChangePermission: true
        });
        var allActions = getAllContextActions(this.ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-bulk-change" }), "Bulk change option found in context menu");
    });

    test('Context Menu for issue contains Print cards item', function () {
        var allActions = getAllContextActions(this.ctx);
        ok(_.findWhere(allActions, { id: 'ghx-issue-ctx-action-print-cards'}), 'Print cards item was found in context menu');
    });

    test("Context Menu for single issue contains delete option", function () {
        var allActions = getAllContextActions(this.ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item found in context menu");
    });

    test("Context Menu for multiple issues does not contain delete option", function () {
        this.addToContext({
            selectedIssues: ['ISS-1', 'ISS-2'],
            singleIssue: false
        });

        var allActions = getAllContextActions(this.ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item not found in context menu");
    });


    test("Calling actionFn for delete function calls delete shortcut method and fires analytics", function () {
        var expectedAnalytics = { name: "gh.issue.ctx.menu.action.delete" };

        sinon.stub(GH.IssueOperationShortcuts, "deleteSelectedIssue");
        sinon.stub(AJS, "trigger");

        var allActions = getAllContextActions(this.ctx);
        var deleteAction = _.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" });
        ok(deleteAction, "Delete item found in context menu");

        deleteAction.actionFn();
        ok(GH.IssueOperationShortcuts.deleteSelectedIssue.called, "Called delete shortcut method");
        ok(AJS.trigger.calledWith("analytics", sinon.match(expectedAnalytics)), "Analytics triggered");
    });

    test("Context Menu for issue shows move to option when only issues are selected.", function () {
        var allActions = getAllContextActions(this.ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-backlog" }), "Send to backlog option not found in context menu");

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-top-backlog" }), "Send to top option found in context menu");
        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-bottom-backlog" }), "Send to bottom option found in context menu");
    });

    test("Context Menu for issue shows archive action when is enabled.", function () {
        this.addToContext({
            issueArchivingEnabled: true
        });

        var allActions = getAllContextActions(this.ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-archive" }), "Archive option found in context menu");
    });

    test("Context Menu for issue does not show archive action when is disabled (default context).", function () {
        var allActions = getAllContextActions(this.ctx);
        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-archive" }), "Archive option not found in context menu");
    });

    test("Context Menu for issue doesn't show move to option when subtasks are selected.", function () {
        this.addToContext({
            containsSubtasks: true
        });

        var allActions = getAllContextActions(this.ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-backlog" }), "Send to backlog option not found in context menu");

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-top-backlog" }), "Send to top option found in context menu");
        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-bottom-backlog" }), "Send to bottom option found in context menu");
    });

    test("Context Menu for issue shows move to top option when subtasks are selected, not in the Backlog.", function () {
        this.addToContext({
            containsSubtasks: true,
            currentModel: {
                getId: function() {
                    return 'notTheBacklog';
                }
            }
        });

        var allActions = getAllContextActions(this.ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-backlog" }), "Send to backlog option found in context menu");

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-top-backlog" }), "Send to top option not found in context menu");
        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-bottom-backlog" }), "Send to bottom option not found in context menu");
    });


    function getAllContextActions(contextData) {
        var optionsArray = PlanContextMenuController.resolveActionsForContext(contextData);
        return _.flatten(_.pluck(optionsArray,"actions"));
    }
});