/* globals
 * GH, GH.RankController, GH.WorkDragAndDrop, GH.DetailsView, GH.Notification,
 */

/**
 * @module jira-agile/rapid/ui/plan/kanban-transition-and-rank
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/kanplan/kan-plan-dialog-utils
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/BacklogView
 * @requires module:jira-agile/rapid/ui/plan/plan-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-drag-and-drop
 * @requires module:jira-agile/rapid/ui/plan/plan-view
 * @requires module:jira-agile/rapid/ui/plan/analytics-helper
 * @requires module:jira/util/events
 */

define('jira-agile/rapid/ui/plan/kanban-transition-and-rank', ['require'], function (require) {
    'use strict';

    var $ = require('jquery');
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
    var PlanControls = require('jira-agile/rapid/ui/plan/plan-controls');
    var IssueMoveController = require('jira-agile/rapid/ui/plan/issue-move-controller');
    var KanPlanDialogUtils = require('jira-agile/rapid/ui/kanplan/kan-plan-dialog-utils');
    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    var Events = require('jira/util/events');
    var strings = require('jira/util/strings');
    var analyticsHelper = require('jira-agile/rapid/ui/plan/analytics-helper');
    var formatter = require('jira/util/formatter');
    var logger = require('jira/util/logger');
    var analytics = require('jira/analytics');
    var PlanDragAndDrop;
    var BacklogController;
    var BacklogView;
    var PlanView;

    var KanbanTransitionAndRank = {};

    KanbanTransitionAndRank._preInitialization = function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
        BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
        PlanDragAndDrop = require('jira-agile/rapid/ui/plan/plan-drag-and-drop');
        PlanView = require('jira-agile/rapid/ui/plan/plan-view');
    };

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', KanbanTransitionAndRank._preInitialization);

    // We need this binding for kanban backlog. When we go down a transition with a view, a dialog will pop up. The user
    // can then fill in mandatory fields and submit OR they can cancel. This basically listens for the cancel. When
    // the user cancels, we call reload to undo all the optimistic updates that have been performed so far.
    // Note, we're not listening for the successful submit event because we have a global listener for this.
    // See: GH.Dialog.XBoardExtension.handleSuccess.
    Events.bind('Dialog.hide', function (e, popup, reason, id) {
        if (GH.RapidBoard.State.isPlanMode() && GH.RapidBoard.State.isKanbanBoard() && id && id.match(/^workflow-transition-(\d+)-dialog$/) && JIRA.Dialogs[id].cancelled) {
            PlanController.reload();

            var selectedForDevSprintModel = BacklogModel.sprintModels[0];
            if (selectedForDevSprintModel) {
                var statusIdsInSelectedForDev = selectedForDevSprintModel.sprintData.column.statusIds;
                var statusToMoveTo = GH.WorkDragAndDrop.transitionTargetStatus;

                var toBacklog = statusIdsInSelectedForDev.indexOf(statusToMoveTo.toString()) === -1;
                fireCloseTransitionScreenDialogAnalytics(toBacklog);
            }
        }
    });

    function showLoadingBacklog() {
        PlanView.showLoadingBacklog();
    }

    function hideLoadingBacklog() {
        PlanView.hideLoadingBacklog();
    }

    KanbanTransitionAndRank.transitionAndRankIssues = function (issueKeys, sprintId, prevRankableId, nextRankableId) {
        if (issueKeys.length === 0) {
            return;
        }

        var issueMoveModel = IssueMoveController.calculateIssueMoveModel(issueKeys, sprintId);

        //Let's just move the issue assuming that all will be fine. If something is wrong later on, we'll just call PlanController.reload or BacklogController.reorderIssuesInSprint
        //to redo plan mode content.
        var requiresReorder = BacklogController.moveIssues(issueKeys, sprintId, prevRankableId, nextRankableId);

        var isMoveToBacklog = IssueMoveController.isMoveToBacklog(issueMoveModel);

        var request = {
            rapidViewId: GH.RapidBoard.State.getRapidViewId(),
            issueKeys: issueKeys,
            targetColumn: getTargetColumnId(issueMoveModel),
            rankCustomFieldId: BacklogModel.getRankCustomFieldId(),
            calculateNewIssuesOrder: requiresReorder,
            rankBeforeIssueIdOrKey: nextRankableId || undefined,
            rankAfterIssueIdOrKey: prevRankableId || undefined,
            activeQuickFilters: PlanControls.quickfilters.getActiveQuickFilters()
        };

        if (IssueMoveController.isOnlyRankOperation(issueMoveModel)) {
            rankIssues(request).done(function () {
                fireSuccessfulRankingAnalytics(request, issueMoveModel);
            }).fail(function () {
                fireRankingFailureAnalytics(request, issueMoveModel);
            });
        } else {
            showLoadingBacklog();

            submitTransitionAndRankRequest(request).done(function (data) {
                handleTransitionAndRankRequest(request, data, sprintId, issueMoveModel, isMoveToBacklog).always(hideLoadingBacklog);
            }).fail(function () {
                fireRankingFailureAnalytics(request, issueMoveModel);
                hideLoadingBacklog();
            });
        }
    };

    function handleTransitionAndRankRequest(request, data, sprintId, issueMoveModel, isMoveToBacklog) {
        var response = data.success;
        var promise;

        if (response.possibleTransitions.length === 0) {
            //issue has been successfully transitioned and ranked
            promise = handleSuccessfulTransitionAndRank(request, response, sprintId, issueMoveModel);
        } else if (response.possibleTransitions.length > 1) {
            //there are multiple transitions available - user needs to choose
            var targetColumnName = isMoveToBacklog ? formatter.I18n.getText('gh.rapid.plan.backlog.name') : issueMoveModel.changes[0].targetSprint.name;
            promise = handleMultipleTransitions(request, response, sprintId, issueMoveModel, targetColumnName);
        } else {
            //there's only one transition but it has a view so need to pop a dialog and get user to fill the form.
            promise = handleTransitionWithView(request, response.possibleTransitions[0], issueMoveModel);
        }

        logger.trace('jira.software.ranking.complete');

        return promise;
    }

    function submitTransitionAndRankRequest(request) {
        var deferred = $.Deferred();
        GH.Ajax.put({
            url: '/xboard/transitionAndRank',
            deferErrorHandling: true,
            data: request
        }).done(deferred.resolve).fail(function (res) {
            var err = res.error.errors && res.error.errors[0] && res.error.errors[0].message || formatter.I18n.getText('gh.error.error');

            var title = request.issueKeys.length === 1 ? formatter.I18n.getText('gh.boards.plan.kanban.transition.single.issue.error.title') : formatter.I18n.getText('gh.boards.plan.kanban.transition.multiple.issues.error.title');
            var body = '<p>' + strings.escapeHtml(err) + '</p>';

            GH.Notification.showError(title, body);

            PlanController.reload(deferred.reject);
        });

        return deferred.promise();
    }

    function handleSuccessfulTransitionAndRank(request, response, sprintId, issueMoveModel) {
        fireSuccessfulRankingAnalytics(request, issueMoveModel, response);

        PlanDragAndDrop.refreshDetailsView(issueMoveModel);

        if (request.calculateNewIssuesOrder) {
            BacklogController.reorderIssuesInSprint(response.newIssuesOrder, sprintId);
        }

        BacklogSelectionController.removeFromSelectionIssuesInDifferentModel(sprintId);

        // expand parents of subtasks if parent was not moved
        request.issueKeys.forEach(function (issueKey) {
            var issue = BacklogModel.getIssueData(issueKey);
            if (issue.parentKey && !_.contains(request.issueKeys, issue.parentKey)) {
                SubtasksExpandingController.toggleExpandStateForKey(issueKey, true);
            }
            if (issue.id) {
                AJS.$(GH).trigger("issueUpdated", { issueId: issue.id });
            }
        });

        //Used by WebDriver tests to check whether the backlog is redrawn after ranking
        PlanDragAndDrop.rankingComplete = true;
        return $.Deferred().resolve().promise();
    }

    /**
     * This function will be triggered *only if* request contains only one issue key
     * Transitioning multiple issues with transition screen is prevented higher in the logic hierarchy
     */
    function handleTransitionWithView(request, transitionWithView, issueMoveModel) {
        fireSuccessfulRankingAnalytics(request, issueMoveModel);
        fireTransitionWithScreenAnalytics(request, issueMoveModel);

        // Since we don't support transitioning multiple issues with transitions with views, this
        // will only be called if it's transitioning a single issue
        var issueId = BacklogModel.getIssueIdForKey(request.issueKeys[0]);

        //Note there's no way to hook into submit and cancel events for this server call. However, we have 2 global listeners
        //that listen to these events. For submit event listener, see GH.Dialog.XBoardExtension.handleSuccess and GH.Dialog.baseExtension.
        //For cancel event listener, see top of this file where we listen to Dialog.hide event. Currently, both handlers will reload
        //the plan controller ensuring we pull down the latest data.
        GH.WorkDragAndDrop.executeWorkflowTransition(issueId, transitionWithView.id, transitionWithView.targetStatus);
        rankIssues(request);

        return $.Deferred().resolve().promise();
    }

    function rankIssues(request) {
        var deferred = $.Deferred();
        GH.RankController.rankIssues(request.rankCustomFieldId, request.issueKeys, request.rankBeforeIssueIdOrKey, request.rankAfterIssueIdOrKey).fail(function () {
            PlanController.reload(deferred.reject);
        }).done(function (response) {
            if (response) {
                var errorRankResult = GH.RankController.getErrorRankResultFromResponse(response);
                if (errorRankResult) {
                    GH.Notification.showWarning(errorRankResult.errors[0]);
                    PlanController.reload(deferred.reject);
                } else {
                    //Used by WebDriver tests to check whether the backlog is redrawn after ranking
                    PlanDragAndDrop.rankingComplete = true;
                    logger.trace('jira.software.ranking.complete');

                    deferred.resolve();
                }
            } // else this ranking is a no-op ('from' and 'to' positions are the same)
        });

        return deferred.promise();
    }

    function handleMultipleTransitions(request, response, sprintId, issueMoveModel, targetColumnName) {
        var dialog = KanPlanDialogUtils.createTransitionStatusDialog(response.possibleTransitions, targetColumnName, request.issueKeys.length);
        var deferred = $.Deferred();

        fireMultipleTransitionDialogShowAnalytics(request, issueMoveModel, response);
        dialog.on('submit', function onSubmit(transitionId) {
            fireMultipleTransitionDialogConfirmAnalytics(request, issueMoveModel, response);

            var selectedTransition = _.find(response.possibleTransitions, function (t) {
                return t.id === parseInt(transitionId, 10);
            });

            if (selectedTransition.hasTransitionView) {
                handleTransitionWithView(request, selectedTransition, issueMoveModel).done(deferred.resolve).fail(deferred.reject);
            } else {
                request.selectedTransitionId = selectedTransition.id;
                submitTransitionAndRankRequest(request).done(function (data) {
                    KanbanTransitionAndRank.showTransitionSuccessMessage(request.issueKeys);
                    handleSuccessfulTransitionAndRank(request, data.success, sprintId, issueMoveModel);
                    deferred.resolve();
                }).fail(function () {
                    fireRankingFailureAnalytics(request, issueMoveModel);
                    deferred.reject();
                });
            }
        }, dialog);

        dialog.on('cancel', function onCancel() {
            fireMultipleTransitionDialogCancelAnalytics(request, issueMoveModel, response);

            //We've optimistically assumed that the user will proceed with the transition and rank and
            //we've already updated our client side model. There isn't an easy way to undo this, so let's
            //reload.
            PlanController.reload(deferred.reject);
        }, dialog);

        dialog.show();
        dialog.updateHeight(); // update height works only after show()

        return deferred.promise();
    }

    KanbanTransitionAndRank.showTransitionSuccessMessage = function (issueKeys) {
        if (issueKeys.length === 1) {
            GH.Notification.showSuccess(formatter.I18n.getText('gh.boards.plan.kanban.transition.single.issue.success', issueKeys));
        } else {
            GH.Notification.showSuccess(formatter.I18n.getText('gh.boards.plan.kanban.transition.multiple.issues.success', issueKeys.join(', ')));
        }
    };

    /**
     * Analytics
     */

    /**
     * Creates analytics object for ranking
     */
    function rankingDataAnalyticsObject(request, issueMoveModel) {
        var eventData = analyticsHelper.baseEventData();

        eventData.src = analyticsHelper.getSrcColumnName(issueMoveModel);
        eventData.dest = analyticsHelper.getDestColumnName(issueMoveModel);

        eventData.rankOnly = eventData.src === eventData.dest;

        eventData.issuesCount = request.issueKeys.length;
        eventData.subtasksCount = BacklogModel.filterOnlySubtasks(request.issueKeys).length;

        return eventData;
    }

    /**
     * Fire analytics event when someone close transition screen dialog
     */
    function fireCloseTransitionScreenDialogAnalytics(moveToBacklog) {
        var eventData = analyticsHelper.baseEventData();

        eventData.src = analyticsHelper.getColumnName(!moveToBacklog);
        eventData.dest = analyticsHelper.getColumnName(moveToBacklog);

        eventData.rankOnly = false;

        analytics.send({
            name: 'jira-software.plan.issuecard.transition.view.dialog.cancelled',
            properties: eventData
        });
    }

    /**
     * Triggered when issue(s) are successfully ranked - may also be transition + screen
     */
    function fireSuccessfulRankingAnalytics(request, issueMoveModel, response) {
        var eventData = rankingDataAnalyticsObject(request, issueMoveModel);
        if (response) {
            eventData.rankingTime = response.rankingTime;
            eventData.transitionTime = response.transitionTime;
        }

        analytics.send({
            name: 'jira-software.plan.issuecard.drop.success',
            properties: eventData
        });
    }

    /**
     * Trigger event when someone display multiple transitions dialog
     */
    function triggerMultipleTransitionAnalyticsEvent(name, request, issueMoveModel, response) {
        var eventData = rankingDataAnalyticsObject(request, issueMoveModel);
        eventData.optionsCount = response.possibleTransitions.length;

        analytics.send({
            name: name,
            properties: eventData
        });
    }

    /**
     * Triggered when issue(s) are successfully ranked - may also be transition + screen
     */
    function fireMultipleTransitionDialogShowAnalytics(request, issueMoveModel, response) {
        triggerMultipleTransitionAnalyticsEvent('jira-software.plan.transition-menu.view.show', request, issueMoveModel, response);
    }

    /**
     * Triggered when issue(s) are successfully ranked - may also be transition + screen
     */
    function fireMultipleTransitionDialogConfirmAnalytics(request, issueMoveModel, response) {
        triggerMultipleTransitionAnalyticsEvent('jira-software.plan.transition-menu.view.confirm', request, issueMoveModel, response);
    }

    /**
     * Triggered when issue(s) are successfully ranked - may also be transition + screen
     */
    function fireMultipleTransitionDialogCancelAnalytics(request, issueMoveModel, response) {
        triggerMultipleTransitionAnalyticsEvent('jira-software.plan.transition-menu.view.cancel', request, issueMoveModel, response);
    }

    /**
     * Triggered when transition screen is displayed
     */
    function fireTransitionWithScreenAnalytics(request, issueMoveModel) {
        var eventData = rankingDataAnalyticsObject(request, issueMoveModel);

        analytics.send({
            name: 'jira-software.plan.transition.screen.view',
            properties: eventData
        });
    }

    /**
     * Triggered when there is an error when someone transition/rank issues
     */
    function fireRankingFailureAnalytics(request, issueMoveModel) {
        var eventData = rankingDataAnalyticsObject(request, issueMoveModel);

        analytics.send({
            name: 'jira-software.plan.issuecard.drop.failure',
            properties: eventData
        });
    }

    function getTargetColumnId(issueMoveModel) {
        var isAddToBacklog = IssueMoveController.isMoveToBacklog(issueMoveModel);
        var targetColumn;
        if (isAddToBacklog) {
            targetColumn = BacklogModel.getBacklogModel2().getColumn();
        } else {
            targetColumn = BacklogModel.getSprintModels()[0].sprintData.column;
        }
        return targetColumn.id;
    }

    return KanbanTransitionAndRank;
});