/* globals
 * GH.RemoveIssuesFromSprintDialog, GH.AddIssuesToActiveSprintDialog
 */

/**
 * @module jira-agile/rapid/ui/plan/issue-move-controller
 * @requires module:underscore
 * @requires module:jquery
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-statistics
 * @requires module:jira-agile/rapid/ui/plan/sprint-model
 */
define('jira-agile/rapid/ui/plan/issue-move-controller', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogStatistics = require('jira-agile/rapid/ui/plan/backlog-statistics');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var BacklogController;

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    var IssueMoveController = {};

    IssueMoveController.confirmMove = function (issueMoveModel) {
        /**
         * Checks whether a confirmation is required and asks the user to confirm. Calls onConfirmFn once the operation is complete.
         */
        var deferred = $.Deferred();

        // first make sure we actually have to show a dialog. directly return otherwise
        var isRemoveFromActiveSprint = IssueMoveController.isMoveFromActiveSprints(issueMoveModel);
        var isAddToActiveSprint = IssueMoveController.isAddToActiveSprint(issueMoveModel);

        if (isRemoveFromActiveSprint) {
            // show dialog
            GH.RemoveIssuesFromSprintDialog.constructDialog(issueMoveModel, deferred);
        } else if (isAddToActiveSprint) {
            GH.AddIssuesToActiveSprintDialog.showDialog(issueMoveModel, deferred);
        } else {
            // directly resolve
            deferred.resolve(issueMoveModel);
        }

        return deferred;
    };

    /**
     * Get the sprint rank model returned by the remove from sprint check. It contains everything needed to
     * figure out whether a confirmation dialog should be shown and what dialog that is.
     */
    IssueMoveController.calculateIssueMoveModel = function (issueKeys, sprintId) {
        /*
         * BIG FAT NOTE: At the moment this method assumes that all issue keys come from the same sprint/backlog,
         *               which is OBVIOUSLY NOT the case when moving the marker. Can't be bothered rewriting the logic
         *               anymore. For this case (drag from future) we move all issues at once anyways so it won't matter.
         */

        var origModel = null,
            targetModel = null;
        var origIsSprint = false,
            targetIsSprint = false;
        _.each(BacklogModel.getAllModels(), function (model) {
            var isSprintModel = SprintModel.isSprintModel(model);
            if (model.getIssueList().areIssuesValid(issueKeys)) {
                origModel = model;
                origIsSprint = isSprintModel;
            }
            if (sprintId && isSprintModel && model.getSprintId() === sprintId) {
                targetModel = model;
                targetIsSprint = true;
            } else if (!sprintId && !isSprintModel) {
                targetModel = model;
                targetIsSprint = false;
            }
        });

        // if we move from an active sprint to a future sprint / backlog, we have to split issues into done/not done,
        // otherwise we move all issues in one go.
        var activeToFuture = origIsSprint && !origModel.isFutureSprint() && (!targetIsSprint || targetModel.isFutureSprint());
        var notDoneKeys = [],
            doneKeys = [];
        if (!activeToFuture) {
            // move all issues
            notDoneKeys = issueKeys;
        } else {
            // extract the issue data
            var issues = [];
            var issueList = origModel.getIssueList();
            _.each(issueKeys, function (issueKey) {
                var data = issueList.getIssueData(issueKey);
                if (data) {
                    issues.push(data);
                }
            });

            // split issue keys into done and not done
            var columns = BacklogController.rapidViewConfig.columns;
            var split = BacklogStatistics.splitIssuesIntoNotDoneAndDone(issues, columns);
            notDoneKeys = _.map(split.notDone, function (data) {
                return data.key;
            });
            doneKeys = _.map(split.done, function (data) {
                return data.key;
            });
        }

        return {
            changes: [{
                currentSprint: origIsSprint ? origModel.getSprintData() : null,
                targetSprint: targetIsSprint ? targetModel.getSprintData() : null,
                issueKeys: notDoneKeys,
                doneIssueKeys: doneKeys
            }]
        };
    };

    /**
     * Is this a move between active sprints?
     */
    IssueMoveController.isMoveBetweenActiveSprints = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            var currentSprintIsActive = change.currentSprint && change.currentSprint.state != 'FUTURE';
            var targetSprintIsActive = change.targetSprint && change.targetSprint.state != 'FUTURE';
            return currentSprintIsActive && targetSprintIsActive;
        });
    };

    /**
     * Is the passed model a move between active/closed sprints?
     */
    IssueMoveController.isMoveFromActiveSprints = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            var currentSprintIsActive = change.currentSprint && change.currentSprint.state != 'FUTURE';
            var targetSprintIsCurrentSprint = change.currentSprint && change.targetSprint && change.currentSprint.id === change.targetSprint.id;
            return currentSprintIsActive && !targetSprintIsCurrentSprint;
        });
    };

    /**
     * Is this a ranking operation only?
     */
    IssueMoveController.isOnlyRankOperation = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            var targetSprintIsCurrentSprint = change.currentSprint && change.targetSprint && change.currentSprint.id === change.targetSprint.id;
            var targetBacklogIsCurrentBacklog = !change.currentSprint && !change.targetSprint;
            return targetSprintIsCurrentSprint || targetBacklogIsCurrentBacklog;
        });
    };

    /**
     * Is this a rank between future sprints and backlog?
     */
    IssueMoveController.isRankBetweenFutureAndBacklog = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            var currentIsBacklogOrFuture = !change.currentSprint || change.currentSprint.state === 'FUTURE';
            var targetIsBacklogOrFuture = !change.targetSprint || change.targetSprint.state === 'FUTURE';
            return currentIsBacklogOrFuture && targetIsBacklogOrFuture;
        });
    };

    IssueMoveController.isMoveToBacklog = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            var currentIsNotBacklog = !!change.currentSprint;
            var targetIsBacklog = !change.targetSprint;
            return currentIsNotBacklog && targetIsBacklog;
        });
    };

    /**
     * Check if user moves issues from backlog
     * @param issueMoveModel
     * @returns {boolean}
     */
    IssueMoveController.isSourceBacklog = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            return !change.currentSprint;
        });
    };

    /**
     * Check if user moves issues to backlog
     * @param issueMoveModel
     * @returns {boolean}
     */
    IssueMoveController.isTargetBacklog = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            return !change.targetSprint;
        });
    };

    IssueMoveController.isAddToActiveSprint = function (issueMoveModel) {
        return _.some(issueMoveModel.changes, function (change) {
            var targetSprintIsActiveSprint = change.targetSprint && change.targetSprint.state !== "FUTURE";
            var targetSprintIsNotCurrentSprint = !change.currentSprint || !change.targetSprint || change.currentSprint.id !== change.targetSprint.id;
            return targetSprintIsNotCurrentSprint && targetSprintIsActiveSprint;
        });
    };

    IssueMoveController.getAllIssueKeys = function (issueMoveModel) {
        var issueKeys = [];
        _.each(issueMoveModel.changes, function (change) {
            issueKeys.push(change.issueKeys);
            issueKeys.push(change.doneIssueKeys);
        });
        return _.flatten(issueKeys, true);
    };

    return IssueMoveController;
});