/* globals
 * GH.Ajax, GH.Notification
 */

/**
 * @module jira-agile/rapid/ui/plan/issue-actions
 * @requires module:underscore
 * @requires module:jquery
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/plan/issue-move-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-drag-and-drop
 */
define('jira-agile/rapid/ui/plan/issue-actions', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var IssueMoveController = require('jira-agile/rapid/ui/plan/issue-move-controller');
    var BacklogController;
    var PlanDragAndDrop;

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
        PlanDragAndDrop = require('jira-agile/rapid/ui/plan/plan-drag-and-drop');
    });

    var IssueActions = {};
    IssueActions.Analytics = {};
    IssueActions.Analytics.removeFromSprint = new AnalyticsTracker("gh.sprint.issue.remove");

    /**
     * Removes issues from the current sprint
     */
    IssueActions.removeIssuesFromSprint = function (rapidViewId, issueKeys) {
        if (!_.isArray(issueKeys)) {
            issueKeys = [issueKeys];
        }

        // check if removing the issues from the active sprints will inadvertedly close it. (removing all issues from a sprint will do this)
        GH.Ajax.get({
            url: '/sprint/rank/' + rapidViewId + '/remove',
            data: {
                issues: issueKeys
            }
        }, "removeIssuesFromActiveSprintsCheck").done(function (removeFromSprintModel) {

            // check for removal
            var deferred = IssueMoveController.confirmMove(removeFromSprintModel);

            // do the actual removal
            deferred.done(IssueActions.removeIssuesFromCurrentSprints);
        });
    };

    /**
     * Removes the given issues from any active sprints they belong to
     *
     * @return {$.Deferred.promise} a promise resolved after the issues are removed and the ui updated
     */
    IssueActions.removeIssuesFromCurrentSprints = function (removeFromSprintModel) {
        var issueKeys = IssueMoveController.getAllIssueKeys(removeFromSprintModel);

        return GH.Ajax.put({
            url: '/sprint/rank/' + removeFromSprintModel.rapidViewId + '/remove',
            data: {
                rapidViewId: BacklogController.rapidViewData.id,
                idOrKeys: issueKeys
            }
        }).done(function (result) {
            var removeFromSprintModel = result.success;

            // one issue means that may be it was called from the details view cog menu, so may be there was a multi
            // selection that we should replace with a single selection
            var issueKeys = IssueMoveController.getAllIssueKeys(removeFromSprintModel);

            // show a success message
            IssueActions.showSprintRankSuccessMessage(removeFromSprintModel);

            // fire an event indicating that the issue got removed from the sprint. let work/plan handle this specially
            $(GH).trigger('issuesRemovedFromSprint', { issueKeys: issueKeys });

            // fire analytics event
            IssueActions.Analytics.removeFromSprint.trigger("remove", "removedIssuesCount", issueKeys.length);
        });
    };

    /**
     * Shows a sprint rank success message according to the model
     */
    IssueActions.showSprintRankSuccessMessage = function (removeFromSprintModel) {
        // stop here if we just rank
        if (IssueMoveController.isOnlyRankOperation(removeFromSprintModel)) {
            return;
        }

        // also don't show a message for ranking between backlog and future sprints
        if (IssueMoveController.isRankBetweenFutureAndBacklog(removeFromSprintModel)) {
            return;
        }

        // show moved message
        GH.Notification.showSuccess(AJS.I18n.getText('gh.sprint.issue.move.success'));
    };

    /**
     * Returns a link to the view issue page for a given issue key
     */
    IssueActions._createIssueKeysLinkHtml = function (issueKeys) {
        if (!_.isArray(issueKeys)) {
            issueKeys = [issueKeys];
        }

        var issueKeysHtml = _.map(issueKeys, function (issueKey) {
            return '<a href="' + GH.Ajax.CONTEXT_PATH + '/browse/' + issueKey + '">' + AJS.escapeHTML(String(issueKey)) + '</a>';
        });

        return issueKeysHtml.join(", ");
    };

    /**
     * Send issues to to a specific sprint
     */
    IssueActions.sendIssuesToBottomOfSprint = function (issueKeys, sprintModel) {
        var lastIssueKey = _.last(sprintModel.getIssueList().getOrder());
        return PlanDragAndDrop.rankIssues(issueKeys, sprintModel.getId(), lastIssueKey, null);
    };

    /**
     * Send issues to to a specific sprint without ranking
     */
    IssueActions.sendIssuesToSprint = function (issueKeys, sprintModel) {
        return PlanDragAndDrop.rankIssues(issueKeys, sprintModel.getId(), null, null);
    };

    /**
     * Send issues to the top of the backlog
     */
    IssueActions.sendIssuesToTopOfBacklog = function (issueKeys, backlogModel2) {
        var issueList = backlogModel2.getIssueList();
        var firstIssueKey = issueList.getParentKey(_.first(issueList.getOrder()));
        PlanDragAndDrop.rankIssues(issueKeys, null, null, firstIssueKey);
    };

    /**
     * Send issues to the bottom of the backlog
     */
    IssueActions.sendIssuesToBottomOfBacklog = function (issueKeys, backlogModel2) {
        var issueList = backlogModel2.getIssueList();
        var lastIssueKey = issueList.getParentKey(_.last(issueList.getOrder()));
        PlanDragAndDrop.rankIssues(issueKeys, null, lastIssueKey, null);
    };

    /**
     * Send issues to the backlog without ranking
     */
    IssueActions.sendIssuesToBacklog = function (issueKeys) {
        PlanDragAndDrop.rankIssues(issueKeys, null, null, null);
    };

    /**
     * Flags/unflags an issue
     */
    IssueActions.flag = function (issueKeys, flag, comment, commentVisibility, formToken, temporaryAttachments) {
        if (!_.isArray(issueKeys)) {
            issueKeys = [issueKeys];
        }

        return GH.Ajax.post({
            url: '/xboard/issue/flag/flag.json',
            data: {
                issueKeys: issueKeys,
                flag: flag,
                comment: comment,
                commentVisibility: commentVisibility,
                formToken: formToken,
                temporaryAttachments: temporaryAttachments
            }

        }).done(function (response) {
            if (_.has(response.success, 'errorMessages')) {
                GH.Notification.showWarnings(_.map(response.success.errorMessages, function (message) {
                    return {
                        autoHideDelay: 10000,
                        message: message,
                        noAutoescape: true
                    };
                }), { showTitle: true });
            }
        });
    };

    /**
     * Show a set of issues in the issue navigator
     */
    IssueActions.viewInIssueNavigator = function (issueKeys) {
        var url = GH.Ajax.CONTEXT_PATH + "/secure/GHViewInNavigator.jspa?issueKeys=" + encodeURIComponent(issueKeys.join(","));
        location.href = url;
    };

    /**
     * Invokes bulk change with a set of issues
     */
    IssueActions.bulkChange = function (issueKeys) {
        var url = GH.Ajax.CONTEXT_PATH + "/secure/GHBulkChange.jspa?issueKeys=" + encodeURIComponent(issueKeys.join(","));
        location.href = url;
    };

    return IssueActions;
});