/* globals
 * GH.sprintActions, GH.planOnboarding, GH.Notification, GH.NumberFormat, GH.ScrollUtils
 */

/**
 * @module jira-agile/rapid/ui/plan/BacklogView
 * @requires module:underscore
 * @requires module:jquery
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/ui/plan/backlog-model2
 * @requires module:jira-agile/rapid/ui/plan/sprint-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-backlog-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-drag-and-drop
 */
define('jira-agile/rapid/ui/plan/BacklogView', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var DynamicColors = require('jira-agile/rapid/ui/component/dynamic-colors');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogModel2 = require('jira-agile/rapid/ui/plan/backlog-model2');
    var SprintView = require('jira-agile/rapid/ui/plan/sprint-view');
    var SprintBacklogView = require('jira-agile/rapid/ui/plan/sprint-backlog-view');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    var PlanDragAndDrop = require('jira-agile/rapid/ui/plan/plan-drag-and-drop');
    var FeatureFlagManager = require('jira/featureflags/feature-manager');
    var BacklogController;

    var FILTERED_ISSUE_CLASS = "ghx-filtered";

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    var BacklogView = {};

    BacklogView.containerSelector = '#ghx-backlog';

    BacklogView.getContainer = function () {
        if (!BacklogView.container) {
            BacklogView.container = $(BacklogView.containerSelector);
        }
        return BacklogView.container;
    };
    /**
     * Draw the Backlog.
     */
    BacklogView.draw = function () {
        var drawStart = new Date().getTime();
        GH.Logger.log("BacklogView.draw", GH.Logger.Contexts.ui);

        // clear out the current view
        BacklogView.container = undefined;
        var $backlogViewContainer = $(BacklogView.containerSelector);
        $backlogViewContainer.html(GH.tpl.backlogview.renderBacklogView({
            cardColorStrategy: BacklogModel.getCardColorStrategy()
        }));

        var issueRenderData = BacklogController.calculateIssueRenderData();

        // draw all sprints
        SprintView.renderAllSprints(issueRenderData);

        // draw backlog
        SprintBacklogView.renderBacklog(issueRenderData);

        // draw onboarding
        GH.planOnboarding.renderOnboarding();

        // Delay event bindings until after rendering for performance reasons
        _.delay(function () {
            // enable d&d
            PlanDragAndDrop.enableDragAndDrop();

            // register click handlers
            BacklogView.registerClickHandlers();

            // register twixies for meta data disclosure
            SprintView.registerTwixies();

            GH.sprintActions.createDropdown();

            DynamicColors.update();

            // set the "rendered" attribute to indicate when the eagle has land... err, board has loaded
            $('#ghx-backlog').attr("data-rendered", new Date().getTime()).attr("data-draw-time", new Date().getTime() - drawStart);
        });
    };

    BacklogView.registerClickHandlers = function () {
        // listener for filter clearing
        $(BacklogView.containerSelector).delegate('.js-clear-all-filters', 'click', function (event) {
            BacklogController.clearFilters();
        });
    };

    BacklogView.redrawChangedModel = function (model) {
        BacklogView.redrawChangedModels([model]);
    };

    BacklogView.redrawChangedModels = function (models) {
        GH.PlanView.showLoadingBacklog();

        setTimeout(function () {
            // rerender each model
            _.each(models, function (model) {
                if (SprintModel.isSprintModel(model)) {
                    SprintView.updateSprintViewForModel(model);
                } else if (BacklogModel2.isBacklogModel(model)) {
                    SprintBacklogView.updateBacklog();
                }
            });

            GH.planOnboarding.renderOnboarding();

            DynamicColors.update();

            // re-enable drag and drop for backlog issue list
            PlanDragAndDrop.enableDragAndDrop();

            // re-create Dropdown for sprint actions
            GH.sprintActions.createDropdown();

            GH.PlanView.hideLoadingBacklog();
        });
    };

    BacklogView.updateSprintsAfterAddRemove = function () {
        var issueRenderData = BacklogController.calculateIssueRenderData();

        // draw all sprints
        SprintView.renderAllSprints(issueRenderData);

        GH.planOnboarding.renderOnboarding();

        DynamicColors.update();

        // re-enable drag and drop for backlog issue list
        PlanDragAndDrop.enableDragAndDrop();

        // re-create Dropdown for sprint actions
        GH.sprintActions.createDropdown();
    };

    /**
     * Updates the selection of the ui
     */
    BacklogView.updateIssueSelectionState = function (doScroll) {
        var scrollToIssueKey = false;
        var scrollToViewElem = null;
        var selectedIssue = BacklogSelectionController.getSelectedIssueKey();
        scrollToIssueKey = selectedIssue;
        $(BacklogView.containerSelector).find('.js-issue').removeClass('ghx-selected ghx-selected-primary').each(function (index, elem) {
            var $elem = $(elem);
            var issueKey = $elem.data('issue-key');
            if (BacklogSelectionController.isInSelection(issueKey)) {
                $elem.addClass('ghx-selected');
                if (selectedIssue == issueKey) {
                    $elem.addClass('ghx-selected-primary');
                }
            }
            if (scrollToIssueKey != null && issueKey === scrollToIssueKey) {
                scrollToViewElem = $elem;
            }
        });

        // scroll the currently selected issue to view
        // this is a slow operation so only do it if necessary
        // not on mouse click
        if (scrollToViewElem && doScroll) {
            BacklogView._scrollToView(scrollToViewElem);
        }
    };

    /**
     * Updates the visibility of all issues and container statistics
     *
     * Re-drawing the whole backlog is slow
     * We instead hide the container div, update the display property on all the elements and re-show the container
     * This means we only do two re-flows, instead of one per issue changed
     * Using .css instead of .hide() / .show() is much faster, as .hide/.show have to store the current display information
     */
    BacklogView.updateHiddenIssues = function () {
        // hide the backlog
        var $backlog = $("#ghx-backlog");
        var isVerticalSplitEnabled = FeatureFlagManager.isFeatureEnabled('com.atlassian.jira.agile.darkfeature.backlog.showmore');
        $backlog.css({ 'display': 'none' });

        // fetch the issueRenderData which contains information about hidden issues
        var issueRenderData = BacklogController.calculateIssueRenderData();
        // update all issue lists
        var containers = $backlog.find('.js-issue-list');
        _.each(containers, function (container) {
            var $issuesContainer = $(container);

            // no sense to update visibility of issues in real backlog
            // if we are going to re-render it
            if (isVerticalSplitEnabled && $issuesContainer.closest('.ghx-backlog-group').length) {
                return;
            }

            // if there are no issues there is nothing to do
            if ($issuesContainer.is('.js-empty-list')) {
                return;
            }

            // update issue visibilities
            var allFiltered = BacklogView._updateIssuesVisibilityInIssueList($issuesContainer, issueRenderData);

            // set flag according to all issues beening filtered
            if (allFiltered) {
                $issuesContainer.removeClass('ghx-has-issues').addClass('ghx-no-issues');
            } else {
                $issuesContainer.removeClass('ghx-no-issues').addClass('ghx-has-issues');
            }
        });

        // update the statistics of all sprints
        SprintView.updateSprintStatistics();

        // update the backlog header
        SprintBacklogView.updateBacklogHeader();

        SprintBacklogView.setShowAllClicked(false);

        if (isVerticalSplitEnabled) {
            GH.PlanView.showLoadingBacklog();

            setTimeout(function () {
                SprintBacklogView.updateBacklog();
                GH.planOnboarding.renderOnboarding();
                $backlog.css({ 'display': '' });
                DynamicColors.update();
                PlanDragAndDrop.enableDragAndDrop();
                GH.PlanView.hideLoadingBacklog();
            });
        } else {
            GH.planOnboarding.renderOnboarding();

            $backlog.css({ 'display': '' });
        }
    };

    /**
     * updates the issue visibility and returns whether all issues are filtered
     * @return boolean true if all issues are hidden
     */
    BacklogView._updateIssuesVisibilityInIssueList = function ($container, issueRenderData) {
        // update the backlog
        var $issues = $container.find(".js-issue");
        var allHidden = true;
        _.each($issues, function (issue) {
            var $issue = $(issue);
            var issueKey = $issue.attr('data-issue-key');
            if (issueRenderData.hiddenIssues[issueKey]) {
                $issue.addClass(FILTERED_ISSUE_CLASS);
            } else {
                $issue.removeClass(FILTERED_ISSUE_CLASS);
                allHidden = false;
            }
        });

        //show/hide containers without issues
        var $fakeContainers = $container.find('.ghx-parent-group');
        $fakeContainers.each(function (index, fakeContainer) {
            var $fakeContainer = $(fakeContainer);

            var $subtasks = $fakeContainer.find('.ghx-issue-subtask');
            var $parentStub = $fakeContainer.find('.ghx-parent-stub');

            // all subtasks are filtered - do not show stub
            var allFiltered = $subtasks.not('.' + FILTERED_ISSUE_CLASS).length === 0;
            $parentStub.toggleClass(FILTERED_ISSUE_CLASS, allFiltered);

            // hidden subtasks counter
            var $counterContainer = $fakeContainer.find('.ghx-hidden-subtasks-count-container');
            var hiddenByQuickFilters = parseInt($counterContainer.attr('data-subtasks-hidden-by-filters'), 10);
            var hiddenBySearch = $subtasks.filter('.ghx-filtered').length;
            var hiddenInTotal = hiddenByQuickFilters + hiddenBySearch;

            $counterContainer.html(GH.tpl.planissuelist.renderSubtasksFilteredContent({
                hiddenSubtasks: hiddenInTotal,
                allHidden: allFiltered
            }));
        });

        return allHidden;
    };

    /** Inline Issue Create expects this exact method to be called in its Acceptance tests, so we cant remove it **/
    BacklogView._scrollToView = function ($elem) {
        var $backlog = $('#ghx-backlog');

        // bypass jQuery which is a pain for that kind of work. getBoundingClientRect is exactly what we need and is supported on all our supported browsers.
        // we assume that the document is not scrollable by design, so we don't have to manage document scroll offset
        var backlogRect = $backlog[0].getBoundingClientRect();
        var elementRect = $elem[0].getBoundingClientRect();

        GH.ScrollUtils.scrollElementToViewPort($backlog, backlogRect, elementRect);
    };

    return BacklogView;
});