/* globals
 * GH.TimeFormat, GH.NumberFormat
 */

/**
 * Calculation of statistics in the backlog
 * @module jira-agile/rapid/ui/plan/backlog-statistics
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/plan/backlog-statistics', [
    'underscore',
    'require'
], function(
    _,
    require
) {
    'use strict';

    var GlobalEvents = require('jira-agile/rapid/global-events');
    var BacklogModel;

    // Resolve circular dependency
    GlobalEvents.on("pre-initialization", function () {
        BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    });

    var BacklogStatistics = {
        EXCLUDE_SUBTASKS_COLUMN_CONSTRAINT: 'issueCountExclSubs'
    };

    /**
     * Statistics we don't calculate/show separately
     */
    BacklogStatistics.ignoredStatisticTypes = {
        issueCount: true,
        none: true
    };

    /**
     * Constructs a sprint marker object that we use for rendering in a few places.
     */
    BacklogStatistics.getIssueListEstimateStatistics = function (issueList) {
        // calculate the raw numbers

        var stats = BacklogStatistics._calculateSprintModelStatistics(issueList);

        // estimate stats
        var estimateField = BacklogModel.estimationStatistic;
        var estimateStats = {};
        estimateStats.shown = (estimateField &&
        estimateField.isEnabled &&
            // only show statistic if it is not issueCount
        !BacklogStatistics.ignoredStatisticTypes[estimateField.typeId]
        ) || false;
        if (estimateStats.shown) {
            estimateStats.name = estimateField ? estimateField.name : "";
            estimateStats.total = {
                raw: stats.estimate.total,
                formatted: BacklogStatistics.formatStatisticForRendering(estimateField, stats.estimate.total)
            };
            estimateStats.visible = {
                raw: stats.estimate.visible,
                formatted: BacklogStatistics.formatStatisticForRendering(estimateField, stats.estimate.visible)
            };
        }

        // tracking stats
        var trackingField = BacklogModel.trackingStatistic;
        var trackingStats = {};
        trackingStats.shown = (trackingField &&
        trackingField.isEnabled &&
            // only show statistic if it is not issueCount
        !BacklogStatistics.ignoredStatisticTypes[estimateField.typeId]
            // only show if different from estimate statistic
        && ((stats.estimate.total != stats.tracking.total) || (stats.estimate.visible != stats.tracking.visible))
        ) || false;
        if (trackingStats.shown) {
            trackingStats.name = trackingField ? trackingField.name : "";
            trackingStats.total = {
                raw: stats.tracking.total,
                formatted: BacklogStatistics.formatStatisticForRendering(trackingField, stats.tracking.total)
            };
            trackingStats.visible = {
                raw: stats.tracking.visible,
                formatted: BacklogStatistics.formatStatisticForRendering(trackingField, stats.tracking.visible)
            };
        }

        return {
            estimate: estimateStats,
            tracking: trackingStats
        };
    };

    /**
     * Update the tally of statistics field based on the current position of the marker and return new marker.
     * @param issueList
     */
    BacklogStatistics._calculateSprintModelStatistics = function (issueList) {
        var issuesInSprint = issueList.getAllIssues();

        var totalEstimateStatistic = 0, visibleEstimateStatistic = 0,
            totalTrackingStatistic = 0, visibleTrackingStatistic = 0,
            totalIssueCount = 0, visibleIssueCount = 0;

        var tmpVal;
        _.each(issuesInSprint, function (issue) {
            var isIssueVisible = issueList.isIssueVisible(issue.key);

            // estimate
            if (!_.isUndefined(issue.estimateStatistic) && !_.isUndefined(issue.estimateStatistic.statFieldValue.value)) {
                tmpVal = issue.estimateStatistic.statFieldValue.value;
                totalEstimateStatistic += tmpVal;
                if (isIssueVisible) {
                    visibleEstimateStatistic += tmpVal;
                }
            }
            // tracking
            if (!_.isUndefined(issue.trackingStatistic) && !_.isUndefined(issue.trackingStatistic.statFieldValue.value)) {
                tmpVal = issue.trackingStatistic.statFieldValue.value;
                totalTrackingStatistic += tmpVal;
                if (isIssueVisible) {
                    visibleTrackingStatistic += tmpVal;
                }
            }
            totalIssueCount++;
            if (isIssueVisible) {
                visibleIssueCount++;
            }
        });

        return {
            estimate: {
                total: totalEstimateStatistic,
                visible: visibleEstimateStatistic
            },
            tracking: {
                total: totalTrackingStatistic,
                visible: visibleTrackingStatistic
            },
            issueCount: {
                total: totalIssueCount,
                visible: visibleIssueCount
            }
        };
    };

    /**
     * Format a statistic value depending on the configured estimation statistic
     */
    BacklogStatistics.formatStatisticForRendering = function (statistic, value) {
        if (statistic.renderer == 'duration') {
            return GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(value);
        } else {
            return GH.NumberFormat.format(value);
        }
    };


    /**
     * Returns all issues which are before the specified rankable ID
     * @param rankableId the ID of the rankable (marker or issue key)
     * @return {Array} issue objects
     */
    BacklogStatistics.getAllIssuesBeforeIndex = function (issues, order, startingPos, condition) {
        var issueKeysBefore = [];
        condition = condition || function (index) {
            return !BacklogStatistics._isRankableId(order[index]);
        };

        // go back from the startingPos, stopping either when we hit 0 or condition is false (ie: when we hit another marker)
        for (var i = startingPos - 1; i >= 0 && condition(i); i--) {
            // prepend, not append (as we're in reverse order)
            issueKeysBefore = [order[i]].concat(issueKeysBefore);
        }

        return BacklogStatistics._mapIssueKeysToIssues(issues, issueKeysBefore);
    };

    /**
     * Determine if the rankable ID is a marker ID (i.e. is a number)
     * @param rankableId
     * @return {*}
     * @private
     */
    BacklogStatistics._isRankableId = function (rankableId) {
        return _.isNumber(rankableId);
    };


    BacklogStatistics._mapIssueKeysToIssues = function (issues, issueKeys) {
        var mappedValues = _.map(issueKeys, function (key) {
            return issues[key];
        });
        return _.filter(mappedValues, function (value) {
            return !_.isUndefined(value);
        });
    };

//
// Data Calculation stuff
//


    /**
     * Calculates issue statistics of an issue list
     * @param issueList
     * @return {{totalCount: number, visibleCount: number, allVisible: boolean, noneVisible: boolean, hasIssues: boolean, empty: boolean, emptyMessage: string}}
     */
    BacklogStatistics.getIssueListStats = function (issueList, isSprint, isActiveSprint) {
        var issues = issueList.getAllIssues();
        var totalIssueCount = _.keys(issues).length;
        var visibleIssueCount = issueList.getNumVisibleIssues();
        var hasIssues = totalIssueCount > 0;

        var emptyMessage;
        if (isSprint) {
            if (isActiveSprint) {
                emptyMessage = BacklogStatistics.getActiveSprintEmptyMessage(!hasIssues);
            } else {
                emptyMessage = BacklogStatistics.getFutureSprintEmptyMessage(!hasIssues);
            }
        } else {
            emptyMessage = BacklogStatistics.getBacklogEmptyMessage(!hasIssues);
        }

        return {
            totalCount: totalIssueCount,
            visibleCount: visibleIssueCount,
            allVisible: totalIssueCount === visibleIssueCount,
            noneVisible: visibleIssueCount === 0,
            hasIssues: hasIssues,
            empty: !hasIssues,
            emptyMessage: emptyMessage
        };
    };

    /**
     * Calculate additional statistics fields values (number of issues visible)
     */
    BacklogStatistics.calculateStatisticsFieldValue = function(issueList, statisticsConfig) {
        if (statisticsConfig.typeId === BacklogStatistics.EXCLUDE_SUBTASKS_COLUMN_CONSTRAINT) {
            var statisticsFieldValue = issueList.getIssuesExcludingSubtasks().length;
            var visibleIssueList = issueList.getVisibleIssuesBeforeRankable(false);
            var visibleIssueExcludingSubtasksList = visibleIssueList.filter(function (issueKey) {
                return _.isUndefined(issueList.getIssueData(issueKey).parentId);
            });

            return {
                statisticsFieldTotal: statisticsFieldValue,
                statisticsFieldVisible: visibleIssueExcludingSubtasksList.length,
                excludeSubtasks: true,
                subtasksTotal: _.keys(issueList.getAllIssues()).length - statisticsFieldValue,
                subtasksVisible: visibleIssueList.length - visibleIssueExcludingSubtasksList.length
            }
        } else {
            return {
                statisticsFieldTotal: _.keys(issueList.getAllIssues()).length,
                statisticsFieldVisible: issueList.getNumVisibleIssues()
            };
        }
    };



    BacklogStatistics.getBacklogEmptyMessage = function (hasNoIssues) {
        if (hasNoIssues) {
            return AJS.I18n.getText('gh.rapid.plan.backlog.empty.help');
        }
        else {
            return AJS.I18n.getText('gh.rapid.plan.backlog.filtered.help');
        }
    };

    BacklogStatistics.getFutureSprintEmptyMessage = function (hasNoIssues) {
        if (hasNoIssues) {
            if (GH.RapidBoard.State.isScrumBoard()) {
                return BacklogModel.isRankable() ? AJS.I18n.getText('gh.rapid.plan.sprint.empty.rankable.help') :
                    AJS.I18n.getText('gh.rapid.plan.sprint.empty.nonrankable.help')
            } else {
                return AJS.I18n.getText('gh.rapid.plan.kanban.selected.for.dev.empty.help');
            }
        }
        else {
            return GH.RapidBoard.State.isScrumBoard() ? AJS.I18n.getText('gh.rapid.plan.sprint.filtered.help') :
                AJS.I18n.getText('gh.rapid.plan.kanban.selected.for.dev.filtered.help');
        }
    };

    BacklogStatistics.getActiveSprintEmptyMessage = function (hasNoIssues) {
        if (hasNoIssues) {
            return AJS.I18n.getText('gh.rapid.plan.sprint.active.empty.help');
        }
        else {
            return AJS.I18n.getText('gh.rapid.plan.sprint.filtered.help');
        }
    };

    /**
     * Calculates the progress stats for all issues in this sprint
     * Note that the numbers will be wrong for future sprints because finished issues are *not* displayed in plan mode
     */
    BacklogStatistics.calculateSprintProgress = function (issueList, columns, estimationStatisticField) {
        // fetch mapping
        var statusToProgress = BacklogStatistics._getTodoInProgressDoneMapping(columns);

        var indicators = [
            {
                nameKey: "gh.boards.notstarted",
                name: AJS.I18n.getText("gh.boards.notstarted"),
                title: '',
                total: {
                    value: 0,
                    text: '0'
                }
            },
            {
                nameKey: "gh.boards.inprog",
                name: AJS.I18n.getText("gh.boards.inprog"),
                title: '',
                total: {
                    value: 0,
                    text: '0'
                }
            },
            {
                nameKey: "gh.boards.done",
                name: AJS.I18n.getText("gh.boards.done"),
                title: '',
                total: {
                    value: 0,
                    text: '0'
                }
            }
        ];

        // calculate the total for each progress state
        _.each(issueList.getAllIssues(), function (issue) {
            // fetch progress for issue
            var progStatus = statusToProgress[issue.statusId];
            if (_.isUndefined(progStatus) || issue.hidden) {
                return;
            }

            if (!issue.estimateStatistic || !issue.estimateStatistic.statFieldValue || !issue.estimateStatistic.statFieldValue.value) {
                return;
            }

            // sum up
            indicators[progStatus].total.value += issue.estimateStatistic.statFieldValue.value;
        });

        // calculate the sum of all totals
        var allIndicatorsTotalValue = _.reduce(indicators, function (memo, progressIndicator) {
            return memo + progressIndicator.total.value;
        }, 0);

        // if we have no values we show a progress bar without values
        if (!allIndicatorsTotalValue) {
            _.each(indicators, function (indicator) {
                indicator.total.text = '';
            });
            return indicators;
        }

        // render the total text for each indicator
        var allIndicatorsTotalText = BacklogStatistics.formatStatisticForRendering(estimationStatisticField, allIndicatorsTotalValue);
        _.each(indicators, function (indicator) {
            // indicator total
            indicator.total.text = BacklogStatistics.formatStatisticForRendering(estimationStatisticField, indicator.total.value);

            // indicator title
            indicator.title = GH.tpl.sprintview.renderProgressIndicatorTitle({
                indicator: indicator,
                allIndicatorsTotalText: allIndicatorsTotalText,
                statLabel: estimationStatisticField.name
            });
        });

        return indicators;
    };

    /**
     * Calculates a status mapping from status
     * @param columns
     */
    BacklogStatistics._getTodoInProgressDoneMapping = function (columns) {
        var statusToProgress = {};

        var mapToStatusProgressFn = function (statusId) {
            statusToProgress[statusId] = progStatus;
        };

        // for now, only the first column's statuses are "not started";
        // only the last column's statuses are "done";
        // the rest are "in progress"
        for (var i = 0; i < columns.length; i++) {
            var column = columns[i];
            var statusIds = column.statusIds;

            // find right bucket
            var progStatus;
            if (i === 0) {
                progStatus = 0; // to do
            }
            else if (i === columns.length - 1) {
                progStatus = 2; // done
            }
            else {
                progStatus = 1; // inprogress
            }

            // map all statuses
            _.each(statusIds, mapToStatusProgressFn);
        }
        return statusToProgress;
    };

    BacklogStatistics.splitIssuesIntoNotDoneAndDone = function (issues, columns) {
        // just reuse the mapping func
        var statusToProgress = BacklogStatistics._getTodoInProgressDoneMapping(columns);

        var notDone = [];
        var done = [];
        _.each(issues, function (issue) {
            var prog = statusToProgress[issue.statusId];
            if (prog == 2) {
                done.push(issue);
            } else {
                notDone.push(issue);
            }
        });
        return {
            notDone: notDone,
            done: done
        };
    };

    /**
     * For a given issue list, returns the total amount of work assigned to each assignee,
     * sorted from highest to lowest.
     *
     * @param {GH.IssueListModel} issueList
     * @returns {{unassigned: Object, assigned: Object[], all: Object}}
     */
    BacklogStatistics.getAssignedWorkStats = function (issueList) {
        function createWorkloadEntry(assignee, assigneeName) {
            return {
                assignee: assignee,
                assigneeName: assigneeName,
                totalEstimate: 0,
                totalTrackingEstimate:0,
                issues: []
            };
        }

        function addTextEstimate(workloadEntry) {
            workloadEntry.totalEstimateText = BacklogStatistics.formatStatisticForRendering(
                BacklogModel.estimationStatistic, workloadEntry.totalEstimate);
            if (!_.isUndefined(BacklogModel.trackingStatistic) && BacklogModel.trackingStatistic.isEnabled) {
                workloadEntry.totalTrackingEstimateText = BacklogStatistics.formatStatisticForRendering(
                    BacklogModel.trackingStatistic, workloadEntry.totalTrackingEstimate);
            }
        }

        var visibleIssues = issueList.getVisibleIssues();

        // Group issues and estimate statistic values by assignee
        var stats = _.reduce(visibleIssues, function (memo, issue) {
            var assignee = issue.assignee;

            // Create entry if necessary
            var workloadEntry;
            if (!assignee) {
                workloadEntry = memo.unassigned;
            } else {
                if (!memo.assignees[assignee]) {
                    memo.assignees[assignee] = createWorkloadEntry(assignee, issue.assigneeName);
                }
                workloadEntry = memo.assignees[assignee];
            }

            // Add issue
            workloadEntry.issues.push(issue);
            memo.all.issues.push(issue);

            // Add estimate
            var estimate = issue.estimateStatistic && issue.estimateStatistic.statFieldValue && issue.estimateStatistic.statFieldValue.value;
            if (estimate) {
                workloadEntry.totalEstimate += estimate;
                memo.all.totalEstimate += estimate;
            }

            var trackingEstimate = issue.trackingStatistic && issue.trackingStatistic.statFieldValue && issue.trackingStatistic.statFieldValue.value;
            if (trackingEstimate) {
                workloadEntry.totalTrackingEstimate += trackingEstimate;
                memo.all.totalTrackingEstimate += trackingEstimate;
            }

            return memo;
        }, {
            unassigned: createWorkloadEntry(null, null),
            assignees: {},
            all: createWorkloadEntry(null, null)
        });

        // Sort by highest estimate statistic total to lowest
        var sortedAssignedWork = _.toArray(stats.assignees).sort(function (a, b) {
            return b.totalEstimate - a.totalEstimate || a.assigneeName.localeCompare(b.assigneeName);
        });

        // Add formatted text for estimates
        var estimateField = BacklogModel.estimationStatistic;
        if (!BacklogStatistics.ignoredStatisticTypes[estimateField.typeId]) {
            _.each(sortedAssignedWork, addTextEstimate);
            addTextEstimate(stats.unassigned);
            addTextEstimate(stats.all);
        }

        return {
            unassigned: stats.unassigned,
            assigned: sortedAssignedWork,
            all: stats.all
        };
    };

    return BacklogStatistics;
});