/* globals
 * GH.ExtraFieldsHelper, GH.Util, GH.EpicController, GH.VersionController
 */

/**
 * Contains the data for the backlog
 * @module jira-agile/rapid/ui/plan/backlog-model2
 * @requires module:jira-agile/rapid/ui/plan/issue-list-model
 */
define('jira-agile/rapid/ui/plan/backlog-model2', ['require'], function (require) {
    var IssueListModel = require('jira-agile/rapid/ui/plan/issue-list-model');

    var BacklogModel2 = function BacklogModel2(issues, column) {
        // create a list for the contained issues
        this.column = column;
        this.issueList = new IssueListModel('backlog', issues, GH.VersionController.getVersionModel());
    };

    BacklogModel2.isBacklogModel = function (model) {
        return model.getType() === 'backlog';
    };

    /**
     * Contains the issues list contained in this model
     *
     * @return {IssueListModel}
     */
    BacklogModel2.prototype.getIssueList = function () {
        return this.issueList;
    };

    /**
     * Get the id of this model
     */
    BacklogModel2.prototype.getId = function () {
        return 'backlog';
    };

    /**
     * Get the type of this model
     */
    BacklogModel2.prototype.getType = function () {
        return 'backlog';
    };

    BacklogModel2.prototype.getColumn = function () {
        return this.column;
    };

    return BacklogModel2;
});

/**
 * Backlog issue support
 * @module jira-agile/rapid/ui/plan/backlog-model
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/plan/sprint-model
 * @requires module:jira-agile/rapid/ui/plan/sprint-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-issue-list-filtering
 * @requires module:jira-agile/rapid/ui/plan/issue-list-model
 */
define('jira-agile/rapid/ui/plan/backlog-model', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var PlanIssueListFiltering = require('jira-agile/rapid/ui/plan/plan-issue-list-filtering');
    var IssueListModel = require('jira-agile/rapid/ui/plan/issue-list-model');
    var SprintController;

    // Resolve circular dependency
    GlobalEvents.on("pre-initialization", function () {
        SprintController = require('jira-agile/rapid/ui/plan/sprint-controller');
    });

    var BacklogModel = {};

    /** The rank custom field ID */
    BacklogModel.rankCustomFieldId = undefined;

    /** The estimation statistic configured for the current board */
    BacklogModel.estimationStatistic = {};

    /** The tracking statistic configured for the current board */
    BacklogModel.trackingStatistic = {};

    /** Contains all sprint models */
    BacklogModel.sprintModels = [];

    /** Contains the backlog model */
    BacklogModel.backlogModel2 = null;

    /** Contains missing parents for subtasks **/
    BacklogModel.missingParentsByKey = {};

    BacklogModel.setData = function (data) {
        BacklogModel.calculateIssueLists(data);

        BacklogModel.hasBulkChangePermission = data.hasBulkChangePermission;
        BacklogModel.issueArchivingEnabled = data.issueArchivingEnabled;
        BacklogModel.cardColorStrategy = data.cardColorStrategy;

        // set up missing parents
        if (data.missingParents) {
            BacklogModel.missingParentsByKey = _.indexBy(data.missingParents, 'key');
        }
    };

    //
    // Sprints and Backlog models
    //
    function calculateIssueListsForScrumBoard(data) {
        var issues = data.issues;
        var sprints = data.sprints;

        // build a lookup map from issue to sprint
        var issueToSprint = {};
        _.each(sprints, function (sprint) {
            _.each(sprint.issuesIds, function (issueId) {
                issueToSprint[issueId] = sprint.id;
            });
        });

        // sort issues into individual lists, ordered according to rank
        var issuesPerSprint = {};
        _.each(sprints, function (sprint) {
            issuesPerSprint[sprint.id] = [];
        });
        var backlogIssues = [];

        _.each(issues, function (issue) {
            GH.ExtraFieldsHelper.prepareExtraFields(issue.extraFields);
            var sprintId = issueToSprint[issue.id];
            if (sprintId) {
                issuesPerSprint[sprintId].push(issue);
            } else {
                backlogIssues.push(issue);
            }
        });

        // create the models
        var sprintModels = [];
        _.each(sprints, function (sprint) {
            var sprintModel = new SprintModel(sprint, issuesPerSprint[sprint.id]);
            sprintModels.push(sprintModel);
        });
        var backlogModel2 = new GH.BacklogModel2(backlogIssues, data.backlogColumn);

        BacklogModel.sprintModels = sprintModels;
        BacklogModel.backlogModel2 = backlogModel2;
    }

    function calculateIssueListsForKanbanBoard(data) {
        var issues = assignParents(data.issues, [].concat(_.toArray(data.issues)).concat(_.toArray(data.missingParents)));
        var backlogStatusIds = data.backlogColumn.statusIds;
        if (data.selectedForDevelopmentColumn) {
            var selectedForDevStatusIds = data.selectedForDevelopmentColumn.statusIds;
        }

        var backlogIssues = [];
        var selectedForDevIssues = [];
        _.each(issues, function (issue) {
            GH.ExtraFieldsHelper.prepareExtraFields(issue.extraFields);
            if (backlogStatusIds.indexOf(issue.statusId) !== -1) {
                backlogIssues.push(issue);
            } else if (selectedForDevStatusIds && selectedForDevStatusIds.indexOf(issue.statusId) !== -1) {
                selectedForDevIssues.push(issue);
            }
        });

        BacklogModel.backlogModel2 = new GH.BacklogModel2(backlogIssues, data.backlogColumn);
        if (data.selectedForDevelopmentColumn) {
            BacklogModel.sprintModels = [new SprintModel({
                id: -1,
                name: data.selectedForDevelopmentColumn.name,
                state: 'FUTURE',
                column: data.selectedForDevelopmentColumn
            }, selectedForDevIssues)];
        } else {
            BacklogModel.sprintModels = [];
        }
    }

    function assignParents(visibleIssues, allIssues) {
        var keysToIssues = _.indexBy(allIssues, 'key');
        return visibleIssues.map(function (issue) {
            if (typeof issue.parentKey === 'string' && issue.parentKey.length > 0) {
                return _.defaults({ parent: keysToIssues[issue.parentKey] }, issue);
            } else {
                return issue;
            }
        });
    }

    /**
     * Calculates the issue lists of the provided data
     */
    BacklogModel.calculateIssueLists = function (data) {
        if (GH.RapidBoard.State.isScrumBoard()) {
            calculateIssueListsForScrumBoard(data);
        } else {
            calculateIssueListsForKanbanBoard(data);
        }
    };

    /**
     * Add a new sprint to the list of sprint models
     */
    BacklogModel.addNewSprint = function (sprintData) {
        var sprintModel = new SprintModel(sprintData, []);
        BacklogModel.sprintModels.push(sprintModel);
    };

    /**
     * Removes a sprint from the model
     *
     * All issues are added at the top of the next sprint/backlog
     *
     * @return boolean whether the backlog was affected
     */
    BacklogModel.removeSprintAndMoveIssues = function (sprintId) {
        // fetch all models
        var models = BacklogModel.getAllModels();
        var sprintModel, nextModel;
        for (var i = 0; i < models.length; i++) {
            var model = models[i];
            if (SprintModel.isSprintModel(model) && model.getSprintId() === sprintId) {
                sprintModel = model;
                nextModel = models[i + 1];
                break;
            }
        }

        var backlogAffected = false;
        if (sprintModel) {
            // add all issues of that model to the next model
            var issues = sprintModel.getIssueList().getIssuesInOrder();
            if (!_.isEmpty(issues)) {
                var targetList = nextModel.getIssueList();
                targetList.prependIssues(issues);
                backlogAffected = GH.BacklogModel2.isBacklogModel(nextModel);
            }

            // remove the sprint model
            BacklogModel.sprintModels = _.reject(BacklogModel.sprintModels, function (sprintModel) {
                return sprintModel.getSprintId() === sprintId;
            });
        }
        return backlogAffected;
    };

    /**
     * Updates the sprint data for a given sprint
     */
    BacklogModel.updateDataForSprint = function (sprintId, sprintData) {
        var sprintModel = BacklogModel.getSprintModel(sprintId);
        if (sprintModel) {
            sprintModel.setSprintData(sprintData);
        }
        return sprintModel;
    };

    /**
     * Move the specified sprint up in the list of sprints
     * @param {Number} sprintId
     * @return {Number} the id of the sprint who was swapped
     */
    BacklogModel.moveSprintUp = function (sprintId) {
        return BacklogModel._swapSprint(sprintId, -1);
    };

    /**
     * Move the specified sprint down in the list of sprints
     * @param {Number} sprintId
     * @return {Number} the id of the sprint who was swapped
     */
    BacklogModel.moveSprintDown = function (sprintId) {
        return BacklogModel._swapSprint(sprintId, 1);
    };

    /**
     * Swap sprint with the one before or after
     * @param {Number} sprintId
     * @param {Number} offset which position to swap with in the array (-1 or +1)
     * @return {Number} the id of the sprint who was swapped
     */
    BacklogModel._swapSprint = function (sprintId, offset) {
        var i = GH.Util.indexOf(BacklogModel.sprintModels, function (sprintModel) {
            if (sprintModel.getSprintId() === sprintId) {
                return sprintModel;
            }
        });
        if (GH.Util.swapArrayItems(BacklogModel.sprintModels, i, i + offset)) {
            var sprintAfter = BacklogModel.sprintModels[i];
            return sprintAfter.getSprintId();
        }
        return null;
    };

    BacklogModel.getSprintModels = function () {
        return BacklogModel.sprintModels;
    };

    BacklogModel.getActiveSprintModels = function () {
        return _.filter(BacklogModel.sprintModels, function (sprintModel) {
            return sprintModel.getSprintData().state === 'ACTIVE';
        });
    };

    BacklogModel.getFutureSprintModels = function () {
        return _.filter(BacklogModel.sprintModels, function (sprintModel) {
            return sprintModel.getSprintData().state === 'FUTURE';
        });
    };

    BacklogModel.getSprintModel = function (sprintId) {
        return _.find(BacklogModel.sprintModels, function (sprintModel) {
            return sprintId === sprintModel.getSprintId();
        });
    };

    /**
     * Returns sprint or backlog model (based on id)
     */
    BacklogModel.findTargetModel = function (id) {
        var targetModelId = id || 'backlog';

        return _.find(BacklogModel.getAllModels(), function (model) {
            return targetModelId === model.getId();
        });
    };

    BacklogModel.getBacklogModel2 = function () {
        return BacklogModel.backlogModel2;
    };

    BacklogModel.getAllModels = function () {
        var sprintModels = BacklogModel.getSprintModels();
        var backlogModel2 = BacklogModel.getBacklogModel2();
        return _.flatten([sprintModels, backlogModel2], true);
    };

    /**
     * Returns all issue lists (sprints and backlog)
     * @return {Array}
     */
    BacklogModel.getAllIssueListsNew = function () {
        var filtered = _.reject(BacklogModel.getAllModels(), _.isEmpty);
        return _.map(filtered, function (model) {
            return model.getIssueList();
        });
    };

    /**
     * Returns all issues in backlog with missing parents
     * @returns {Map}
     */
    BacklogModel.getAllIssuesWithMissingParents = function () {
        var allIssuesByKey = BacklogModel.getAllModels().reduce(function (o, model) {
            return _.extend({}, o, model.issueList.getAllIssues());
        }, {});

        // BacklogModel.missingParentsByKey may contain more information then allIssuesByKey
        // e.g. issues that were hidden because of filters - they won't get summary
        return _.extend({}, allIssuesByKey, BacklogModel.missingParentsByKey);
    };

    /**
     * Retrieve the model the passed issues belong to.
     *
     * @param issueKeyOrId an issue key, issue id, or an array of either to look for
     * @return {GH.BacklogModel2} or {SprintModel}
     */
    BacklogModel.findModelWithIssue = function (issueKeyOrId) {
        // find the list with the issues in it
        var matchingModel = null;
        _.any(BacklogModel.getAllModels(), function (model) {
            if (model.getIssueList().isIssueValid(issueKeyOrId)) {
                matchingModel = model;
                return true;
            }
            return false;
        });

        return matchingModel;
    };

    /**
     * Find models containing a fake parent matching the passed issueKey
     * @param issueKey
     */
    BacklogModel.findModelsWithFakeParent = function (issueKey) {
        return _.filter(BacklogModel.getAllModels(), function (model) {
            if (model.issueList.isIssueValid(issueKey)) {
                return false; // this issue is not fake in this model
            }
            return _.any(model.issueList.data.issueByKey, function (issue) {
                return issue.parentKey === issueKey;
            });
        });
    };

    /**
     * Find the model that follows the passed model
     * @param currentModel
     */
    BacklogModel.getNextModel = function (currentModel) {
        var models = BacklogModel.getAllModels();
        for (var i = 0; i < models.length; i++) {
            if (currentModel.getId() === models[i].getId()) {
                if (i + 1 < models.length) {
                    return models[i + 1];
                } else {
                    break;
                }
            }
        }
        return null;
    };

    /**
     * Calculates the rerank operation data that corresponds to the marker move operation
     */
    BacklogModel.calculateRankingDataForSprintMarkerMove = function (draggedSprintId, currentSprintId, previousIssueKey) {

        // calculate the models to consider for the calculation
        var data = BacklogModel._calculateIncludedModelsData(draggedSprintId, currentSprintId);

        // handle drag in the same sprint, e.g. removing issues from the sprint?
        if (draggedSprintId === currentSprintId) {
            var issuesToRemove = [];
            if (previousIssueKey) {
                issuesToRemove = data.draggedSprintModel.getIssueList().getVisibleIssuesDataAfterKey(previousIssueKey);
            } else {
                issuesToRemove = data.draggedSprintModel.getIssueList().getAllVisibleIssuesData();
            }
            if (!_.isEmpty(issuesToRemove)) {
                // find the next sprint, as we have to move issues to the top of that one
                var nextModel = BacklogModel.getNextModel(data.draggedSprintModel);
                if (nextModel) {
                    var targetSprintId = SprintModel.isSprintModel(nextModel) ? nextModel.getSprintId() : null;
                    var firstIssue = _.first(nextModel.getIssueList().getIssuesInOrder());
                    var firstIssueKey = firstIssue ? firstIssue.key : null;
                    var issueKeys = _.map(issuesToRemove, function (issue) {
                        return issue.key;
                    });
                    if (!_.isEmpty(issueKeys)) {
                        return {
                            sprintId: targetSprintId,
                            nextRankableId: firstIssueKey,
                            issueKeys: issueKeys
                        };
                    }
                }
            }
        }
        // handle drag to add issues to the sprint
        else {
                var issuesToAdd = BacklogModel.getIssueDataForVisisbleIssuesInModelsUpToIssueKey(data.otherAffectedModels, currentSprintId, previousIssueKey);
                var targetSprintId = draggedSprintId;
                var lastIssue = _.last(data.draggedSprintModel.getIssueList().getIssuesInOrder());
                var lastIssueKey = lastIssue ? lastIssue.key : null;
                var issueKeys = _.map(issuesToAdd, function (issue) {
                    return issue.key;
                });
                if (!_.isEmpty(issueKeys)) {
                    return {
                        sprintId: targetSprintId,
                        prevRankableId: lastIssueKey,
                        issueKeys: issueKeys
                    };
                }
            }

        // nothing to do
        return null;
    };

    /**
     * Calculates an issue list for a given sprint, taking into account a dragged marker which might currently reside in a different sprint.
     */
    BacklogModel.calculateTemporaryDataForSprintWithMarker = function (draggedSprintId, currentSprintId, previousIssueKey) {

        // calculate the models to consider for the calculation
        var data = BacklogModel._calculateIncludedModelsData(draggedSprintId, currentSprintId);

        // add all hidden issues of the dragged sprint, these are included regardless of the marker position
        var hiddenIssuesData = data.draggedSprintModel.getIssueList().getAllInvisibleIssuesData();

        // Find all visible issues above the marker
        var visibleIssuesData = BacklogModel.getIssueDataForVisisbleIssuesInModelsUpToIssueKey(data.models, currentSprintId, previousIssueKey);

        // put together a fake model with this data
        var allIssues = _.flatten([hiddenIssuesData, visibleIssuesData], true);
        var hiddenIssueKeys = _.map(hiddenIssuesData, function (issue) {
            return issue.key;
        });
        var resultIssueList = new IssueListModel('fakesprint-' + draggedSprintId, allIssues, GH.VersionController.getVersionModel());
        resultIssueList.setHiddenBySearchIssues(hiddenIssueKeys);

        return {
            issueList: resultIssueList,
            modelsData: data
        };
    };

    /**
     * Returns the list of models that are contained between and including draggedSprintId and currentSprintId.
     *
     * Note: collapsed sprints are ignored!
     *
     * @param draggedSprintId the sprint id of the marker currently being dragged
     * @param currentSprintId the sprint id in which the marker currently resides. null for backlog
     * @private
     */
    BacklogModel._calculateIncludedModelsData = function (draggedSprintId, currentSprintId) {
        var sprintModels = BacklogModel.getSprintModels();

        var afterDraggedSprint = false;

        var resultData = {
            draggedSprintModel: null,
            otherAffectedModels: [],
            models: []
        };

        for (var i = 0; i < sprintModels.length; i++) {
            var sprintModel = sprintModels[i];
            var sprintId = sprintModel.getSprintId();
            if (sprintId == draggedSprintId) {
                resultData.draggedSprintModel = sprintModel;
                resultData.models.push(sprintModel);
                afterDraggedSprint = true;
            } else if (afterDraggedSprint) {
                // Only add the model if the sprint isn't collapsed currently
                if (SprintController.isTwixieOpen(sprintId)) {
                    resultData.otherAffectedModels.push(sprintModel);
                    resultData.models.push(sprintModel);
                }
            }
            // stop if we are at current sprint id
            if (sprintId == currentSprintId) {
                break;
            }
        }

        // if currentSprintId is null, add the backlog model
        if (!currentSprintId) {
            resultData.otherAffectedModels.push(BacklogModel.getBacklogModel2());
            resultData.models.push(BacklogModel.getBacklogModel2());
        }

        return resultData;
    };

    BacklogModel.getIssueDataForVisisbleIssuesInModelsUpToIssueKey = function (models, lastModelId, lastIncludedIssueKey) {
        var visibleIssuesData = [];
        for (var i = 0; i < models.length; i++) {
            var model = models[i];
            var issueList = model.getIssueList();

            // is this the last model? if so only add up to previousIssueKey
            var isSprint = SprintModel.isSprintModel(model);
            if (isSprint && model.getSprintId() === lastModelId || !isSprint && !lastModelId) {
                if (lastIncludedIssueKey) {
                    visibleIssuesData.push(issueList.getVisibleIssuesDataBeforeIncludingKey(lastIncludedIssueKey));
                } else {
                    // no previous key = no issues included in this sprint!
                }
            } else {
                visibleIssuesData.push(issueList.getAllVisibleIssuesData());
            }
        }
        visibleIssuesData = _.flatten(visibleIssuesData, true);
        return visibleIssuesData;
    };

    //
    // Issue Ranking
    //

    BacklogModel.reorderIssuesInModel = function (issueKeys, sprintId) {
        var targetModel = BacklogModel.findTargetModel(sprintId);

        if (targetModel) {
            var issueList = targetModel.getIssueList();
            var allIssues = issueList.getAllIssues();

            issueList.setOrder(issueKeys.filter(function (key) {
                return allIssues[key];
            }));

            return [targetModel];
        }
        return [];
    };

    /**
     * Moves issues to a new position of a given sprint. issueKeys might come from different sprints/the backlog,
     * so this move takes care of correctly updating everything
     *
     * @return Array a list of changed models
     */
    BacklogModel.moveIssuesNew = function (issueKeys, sprintId, prevRankableId, nextRankableId) {
        if (issueKeys.length === 0) {
            return;
        }

        // keep removed data plus which list is the target list
        var removedIssuesData = [];
        var changedModels = [];
        var targetModelId = sprintId || 'backlog';
        var targetModel = null;

        // remove from all models that are not the target
        _.each(BacklogModel.getAllModels(), function (model) {
            if (model.getId() === targetModelId) {
                targetModel = model;
            } else {
                var list = model.getIssueList();
                var result = list.removeIssues(issueKeys);
                if (!_.isEmpty(result)) {
                    changedModels.push(model);
                    removedIssuesData.push(result);
                }
            }
        });

        // flatten to get the issue data list of all removed issues
        removedIssuesData = _.flatten(removedIssuesData, true);

        if (targetModel) {
            // add issues to target model
            var targetList = targetModel.getIssueList();
            targetList.addIssues(removedIssuesData);

            // then rerank
            targetList.reorderIssues(issueKeys, prevRankableId, nextRankableId);

            changedModels.push(targetModel);
        }

        return changedModels;
    };

    /**
     * Determines if any of the passed issue keys are present in the model and are subtasks.
     * @param issueKeys the selection of issues to query.
     * @returns {boolean} are there any subtasks in the given selection of issue keys
     */
    BacklogModel.hasAnySubtasks = function (issueKeys) {
        if (!issueKeys) {
            throw "Selection of issueKeys as an {array} must be provided.";
        }
        return issueKeys.some(function (key) {
            return this.isSubtask(key);
        }.bind(this));
    };

    BacklogModel.afterIssueUpdate = function (updatedIssue) {
        var _this = this;

        if (this.isSubtask(updatedIssue.key)) {
            if (typeof updatedIssue.parentKey === 'string') {
                var issueList = this.findModelWithIssue(updatedIssue.key).getIssueList();
                var issue = issueList.getIssueData(updatedIssue.key);
                var parent = typeof issue.parentKey === 'string' && this.getAllIssuesWithMissingParents()[issue.parentKey];
                if (_.isObject(parent)) {
                    issueList.updateIssue(_.extend({}, issue, { parent: parent }));
                }
            }
        } else {
            var subTasks = _.chain(this.getAllIssuesWithMissingParents()).filter(function (issue) {
                return _this.isSubtask(issue.key);
            }).filter(function (subTask) {
                return subTask.parentKey === updatedIssue.key;
            }).value();

            subTasks.forEach(function (subTask) {
                subTask.parent = updatedIssue;
            });
        }
    };

    /**
     * Filter only subtasks in given issue list
     * @param issueKeys the selection of issues to query.
     * @returns {Array} issue keys that are subtasks
     */
    BacklogModel.filterOnlySubtasks = function (issueKeys) {
        if (!issueKeys) {
            return [];
        }

        return issueKeys.filter(function (key) {
            return this.isSubtask(key);
        }.bind(this));
    };

    /**
     * Determines if given issue is a subtask
     * @param key {String} issue key
     * @returns {boolean} is given issue a subtask
     */
    BacklogModel.isSubtask = function (key) {
        var issue = this.getIssueData(key);
        return !!(issue && issue.parentKey);
    };

    //
    // Issue Selection
    //

    /**
     * Is the current issue selectable?
     */
    BacklogModel.isIssueSelectable = function (issueKey) {
        var issueLists = BacklogModel.getAllIssueListsNew();
        return _.any(issueLists, function (issueList) {
            // use visible instead of valid as hidden issues are not selectable
            return issueList.isIssueValid(issueKey) && issueList.isIssueVisible(issueKey);
        });
    };

    /**
     * Get the index of an issue relative to all the <b>visible</b> issues.
     * @param issueKey
     */
    BacklogModel.getIssueIndex = function (issueKey) {
        var index = -1;
        var issueLists = BacklogModel.getAllIssueListsNew();
        _.any(issueLists, function (issueList) {
            if (!issueList.isIssueValid(issueKey)) {
                return false;
            }
            index = issueList.getIssueIndex(issueKey);
            return true;
        });
        return index;
    };

    /**
     * Is the passed issue selectable
     * @param selectedIssueKeys the currently selected issues
     * @param issueKey
     */
    BacklogModel.canAddToSelection = function (selectedIssueKeys, issueKey) {
        var issueLists = BacklogModel.getAllIssueListsNew();
        return _.any(issueLists, function (issueList) {
            return issueList.isIssueValid(issueKey) && issueList.canAddToSelection(selectedIssueKeys, issueKey);
        });
    };

    /**
     * Get an issue range. The issue keys returned should be in the order as specified by from and to.
     */
    BacklogModel.getIssueRange = function (fromKey, toKey) {
        var keys = [];

        var issueLists = BacklogModel.getAllIssueListsNew();
        _.any(issueLists, function (issueList) {
            if (!issueList.isIssueValid(fromKey) || !issueList.isIssueValid(toKey)) {
                return false;
            }

            keys = issueList.getIssueRange(fromKey, toKey);
            return true;
        });

        return keys;
    };

    /**
     * Get the next visible issue according to a given issue key
     */
    BacklogModel.getNextIssueKey = function (issueKey) {
        var models = BacklogModel.getAllModels();
        for (var i = 0; i < models.length; i++) {
            var model = models[i];
            if (!model.getIssueList().isIssueValid(issueKey)) {
                continue;
            }
            var nextIssueKey = false;
            var isSprintModel = SprintModel.isSprintModel(model);
            var isSprintTwixieOpen = SprintController.isTwixieOpen(model.getId());
            // if current issue is inside an expanded sprint or in the backlog, try to get the next issue key
            if (isSprintModel && isSprintTwixieOpen || !isSprintModel) {
                nextIssueKey = model.getIssueList().getNextIssueKey(issueKey);
                if (nextIssueKey) {
                    return nextIssueKey;
                }
            }
            // no next issue key, check if we can jump to the next issue lists
            for (var j = i + 1; j < models.length; j++) {
                var nextModel = models[j];
                isSprintModel = SprintModel.isSprintModel(nextModel);
                isSprintTwixieOpen = SprintController.isTwixieOpen(nextModel.getId());
                // If model is a sprint and it's twixie is closed, ignore it
                if (isSprintModel && !isSprintTwixieOpen) {
                    continue;
                }
                nextIssueKey = nextModel.getIssueList().getFirstVisibleIssueKey();
                if (nextIssueKey) {
                    break;
                }
            }
            return nextIssueKey;
        }
    };

    /**
     * Get a previous visible issue according to a given issue key
     */
    BacklogModel.getPreviousIssueKey = function (issueKey) {
        var models = BacklogModel.getAllModels();
        for (var i = 0; i < models.length; i++) {
            var model = models[i];
            if (!model.getIssueList().isIssueValid(issueKey)) {
                continue;
            }
            var previousIssueKey = false;
            var isSprintModel = SprintModel.isSprintModel(model);
            var isSprintTwixieOpen = SprintController.isTwixieOpen(model.getId());
            // if current issue is inside an expanded sprint or in the backlog, try to get the previous issue key
            if (isSprintModel && isSprintTwixieOpen || !isSprintModel) {
                previousIssueKey = model.getIssueList().getPreviousIssueKey(issueKey);
                if (previousIssueKey) {
                    return previousIssueKey;
                }
            }
            // no previous issue key, check if we can jump to the next previous lists
            for (var j = i - 1; j >= 0; j--) {
                var previousModel = models[j];
                isSprintModel = SprintModel.isSprintModel(previousModel);
                isSprintTwixieOpen = SprintController.isTwixieOpen(previousModel.getId());
                // If model is a sprint and it's twixie is closed, ignore it
                if (isSprintModel && !isSprintTwixieOpen) {
                    continue;
                }
                previousIssueKey = previousModel.getIssueList().getLastVisibleIssueKey();
                if (previousIssueKey) {
                    break;
                }
            }
            return previousIssueKey;
        }
    };

    //
    // Ranking
    //

    /**
     * Get the first rankable issue key in the column.
     */
    BacklogModel.getFirstRankableIssueKeyInColumn = function (issueKey, useParentIssue) {
        // TODO which issueList do we pick from if the issueKey is null? for now choose the first list
        var issueKeySupplied = issueKey != null;
        var issueKeys = issueKeySupplied ? [issueKey] : [];
        var first = false;
        var issueLists = BacklogModel.getAllIssueListsNew();
        _.any(issueLists, function (issueList) {
            if (issueKeySupplied && !issueList.isIssueValid(issueKey)) {
                return false;
            }
            first = issueList.getFirstRankableIssueKeyInColumn(issueKeys, useParentIssue);
            return true;
        });
        return first;
    };

    /**
     * Get the last rankable issue key in the column, ignoring the passed issue key.
     * This means if you pass the last rankable issue key in the column, it'll return the one *before* it.
     */
    BacklogModel.getLastRankableIssueKeyInColumn = function (issueKey, useParentIssue) {
        // TODO which issueList do we pick from if the issueKey is null? for now choose the first list
        var issueKeySupplied = issueKey != null;
        var issueKeys = issueKeySupplied ? [issueKey] : [];
        var last = false;
        var issueLists = BacklogModel.getAllIssueListsNew();
        _.any(issueLists, function (issueList) {
            if (issueKeySupplied && !issueList.isIssueValid(issueKey)) {
                return false;
            }
            last = issueList.getLastRankableIssueKeyInColumn(issueKeys, useParentIssue);
            return true;
        });
        return last;
    };

    /**
     * Finds the next issue to select given a set of to-be-ranked issues
     */
    BacklogModel.getBestFitSelectionAfterRank = function (toBeRankedIssueKeys) {
        var model = BacklogModel.findModelWithIssue(_.first(toBeRankedIssueKeys));
        if (model) {
            return model.getIssueList().getBestFitSelectionAfterRank(toBeRankedIssueKeys);
        }
        return null;
    };

    /**
     * Set the rank custom field ID
     */
    BacklogModel.setRankCustomFieldId = function (rankCustomFieldId) {
        BacklogModel.rankCustomFieldId = rankCustomFieldId;
    };

    /**
     * Set the estimate statistic (provided as part of the view configuration)
     */
    BacklogModel.setEstimationStatistic = function (estimationStatistic) {
        BacklogModel.estimationStatistic = {
            isEnabled: estimationStatistic.isEnabled,
            typeId: estimationStatistic.typeId,
            name: estimationStatistic.name,
            renderer: estimationStatistic.renderer
        };
    };

    /**
     * Set the tracking statistic (provided as part of the view configuration)
     */
    BacklogModel.setTrackingStatistic = function (trackingStatistic) {
        BacklogModel.trackingStatistic = {
            isEnabled: trackingStatistic.isEnabled,
            typeId: trackingStatistic.typeId,
            name: trackingStatistic.name,
            renderer: trackingStatistic.renderer
        };
    };

    /**
     * Can we rank?
     */
    BacklogModel.isRankable = function () {
        return !_.isUndefined(BacklogModel.rankCustomFieldId);
    };

    /**
     * Get the rank custom field id.
     */
    BacklogModel.getRankCustomFieldId = function () {
        return BacklogModel.rankCustomFieldId;
    };

    /**
     * Return a map of the issues hidden by search
     */
    BacklogModel.getHiddenBySearchIssues = function () {
        var result = {};
        _.each(BacklogModel.getAllIssueListsNew(), function (issueList) {
            var hiddenIssueKeys = issueList.getHiddenBySearchIssues();
            result = _.extend(result, hiddenIssueKeys);
        });
        return result;
    };

    BacklogModel.getIssueData = function (issueKey) {
        return BacklogModel._anyIssueList(issueKey, function (issueList) {
            return issueList.getIssueData(issueKey);
        });
    };

    BacklogModel.getIssueDataById = function (issueId) {
        return BacklogModel._anyIssueList(issueId, function (issueList) {
            return issueList.getIssueDataById(issueId);
        });
    };

    BacklogModel.getIssueIdForKey = function (issueKey) {
        return BacklogModel._anyIssueList(issueKey, function (issueList) {
            return issueList.getIssueIdForKey(issueKey);
        });
    };

    /**
     * Looks in all issue lists for the issue with the specified key, to see if the issue is 'valid' for that list. Once it
     * finds the first list which the issue is valid in, it will execute the callback function specified on that issue list.
     * The callback function should take as its only argument the issue list which has the issue.
     *
     * @param issueKeyOrId the key or id we are looking for
     * @param callback the callback to execute on the found issue list
     * @return {Object} the result of the callback function, or false if none was executed.
     */
    BacklogModel._anyIssueList = function (issueKeyOrId, callback) {
        var result = null;
        _.any(BacklogModel.getAllIssueListsNew(), function (issueList) {
            if (!issueList.isIssueValid(issueKeyOrId)) {
                return false;
            }
            result = callback(issueList);
            return true;
        });
        return result;
    };

    BacklogModel.isIssueVisible = function (issueKey) {
        var model = BacklogModel.findModelWithIssue(issueKey);
        if (model) {
            var issueList = model.getIssueList();
            return issueList.isIssueValid(issueKey) && issueList.isIssueVisible(issueKey);
        } else {
            return false;
        }
    };

    BacklogModel.hasAnyIssueInvisible = function (issueKeys) {
        return issueKeys.some(function (issueKey) {
            return !BacklogModel.isIssueVisible(issueKey);
        });
    };

    /**
     * @return {boolean}
     */
    BacklogModel.canManageSprints = function () {
        return BacklogModel._canManageSprints;
    };

    BacklogModel.setCanManageSprints = function (canManageSprints) {
        BacklogModel._canManageSprints = canManageSprints;
    };

    /**
     * @return {boolean}
     */
    BacklogModel.canCreateIssue = function () {
        return BacklogModel._canCreateIssue;
    };

    BacklogModel.setCanCreateIssue = function (canCreateIssue) {
        BacklogModel._canCreateIssue = canCreateIssue;
    };

    /**
     * @return {boolean}
     */
    BacklogModel.supportsPages = function () {
        return BacklogModel._supportsPages;
    };

    BacklogModel.setSupportsPages = function (supportsPages) {
        BacklogModel._supportsPages = supportsPages;
    };

    /**
     * Updates all lists with the current filters
     * @return Boolean true if something changed
     */
    BacklogModel.updateFiltering = function () {
        // fetch all lists
        var issueLists = BacklogModel.getAllIssueListsNew();

        // apply
        return PlanIssueListFiltering.applyToIssueLists(issueLists);
    };

    /**
     * Returns the filtered status of epics based on the given version id.
     *
     * If an epic contains issue that also have the given versionId as a fix version, the epic won't be filtered.
     * @param versionId The versionId for which to get the epics filtered status. Can be either a number or "none" (which the no versions filter)
     */
    BacklogModel.getEpicsFilteredForVersion = function (versionId) {
        var versionNone = false;
        if (_.isString(versionId)) {
            if (PlanIssueListFiltering.isNoneVersionFilterId(versionId)) {
                versionNone = true;
            } else {
                versionId = parseInt(versionId, 10);
            }
        }

        var epicIssueList = GH.EpicController.getEpicModel().getEpicList();

        if (!epicIssueList || epicIssueList.getIssueCount() === 0) {
            return [];
        }

        var epicsToProcess = epicIssueList.getOrder();
        var unfilteredEpics = [];

        // loop over all issues and determine if they belong to the given versionId and an epic.
        // Once all epics have been processed or all issues are checked, the unprocessed epics are marked as unfiltered while the processed epics are considered unfiltered
        // An epic is considered processed when an issue was found that was linked to the epic and that issue has the given versionId in its fixVersions
        _.each(BacklogModel.getAllIssueListsNew(), function (issueList) {
            if (epicsToProcess.length > 0) {
                var issues = issueList.getAllIssues();
                _.each(issues, function (issue) {
                    // Is issue linked to epic that hasn't been processed yet?
                    if (issue.epic && _.contains(epicsToProcess, issue.epic)) {
                        var markEpic = false;
                        // If version is none and issue doesn't have any fix versions, mark epic
                        if (versionNone && _.isEmpty(issue.fixVersions)) {
                            markEpic = true;
                        }
                        // If issue contains versionId in it's fixVersions, mark epic
                        else if (!versionNone && _.contains(issue.fixVersions, versionId)) {
                                markEpic = true;
                            }
                        if (markEpic) {
                            unfilteredEpics.push(issue.epic);
                            epicsToProcess = _.without(epicsToProcess, issue.epic);
                            if (epicsToProcess.length === 0) {
                                return false;
                            }
                        }
                    }
                });
            }
        });

        return {
            filtered: epicsToProcess,
            unfiltered: unfilteredEpics
        };
    };

    /**
     * Removes the epic from the issues with the given issue keys.
     *
     * @param issueKeys
     * @returns {Array} of issues who have been removed from their associated epic
     */
    BacklogModel.removeEpicFromIssues = function (issueKeys) {
        var issues = [];
        _.each(issueKeys, function (issueKey) {
            var issueData = BacklogModel.getIssueData(issueKey);
            if (issueData && issueData.epic) {
                delete issueData.epic;
                issues.push(issueData);
            }
        });
        return issues;
    };

    BacklogModel.getCardColorStrategy = function () {
        return BacklogModel.cardColorStrategy;
    };

    return BacklogModel;
});