/**
 * Notification system. Responsible for showing things like error messages.
 *
 * @since 7.6 "jira-agile/rapid/ui/notification" AMD module
 */
define('jira-agile/rapid/ui/notification', ['require'], function (require) {
    'use strict';

    var $ = require('jquery');
    var _ = require('underscore');
    var auiFlag = require('aui/flag');
    var logger = require('jira/util/logger');
    var analyticsTrigger = require('jira/analytics');

    var Notification = {
        /**
         * Display a list of global (not field related) errors on the page. The errors
         * are expected to have been translated on the server side.
         *
         * @param {{errors: Array.<{message: string, noAutoescape: boolean}>}} errors
         * @param {boolean=} globalErrorDismissable optional flag to set error message to be closeable
         * @param {{autoHide: boolean, showTitle: boolean}=} opts extra options which may be specified
         */
        showErrors: function showErrors(errors) {
            var globalErrorDismissable = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
            var opts = arguments[2];

            var title = AJS.I18n.getText('gh.error.errortitle');
            var body = GH.tpl.rapid.notification.renderErrorMessages(errors);

            Notification.showError(title, body, globalErrorDismissable, opts);
        },
        /**
         * @param title flag title
         * @param body flag content
         * @param {boolean=} globalErrorDismissable optional flag to set error message to be closeable
         * @param {{autoHide: boolean, showTitle: boolean}=} opts extra options which may be specified
         */
        showError: function showError(title, body) {
            var globalErrorDismissable = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
            var opts = arguments[3];

            var defaultOpts = {
                autoHide: false,
                closeable: globalErrorDismissable,
                showTitle: true,
                type: 'error'
            };
            opts = _.extend({}, defaultOpts, opts);

            createFlag(body, title, opts);
        },
        /**
         * Display a list of messages as warnings.
         *
         * @param {string|Array.<{message: string, noAutoescape: boolean}>} messages an array of message JSON objects (each containing a message field)
         * @param {{autoHide: boolean, showTitle: boolean, closeable: boolean, preRendered: boolean}=} opts extra options which may be specified
         */
        showWarnings: function showWarnings(messages, opts) {
            var defaultOpts = {
                preRendered: false,
                autoHide: true,
                closeable: true,
                showTitle: false,
                type: 'warning'
            };
            opts = _.extend({}, defaultOpts, opts);

            var body = void 0;
            if (!opts.preRendered) {
                body = GH.tpl.rapid.notification.renderErrorMessages({
                    errors: messages
                });
            } else {
                body = messages;
            }

            createFlag(body, AJS.I18n.getText('gh.error.warningtitle'), opts);
        },
        /**
         * Show a single warning message. The message must already be HTML rendered.
         * @param message
         */
        showWarning: function showWarning(message) {
            Notification.showWarnings(message, {
                preRendered: true
            });
        },
        /**
         * Show a single success message. The message must already be HTML rendered.
         * @param message
         * @param {{autoHide: boolean, showTitle: boolean, closeable: boolean}=} opts extra options which may be specified
         */
        showSuccess: function showSuccess(message, opts) {
            var defaultOptions = {
                closeable: true,
                autoHide: true,
                showTitle: false,
                type: 'success'
            };
            opts = _.extend({}, defaultOptions, opts);

            createFlag(message, null, opts);
        },
        /**
         * Remove any existing notifications.
         */
        clear: function clear() {
            $('#aui-flag-container').empty();
        },
        /**
         * Handles a date mismatch validation error. Shows a warning pointing to the advanced settings to fix the configuration
         */
        handleDateFormatMismatchError: function handleDateFormatMismatchError(e) {
            var link = '<a href="' + GH.Ajax.CONTEXT_PATH + '/secure/admin/AdvancedApplicationProperties.jspa">';
            link += AJS.I18n.getText('admin.advancedproperties.settings.heading');
            link += '</a>';
            var message = AJS.I18n.getText('gh.error.java.javascript.dateformat.mismatch', link);
            Notification.showWarnings(message, {
                autoHide: false,
                preRendered: true
            });
        },
        auiMessage: function auiMessage() {
            logger.warn('[deprecated] GH.Notification.auiMessage has no effect');
        },
        /**
         * Adaption of IssueNavigator.Shortcuts.js -> flashIssueRow()
         */
        showIssueMessage: function showIssueMessage(issueId) {
            var sessionstorage = JIRA.SessionStorage;
            var selectedIssueMsg = null;
            var selectedIssueKey = null;
            if (!issueId) {
                // ok try session storage next
                issueId = sessionstorage.getItem('selectedIssueId');
            }
            if (issueId) {
                selectedIssueKey = sessionstorage.getItem('selectedIssueKey');
                selectedIssueMsg = sessionstorage.getItem('selectedIssueMsg');
                Notification.showIssueMessageImpl(issueId, selectedIssueMsg, selectedIssueKey);
            }
            // always remove the state.  We want to be empty until some other action decides it wants to record this
            sessionstorage.removeItem('selectedIssueId');
            sessionstorage.removeItem('selectedIssueKey');
            sessionstorage.removeItem('selectedIssueMsg');
        },
        /**
         * Adaption of IssueNavigator.Shortcuts.js -> flashIssueRowWithId
         */
        showIssueMessageImpl: function showIssueMessageImpl(issueId, selectedIssueMsg, selectedIssueKey) {
            // if we don't have an message, use a generic one if possible
            if (!selectedIssueMsg) {
                selectedIssueMsg = 'thanks_issue_updated';
            }

            var msgText = AJS.params[selectedIssueMsg];
            if (msgText && selectedIssueKey) {
                msgText = AJS.format(msgText, selectedIssueKey);
                Notification.showSuccess(msgText);
            }
        },
        /**
         * Add a message to be displayed on the next page load
         */
        addPageLoadMessage: function addPageLoadMessage(message, type) {
            var messages = GH.storage.get('gh.pageloadmessages', true) || [];
            messages.push({
                message: message,
                type: type || 'success'
            });
            GH.storage.put('gh.pageloadmessages', messages, true);
        },
        _displayPageLoadMessages: function _displayPageLoadMessages() {
            var messages = GH.storage.get('gh.pageloadmessages', true);
            if (messages) {
                var type = messages[0].type;
                var message = '';
                _.each(messages, function (entry) {
                    if (message.length > 0) {
                        message += '<br>';
                    }
                    message += entry.message;
                });

                switch (type) {
                    case 'warning':
                        Notification.showWarning(message);
                        break;
                    case 'success':
                    /* falls through */
                    default:
                        Notification.showSuccess(message);
                        break;
                }

                // delete messages
                GH.storage.put('gh.pageloadmessages', null, true);
            }
        },
        /**
         * Display a message which can trigger a reload or be ignored
         * @return {AJS.$.Deferred} a promise which when resolved indicates the user wants to update
         *         the board. rejected indicates they want to ignore the update
         */
        showBoardUpdatedMessage: function showBoardUpdatedMessage() {
            // Remove it if it already exists
            Notification.triggerShowMessageAnalytics();
            Notification.clearBoardUpdatedMessage();
            var deferred = $.Deferred();
            var $msg = $(GH.tpl.rapid.notification.renderGHtvMessage()).appendTo('#gh');
            $msg.data('deferred', deferred);
            $msg.find('.js-refresh-now').click(function (e) {
                e.preventDefault();
                $msg.removeData('deferred'); // Remove the deferred so it isn't rejected by default
                Notification.clearBoardUpdatedMessage();
                deferred.resolve();
            });
            $msg.find('.js-ignore-refresh').click(function (e) {
                e.preventDefault();
                Notification.clearBoardUpdatedMessage();
            });
            return deferred.promise();
        },
        clearBoardUpdatedMessage: function clearBoardUpdatedMessage() {
            var $msg = $('#ghx-update-message');
            if ($msg.length) {
                var deferred = $msg.data('deferred');
                if (deferred && deferred.reject) {
                    deferred.reject();
                }
                $msg.remove();
            }
        },
        isBoardUpdatedMessageVisible: function isBoardUpdatedMessageVisible() {
            var $msg = $('#ghx-update-message');
            return $msg.length > 0;
        },
        triggerShowMessageAnalytics: function triggerShowMessageAnalytics() {
            analyticsTrigger.send({
                name: 'gh.rapidboard.updated.message.show',
                properties: {
                    'initial': !Notification.isBoardUpdatedMessageVisible()
                }
            });
        }
    };

    /**
     * @param {string} body
     * @param {string} title
     * @param {{autoHide: boolean, showTitle: boolean, closeable: boolean, type: string}} opts extra options passed to aui-flag
     * @returns {HTMLElement} flag DOM element
     */
    function createFlag(body, title, opts) {
        return cleanUpFlag(auiFlag({
            body: body,
            title: opts.showTitle ? title : undefined,
            close: opts.autoHide ? 'auto' : opts.closeable ? 'manual' : 'never',
            type: opts.type
        }));
    }

    function cleanUpFlag(flag) {
        // AUI-4406 AUI-2828: title is rendered as an empty element even if not used :(
        var $title = $(flag).find('.title');
        if ($.trim($title.text()) === "") {
            $title.remove();
        }
        return flag;
    }

    return Notification;
});
AJS.namespace('GH.Notification', null, require('jira-agile/rapid/ui/notification'));

// Register a document ready handler to show any not yet displayed message
AJS.$(document).ready(function () {
    GH.Notification.showIssueMessage();
    if (GH.storage) {
        GH.Notification._displayPageLoadMessages();
    }
});