define('jira-agile/rapid/ui/kanplan/KanplanEpicsInlineDialog',
    [
        'jquery',
        'jira-help-tips/feature/help-tip',
        'jira/util/formatter',
        'jira-agile/rapid/ui/kanplan/kan-plan-feature-service',
        'underscore'
    ],
    function ($, HelpTip, formatter, KanPlanFeatureService, _) {
        'use strict';

        const PLAN_MODE = 'plan';
        const WORK_MODE = 'work';
        const TITLE = formatter.I18n.getText("gh.boards.kanplan.work.epics.inlinedialog.title");
        const WORK_TEXT = formatter.I18n.getText("gh.boards.kanplan.work.epics.inlinedialog.text");
        const WORK_NEXT = formatter.I18n.getText("gh.boards.kanplan.work.epics.inlinedialog.button.next");
        const WORK_CANCEL = formatter.I18n.getText("gh.boards.kanplan.work.epics.inlinedialog.button.cancel");
        const PLAN_TEXT_ADMIN = formatter.I18n.getText("gh.boards.kanplan.plan.epics.inlinedialog.text.admin");
        const PLAN_TEXT_USER = formatter.I18n.getText("gh.boards.kanplan.plan.epics.inlinedialog.text.user");
        const PLAN_NEXT = formatter.I18n.getText("gh.boards.kanplan.plan.epics.inlinedialog.button.next");
        const PLAN_CANCEL = formatter.I18n.getText("gh.boards.kanplan.plan.epics.inlinedialog.button.cancel");

        const ANALYTICS_ON_WORK_MODE = Object.freeze({
            TAKE_ME_THERE: 'jira-software.kanplan.epics.onboarding.on.work.mode.takemethere.clicked',
            NO_THANKS: 'jira-software.kanplan.epics.onboarding.on.work.mode.nothanks.clicked',
            SHOWN: 'jira-software.kanplan.epics.onboarding.on.work.mode.shown',
        });
        const ANALYTICS_ON_PLAN_MODE = Object.freeze({
            OK_GOT_IT: 'jira-software.kanplan.epics.onboarding.on.plan.mode.okgotit.clicked',
            LET_ME_CHANGE_THIS: 'jira-software.kanplan.epics.onboarding.on.plan.mode.letmechangethis.clicked',
            SHOWN: 'jira-software.kanplan.epics.onboarding.on.plan.mode.shown',
        });

        class EpicsInlineDialog {
            constructor (options) {
                this.mode = options.mode;
                this.isEpicsPanelOpen = options.epicsPanelState;
                this.epicsCount = options.epicsCount;

                let shownEventName = (this.mode === WORK_MODE) ? ANALYTICS_ON_WORK_MODE.SHOWN : ANALYTICS_ON_PLAN_MODE.SHOWN;
                KanPlanFeatureService.sendAnalyticsInfo(shownEventName, this.extraEventAttributes(), true);
            }

            render () {
                this.dialog = this.createDialog();
                this.dialog.show();
                this.dialog.refresh();
            }

            hide () {
                if (this.dialog) {
                    this.dialog.view.dismiss();
                }
            }

            createDialog () {
                let defaults = {
                    inlineDialogOpts: {
                        gravity: 'w',
                        offsetY: 50
                    },
                    callbacks: {
                        init: () => {
                            //init events only after HelpTip is rendered
                            this.bindEvents();
                        }
                    }
                };

                return new HelpTip(_.extend(defaults, this.getOptionsForMode()));
            }

            bindEvents () {
                let $dialogContainer = this.dialog.view.$el;

                $dialogContainer.find('.helptip-next').on('click', () => {
                    this.nextHandler();
                });

                $dialogContainer.find('.helptip-close').on('click', () => {
                    this.closeHandler();
                });
            }

            getOptionsForMode () {
                if (this.mode === WORK_MODE) {
                    return {
                        title: TITLE,
                        bodyHtml: WORK_TEXT,
                        anchor: $('.aui-icon.agile-icon-plan').closest('a'),
                        showNextButton: true,
                        nextButtonText: WORK_NEXT,
                        showCloseButton: true,
                        closeButtonText: WORK_CANCEL,
                        inlineDialogOpts: {
                            offsetY: 105,
                            arrowOffsetY: -60,
                            gravity: 'w'
                        },
                    }
                } else {
                    return {
                        title: TITLE,
                        bodyHtml: this.getTextForPlanMode(),
                        anchor: this.getAnchor(),
                        showNextButton: true,
                        nextButtonText: PLAN_NEXT,
                        showCloseButton: this.isBoardAdmin(),
                        closeButtonText: PLAN_CANCEL
                    }
                }
            }

            getAnchor () {
                return this.isEpicsPanelOpen ? $('#ghx-epic-column .ghx-classification-header') : $('.js-epic-toggle');
            }

            getTextForPlanMode () {
                if (this.isBoardAdmin()) {
                    return PLAN_TEXT_ADMIN;
                } else {
                    return PLAN_TEXT_ADMIN.concat('<br/>', PLAN_TEXT_USER);
                }
            }

            isBoardAdmin () {
                //The user is a board admin (or global admin)
                return GH.UserData.userConfig.isGlobalAdmin || GH.RapidViewConfig.isBoardAdmin;
            }

            nextHandler () {
                KanPlanFeatureService.dismissEpicsOnboardingDialog().done(() => {
                    if (this.mode === WORK_MODE) {
                        GH.RapidBoard.ViewController.setMode('plan');
                        KanPlanFeatureService.sendAnalyticsInfo(ANALYTICS_ON_WORK_MODE.TAKE_ME_THERE, this.extraEventAttributes(), true);
                    } else {
                        KanPlanFeatureService.sendAnalyticsInfo(ANALYTICS_ON_PLAN_MODE.OK_GOT_IT, this.extraEventAttributes(), true);
                    }
                });
            }

            closeHandler () {
                KanPlanFeatureService.dismissEpicsOnboardingDialog().done(() => {
                    if (this.mode === PLAN_MODE) {
                        // If the timeout is not there the page will redirect while the dialog is being hidden, leaving an awkward ghost dialog
                        setTimeout(()=> {
                            GH.RapidBoard.gotoRapidViewConfiguration(GH.RapidBoard.State.getRapidViewId(), 'columns');
                        }, 0);
                        KanPlanFeatureService.sendAnalyticsInfo(ANALYTICS_ON_PLAN_MODE.LET_ME_CHANGE_THIS, this.extraEventAttributes(), true);
                    } else {
                        KanPlanFeatureService.sendAnalyticsInfo(ANALYTICS_ON_WORK_MODE.NO_THANKS, this.extraEventAttributes(), true);
                    }
                });
            }

            updatePosition (epicsPanelState) {
                this.isEpicsPanelOpen = epicsPanelState;
                this.dialog.dismiss();
                this.render();
            }

            extraEventAttributes () {
                return {
                    epicsCount: this.epicsCount
                }
            }
        }

        return EpicsInlineDialog;
    }
);