define('jira-agile/rapid/ui/kanplan/KanPlanInlineDialogController',
    [
        'jira-agile/rapid/events',
        'jira-agile/rapid/ui/kanplan/KanPlanInlineDialog',
        'jira-agile/rapid/ui/kanplan/KanplanEpicsInlineDialog',
        'jira-agile/rapid/ui/kanplan/kan-plan-feature-service'
    ],
    function (Events,
              KanPlanInlineDialog,
              KanplanEpicsInlineDialog,
              KanPlanFeatureService) {
        'use strict';

        var kanPlanInlineDialog = new KanPlanInlineDialog();

        var DIALOG_DISMISSED_KEY = 'ghx.kanplan.inlineDialog.dismissed';
        var ANALYTICS_NAMESPACE = 'jira-software.kanplan.enable-prompt.';
        var ISSUE_COUNT_MIN_THRESHOLD = 20;
        var COLUMN_MIN_THRESHOLD = 3;

        var issuesCount;

        var KanPlanInlineDialogController = {};

        KanPlanInlineDialogController.init = function () {
            kanPlanInlineDialog.on('moreButton:click', function () {
                KanPlanInlineDialogController.trigger(KanPlanInlineDialogController.CONTINUE_EVENT);
                KanPlanFeatureService.sendAnalyticsInfo(ANALYTICS_NAMESPACE + 'tellmemore', {
                    issueCount: issuesCount,
                    boardId: GH.RapidBoard.State.getRapidViewId()
                });
            });

            kanPlanInlineDialog.on('closeButton:click', function () {
                KanPlanInlineDialogController.markAsDismissed();
                KanPlanFeatureService.showKanPlanInlineDialogDismissedFlag();
                KanPlanFeatureService.sendAnalyticsInfo(ANALYTICS_NAMESPACE + 'notnow', {
                    issueCount: issuesCount,
                    boardId: GH.RapidBoard.State.getRapidViewId()
                });
            });

            this.off(this);
            this.on('showepicsonboarding', this.showEpicsOnboarding, this);
            this.on('updateepicsonboardingposition', this.updateEpicsOnboardingDialogPosition, this);
        };

        /**
         * @param {GH.GridDataModel} model
         */
        KanPlanInlineDialogController.changeVisibility = function (model) {
            kanPlanInlineDialog.setInlineDialogAnchor(GH.WorkView.getFirstColumnHeader());

            if (KanPlanInlineDialogController.isVisible(model)) {
                kanPlanInlineDialog.show();
                KanPlanFeatureService.sendAnalyticsInfo(ANALYTICS_NAMESPACE + 'show', {
                    issueCount: issuesCount,
                    boardId: GH.RapidBoard.State.getRapidViewId()
                });
            }
            else {
                KanPlanInlineDialogController.showEpicsOnboarding('work');
            }
        };

        KanPlanInlineDialogController.showEpicsOnboarding = (mode) => {
            if (KanPlanInlineDialogController.shouldShowEpicsOnboarding(mode)) {
                KanPlanInlineDialogController.createEpicsOnboardingDialog(mode, GH.PlanController.isEpicsColumnVisible());
            }
        };

        KanPlanInlineDialogController.createEpicsOnboardingDialog = function (mode, epicsPanelState) {
            let epicsCount = getEpicsCount(mode);
            this.epicsOnboardingDialog = new KanplanEpicsInlineDialog({
                mode,
                epicsPanelState,
                epicsCount,
            });
            this.epicsOnboardingDialog.render();
            return this.epicsOnboardingDialog;
        };

        KanPlanInlineDialogController.shouldShowEpicsOnboarding = function (mode) {
            // Make sure existing dialogues are hidden
            if (this.epicsOnboardingDialog) {
                this.epicsOnboardingDialog.hide();
            }

            // The current rapidview is a Kanban board, with a mapped Kanban backlog.
            if (!GH.RapidBoard.State.isKanbanBoard() || !GH.RapidViewConfig.kanPlanEnabled) {
                return false;
            }

            // Kanplan onboarding banners are not displayed
            if (KanPlanFeatureService.shouldShowAcknowledge()) {
                return false;
            }

            // KanPlan, Epics and Versions should not be disabled, epics panel toggle should be on.
            if (!KanPlanFeatureService.shouldShowEpicsPanel()) {
                return false;
            }

            // There are no epics
            if (getEpicsCount(mode) < 1) {
                return false;
            }

            // Do not show if dialog has been dismissed previously in current mode
            if (!KanPlanFeatureService.shouldShowEpicsOnboarding(mode)) {
                return false;
            }

            return true;
        };

        KanPlanInlineDialogController.updateEpicsOnboardingDialogPosition = function (epicsPanelState) {
            if (this.epicsOnboardingDialog) {
                this.epicsOnboardingDialog.updatePosition(epicsPanelState);
            }
        };

        KanPlanInlineDialogController.hide = function () {
            kanPlanInlineDialog.hide();
        };

        /**
         * @param {GH.GridDataModel} model
         * @returns {boolean}
         */
        KanPlanInlineDialogController.isVisible = function (model) {

            //The current rapidview is a Kanban board, with an unmapped Kanban backlog.
            if (!GH.RapidBoard.State.isKanbanBoard() || GH.RapidViewConfig.kanPlanEnabled) {
                return false;
            }

            // KanPlan should not be disabled.
            if (!KanPlanFeatureService.isFeatureEnabled()) {
                return false;
            }

            //The user is a board admin (or global admin)
            if (!(GH.UserData.userConfig.isGlobalAdmin || GH.RapidViewConfig.isBoardAdmin)) {
                return false;
            }


            //Dialog has not been shown to the user 1 or more times before on this board
            if (!!GH.BoardState.getPerViewValue(DIALOG_DISMISSED_KEY)) {
                return false;
            }

            //The board has at least the minimum number of mapped columns (visible columns)
            if (model.getColumns().length < COLUMN_MIN_THRESHOLD) {
                return false;
            }

            // The boards first column has more than the minimum number of visible issues.
            // Store the issueCount for analytics data
            issuesCount = model.getColumnIssueCount(model.getFirstColumn().id);
            if (issuesCount < ISSUE_COUNT_MIN_THRESHOLD) {
                return false;
            }
            return true;
        };

        KanPlanInlineDialogController.markAsDismissed = function () {
            GH.BoardState.setPerViewValue(DIALOG_DISMISSED_KEY, true);
        };

        KanPlanInlineDialogController.CONTINUE_EVENT = 'kanplan.inlineDialog.continue.event';

        return new Events(KanPlanInlineDialogController);


        function getEpicsCount(mode) {
            return mode === 'plan' ? GH.EpicController.getEpicModel().getEpicList().getIssueCount()
                : GH.GridDataController.getEpicsCount();
        }
    }
);