/* global AJS, jQuery, sinon, module, test, ok */
AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "jira.webresources:calendar", "jira.webresources:calendar-en", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    "use strict";

    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');

    var GlobalEvents = require('jira-agile/rapid/global-events');
    GlobalEvents.trigger('pre-initialization');

    var fakeEpic1 = GH.AjaxTest.Data.EpicsList.epics[0];
    var fakeEpic2 = GH.AjaxTest.Data.EpicsList.epics[2];
    var fakeEpicModel = new GH.EpicModel();

    module("EpicView", {
        setup: function setup() {
            jQuery.fx.off = true;
            QUnit.GhFixtures.addFixture("<div id='ghx-epic-column'></div>");
            this.sandbox = sinon.sandbox.create();
            GH.Test.setUpFakeServer(this);
            GH.Test.respondToGetWith200(this, "/xboard/config.json", GH.AjaxTest.Data.SampleScrumBoardConfig);
            GH.Test.respondToGetWith200(this, "/xboard/plan/backlog/data.json", GH.AjaxTest.Data.SampleScrumBoardWithEpicsAdded);
            GH.Test.respondToGetWith200(this, "/xboard/plan/backlog/epics.json", GH.AjaxTest.Data.EpicsList);
            fakeEpicModel.setData(GH.AjaxTest.Data.EpicsList);
            this.sandbox.stub(GH.EpicController, 'getEpicModel').returns(fakeEpicModel);
            this.sandbox.stub(GH.EpicView, 'getTwixieState'); // Otherwise test starting state is indeterminate
            GH.EpicView.getTwixieState.withArgs(fakeEpic1.key).returns('open'); // Otherwise test starting state is indeterminate
            GH.EpicView.getTwixieState.withArgs(fakeEpic2.key).returns('closed'); // Otherwise test starting state is indeterminate
            this.sandbox.stub(GH.EpicView, 'setTwixieState'); // Otherwise test starting state is indeterminate
            this.sandbox.stub(BacklogModel, 'getAllIssueListsNew').returns([]);
            this.oldUserConfig = GH.UserData.userConfig;
            this.oldEpicConfig = GH.EpicConfig.epicConfig;
            GH.UserData.userConfig = { canCreateIssue: true };
            GH.EpicConfig.epicConfig = { epicStatusDoneValueId: 10001 };
            this.toggleSpy = this.sandbox.spy(GH.EpicView, 'toggleFiltering');
            // this.dialogSpy = this.sandbox.spy(GH.EpicView ,'showCreateIssueAssociatedWithEpicDialog');
            GH.EpicView.showCreateIssueAssociatedWithEpicDialog = sinon.spy();
            this.sandbox.stub(GH.PlanController, 'isEpicsColumnVisible').returns(true);
        },
        teardown: function teardown() {
            this.sandbox.restore();
            GH.Test.restoreServer(this);
            GH.UserData.userConfig = this.oldUserConfig;
            GH.EpicConfig.epicConfig = this.oldEpicConfig;
            jQuery.fx.off = false;
        },
        setupScrum: function setupScrum() {
            this.sandbox.stub(GH.RapidBoard.State, 'isScrumBoard').returns(true);
            this.sandbox.stub(GH.RapidBoard.State, 'isKanbanBoard').returns(false);

            GH.EpicView.init();
            GH.EpicView.draw();
            GH.EpicView.updateEpics();
        },
        setupScrumWithError: function setupScrumWithError() {
            this.sandbox.stub(GH.RapidBoard.State, 'isScrumBoard').returns(true);
            this.sandbox.stub(GH.RapidBoard.State, 'isKanbanBoard').returns(false);

            GH.EpicView.init();
            GH.EpicView.draw();
            GH.EpicView.renderError();
        },
        setupKanban: function setupKanban() {
            this.sandbox.stub(GH.RapidBoard.State, 'isScrumBoard').returns(false);
            this.sandbox.stub(GH.RapidBoard.State, 'isKanbanBoard').returns(true);
            this.sandbox.stub(GH.PlanController, 'isEpicsPanelEnabled').returns(false);

            GH.EpicView.init();
            GH.EpicView.draw();
            GH.EpicView.updateEpics();
        }
    });

    test("Clicking the epic filters the backlog", function () {
        this.setupScrum();
        var $epic = jQuery(".ghx-classification-item[data-epic-id=" + fakeEpic1.id + "]");

        $epic.click();
        ok(this.toggleSpy.calledOnce, "should have filtered the backlog");
    });

    test("Clicking the expand/collapse icon collapses the epic info", function () {
        this.setupScrum();
        var $epic = jQuery(".ghx-classification-item[data-epic-id=" + fakeEpic1.id + "]");
        var $icon = jQuery(".js-epic-details-toggle", $epic);
        ok($epic.hasClass("ghx-open"), "should start open");

        $icon.click();
        ok($epic.hasClass("ghx-closed"), "shouldn't be considered open any more");
        ok(!$epic.hasClass("ghx-open"), "should be considered closed");
        ok(this.toggleSpy.notCalled, "should not toggle backlog filtering");
    });

    test("Epic doesn't have expanded/collapsed html if not expanded", function () {
        this.setupScrum();
        var $epic = jQuery(".ghx-classification-item[data-epic-id=" + fakeEpic1.id + "]");
        var $icon = jQuery(".js-epic-details-toggle", $epic);
        ok($epic.hasClass("ghx-open"), "should start open");
        ok($epic.find('.ghx-expanded').length === 1, "should have expanded info");

        $icon.click();
        ok($epic.hasClass("ghx-closed"), "shouldn't be considered open any more");
        ok(!$epic.hasClass("ghx-open"), "should be considered closed");
        ok($epic.find('.ghx-expanded').length === 0, "shouldn't have expanded info html if closed");
    });

    test("Epic doesn't have expanded/collapsed html if not expanded on init", function () {
        this.setupScrum();
        var $epic = jQuery(".ghx-classification-item[data-epic-id=" + fakeEpic2.id + "]");
        var $icon = jQuery(".js-epic-details-toggle", $epic);
        ok($epic.hasClass("ghx-closed"), "should start closed");
        ok($epic.find('.ghx-expanded').length === 0, "should not have expanded info html");

        $icon.click();
        ok($epic.hasClass("ghx-open"), "shouldn't be considered closed any more");
        ok(!$epic.hasClass("ghx-closed"), "should be considered open");
        ok($epic.find('.ghx-expanded').length === 1, "should have expanded info html if open");
    });

    test("Clicking the issue link within the epic does not filter the backlog", function () {
        this.setupScrum();
        var $epic = jQuery(".ghx-classification-item[data-version-id=" + fakeEpic1.id + "]");
        var $link = jQuery(".js-epic-key-link", $epic);
        $link.on("click", function (e) {
            e.preventDefault();
        }); // So we don't follow the href
        $link.click();
        ok(this.toggleSpy.notCalled, "should not toggle backlog filtering");
    });

    test("Clicking the 'Create issue in epic' link does not filter the backlog", function () {
        this.setupScrum();
        var $epic = jQuery(".ghx-classification-item[data-epic-id=" + fakeEpic1.id + "]");
        var $link = jQuery(".js-create-epic-issue", $epic);
        $link.on("click", function (e) {
            e.preventDefault();
        }); // So we don't follow the href
        $link.click();
        ok(this.toggleSpy.notCalled, "should not toggle backlog filtering");
    });

    test("Clicking the 'Create issue in epic' link shows a dialog", function () {
        this.setupScrum();
        var $epic = jQuery(".ghx-classification-item[data-epic-id=" + fakeEpic2.id + "]");
        var $icon = jQuery(".js-epic-details-toggle", $epic);
        //Link appears only on expanded epics
        $icon.click();
        var $link = jQuery('.js-create-epic-issue', $epic);
        $link.click();

        ok(GH.EpicView.showCreateIssueAssociatedWithEpicDialog.called, 'click on link should call a dialog');
        ok(this.toggleSpy.notCalled, "should not toggle backlog filtering");
    });

    test("Should display estimation group when Scrum", function () {
        this.setupScrum();
        console.info('isScrumBoard', GH.RapidBoard.State.isScrumBoard());
        var estimates = jQuery(".ghx-estimates");
        var progressBar = jQuery(".ghx-classification-progress");

        ok(estimates.length > 0, 'estimates should be visible');
        ok(progressBar.length > 0, 'progress bar should be visible');
    });

    test("Should not display estimation group when Kanban", function () {
        this.setupKanban();
        console.info('isScrumBoard', GH.RapidBoard.State.isScrumBoard());
        var estimates = jQuery(".ghx-estimates");
        var progressBar = jQuery(".ghx-classification-progress");

        ok(estimates.length === 0, 'estimates should not be visible');
        ok(progressBar.length === 0, 'progress bar should not be visible');
    });

    test("Should display error area", function () {
        this.setupScrumWithError();
        var $epicErrorArea = jQuery('#ghx-epic-column .ghx-classification-item.ghx-classification-error');
        var errorText = $epicErrorArea.find('.ghx-name .error').text();
        var errorButton = $epicErrorArea.find('.ghx-inner .aui-button');

        ok($epicErrorArea.length === 1, 'should render error');
        ok(errorText === 'gh.issue.error.epics.title', 'should display proper error text');
        ok(errorButton.length === 1, 'should display retry button');
        ok(errorButton.text() === 'gh.issue.error.retry', 'should display proper button text');
    });
});