/**
 * Epic view component.
 */
(function(){

const KanPlanInlineDialogController = require('jira-agile/rapid/ui/kanplan/KanPlanInlineDialogController');
const UiUtils = require('jira-agile/ui-utils');

GH.EpicView = {};

GH.EpicView.init = function() {
    GH.EpicView.initializeInlineEdits();

    // click handler for changing epic colour
    AJS.$(document).delegate('.js-change-epic-color', 'click', GH.EpicView.changeEpicColor);

    // click handler for marking epic as done
    AJS.$(document).delegate('.js-mark-epic-done', 'click', GH.EpicView.handleMarkEpicAsDone);
};

GH.EpicView.colors = [
    "ghx-label-1",
    "ghx-label-2",
    "ghx-label-3",
    "ghx-label-4",
    "ghx-label-10",
    "ghx-label-11",
    "ghx-label-5",
    "ghx-label-12",
    "ghx-label-6",
    "ghx-label-13",
    "ghx-label-7",
    "ghx-label-8",
    "ghx-label-9",
    "ghx-label-14"
];

/**
 * Options for this Epic view
 */
GH.EpicView.opts = {
    canClose: true
};

GH.EpicView.dropdowns = [];

GH.EpicView.draw = function() {
    const $column = AJS.$('#ghx-epic-column');

    // render the shell
    $column.html(GH.tpl.epic.renderShell({
        isEpicPanelExpanded: GH.PlanController.isEpicsColumnVisible(),
        canCreateIssue: GH.UserData.userConfig.canCreateIssue
    }));

    // register the create epic dialog
    GH.EpicQuickCreate.registerCreateEpicDialog();
};

GH.EpicView.hide = function() {
    AJS.$('#ghx-epic-column').empty();
};

/**
 * Render the epics into view and initialise Epic Drag and Drop functionality.
 */
GH.EpicView.updateEpics = function() {
    // render
    GH.EpicView.renderEpics();

    // fix up dnd
    GH.PlanDragAndDrop.initializeClassificationDnDEpics();
};

GH.EpicView.renderEpics = function() {

    // fetch the data to render
    var epicModel = GH.EpicController.getEpicModel();

    var $column = AJS.$('#ghx-epic-column');
    var $cardsContainer = $column.find('.ghx-classification-scrollview');

    var isEpicExpandedByKey = {};
    var epicsList = epicModel.getEpicList();
    var selectedEpicKey = GH.EpicController.getFilteredEpicKey();
    let showEstimates = GH.RapidBoard.State.isScrumBoard();

    _.each(epicsList.getAllIssues(), function(epic) {
        isEpicExpandedByKey[epic.key] = GH.EpicView.getTwixieState(epic.key) === 'open';

        if (showEstimates) {
            // render the estimate vs done values
            epic.epicStats.estimateDone = GH.BacklogStatistics.formatStatisticForRendering(GH.BacklogModel.estimationStatistic, epic.epicStats.doneEstimate);
            epic.epicStats.estimateTotal = GH.BacklogStatistics.formatStatisticForRendering(GH.BacklogModel.estimationStatistic, epic.epicStats.totalEstimate);
            epic.epicStats.isIssueCountStatistic = GH.BacklogModel.estimationStatistic.typeId === 'issueCount';
            epic.epicStats.estimateStatisticName = GH.BacklogModel.estimationStatistic.name;
        }
    });

    var renderEpicsContext = {
        epicKeys: epicsList.getVisibleRankables(),
        epics: epicsList.getAllIssues(),
        isEpicExpandedByKey: isEpicExpandedByKey,
        selectedEpicKey: selectedEpicKey,
        isNoneSelected: GH.PlanIssueListFiltering.isNoneEpicFilterKey(selectedEpicKey),
        isAllSelected: !selectedEpicKey,
        canEditEpics: epicModel.canUserEditEpics(),
        canMarkAsDone: GH.EpicConfig.canMarkEpicAsDone(),
        canCreateIssue: GH.UserData.userConfig.canCreateIssue,
        colors: GH.EpicView.colors,
        showPages: GH.BacklogModel.supportsPages(),
        showEstimates,
    };
    $cardsContainer.html(GH.tpl.epic.renderEpics(renderEpicsContext));

    // TODO: probably not needed anymore
    GH.PlanView.updateFixedElementPositioning();

    // click handlers
    $column.find('.ghx-classification-item').click(function(e) {
        // elements to be ignored
        if (AJS.$(e.target).closest('.js-epic-actions, .js-create-epic-issue, .js-editing-field, .js-epic-details-toggle, .js-epic-key-link, .js-view-entity-pages').length) {
            return;
        }

        GH.EpicView.toggleFiltering(AJS.$(this));
    });
    $column.find('.js-epic-details-toggle').click(GH.EpicView.toggleEpicDisplay);

    // ignore these - name uses anchor to inherit link colour, but is not a link
    $column.find('.js-epic-name').click(function(event){
        event.preventDefault();
    });

    // dropdowns
    var dropdownTriggers = AJS.$(".js-epic-actions");

    _.each(dropdownTriggers, function(trigger) {
        const $parent = AJS.$(trigger).closest('.ghx-classification-item');
        const content = GH.tpl.epic.renderEpicDropdown({
            issueKey: $parent.data('epic-key'),
            issueId: $parent.data('epic-id'),
            canEditEpics: epicModel.canUserEditEpics(),
            canMarkAsDone: GH.EpicConfig.canMarkEpicAsDone(),
            colors: GH.EpicView.colors,
        });

        const dropdown = AJS.Dropdown.create({
            trigger,
            content,
            alignment: AJS.LEFT,
        });
        AJS.$(dropdown).bind("showLayer", function () {
            GH.RapidBoard.Util.InlineEditable.submitActiveEdits();
        });

        GH.EpicView.dropdowns.push(dropdown);
    });

    // create epic issue
    $column.find(".js-create-epic-issue").click(GH.EpicView.showCreateIssueAssociatedWithEpicDialog);
    $column.find(".js-view-entity-pages").each(function(){
        var $trigger = AJS.$(this);
        var epicKey = $trigger.closest(".ghx-classification-item").attr('data-epic-key');
        GH.LinkedPagesController.initDialogTrigger(
            GH.LinkedPagesController.EPIC, GH.LinkedPagesController.PLAN, epicKey, $trigger
        );
    });

    GH.EpicView.dropdowns = _.flatten(GH.EpicView.dropdowns);

    GH.EpicController.applyVersionFiltering();

    KanPlanInlineDialogController.trigger('showepicsonboarding', 'plan');
};

GH.EpicView.renderError = function () {
    const $column = AJS.$('#ghx-epic-column');
    const $cardsContainer = $column.find('.ghx-classification-scrollview');

    const selectedEpicKey = GH.EpicController.getFilteredEpicKey();

    const renderEpicsContext = {
        isAllSelected: !selectedEpicKey,
        isNoneSelected: GH.PlanIssueListFiltering.isNoneEpicFilterKey(selectedEpicKey),
    };
    $cardsContainer.html(GH.tpl.epic.renderEpicsError(renderEpicsContext));

    // click handlers
    $column.find('.ghx-classification-item:not(.ghx-classification-error)').click(function () {
        GH.EpicView.toggleFiltering(AJS.$(this));
    });
    $column.find('.ghx-classification-item .js-try-again').click(function() {
        GH.BacklogController.updateEpicData();
    });
};

GH.EpicView.renderEpicsExpanded = function(epicKey) {
    const epicList = GH.EpicController.getEpicModel().getEpicList();
    const issue = epicList.getAllIssues()[epicKey];

    if (!issue) {
        return;
    }

    const $column = AJS.$('#ghx-epic-column');
    const showEstimates = GH.RapidBoard.State.isScrumBoard();
    const $epic = $column.find(`.ghx-classification-item[data-epic-key=${epicKey}]`);
    const $epicHeader = $epic.find(`.ghx-header`);

    const renderEpicExpandedContext = {
        issue,
        showEstimates,
        canCreateIssue: GH.UserData.userConfig.canCreateIssue,
        showPages: GH.BacklogModel.supportsPages(),
    };

    const $html = AJS.$(GH.tpl.epic.renderEpicExpanded(renderEpicExpandedContext));
    // Create epic issue
    $html.find(".js-create-epic-issue").click(GH.EpicView.showCreateIssueAssociatedWithEpicDialog);

    $html.insertAfter($epicHeader);
};

GH.EpicView.showCreateIssueAssociatedWithEpicDialog = function (event) {
    event.preventDefault();

    var epicKey = AJS.$(this).parents(".ghx-classification-item").attr("data-epic-key");
    var epicName = AJS.$(this).parents(".ghx-classification-item").find("[data-fieldname='epicLabel']").text();

    GH.RapidBoard.Util.InlineEditable.submitActiveEdits();

    var form = GH.EpicQuickCreate.createCreateStoryForm();

    // init epic picker with preselected epic
    form.bind("contentRefreshed", function () {
        AJS.$(document).unbind(JIRA.EpicPicker.READY_EVENT);
        AJS.$(document).unbind("QuickCreateIssue.validationError");
        AJS.$(document).bind(JIRA.EpicPicker.READY_EVENT, function (e, epicPicker) {
            epicPicker.setSelection(new AJS.ItemDescriptor({
                value: "key:" + epicKey,
                label: epicName
            }));
            // ensure epic picker is disabled
            setTimeout(function(){
                epicPicker.disable();
            }, 0);

            // after the form has a validation error, disable the epic picker
            AJS.$(document).bind("QuickCreateIssue.validationError", function(e){
                epicPicker.disable();
            });
        });
    });

    // callback to really add an issue to the epic
    var afterIssueCreated = function (issues) {
        return GH.EpicController.addIssuesToEpicOnServer(epicKey, _.pluck(issues, 'issueKey'));
    };

    // register
    var ISSUE_CREATE_CALLBACK_ID = 'addToEpicCallback';
    GH.BacklogController.registerIssueCreatedCallback(ISSUE_CREATE_CALLBACK_ID, afterIssueCreated);

    var dialog = form.asDialog({
        windowTitle:AJS.I18n.getText('gh.epic.operations.create.title', GH.EpicConfig.getStoryIssueTypeId()),
        id:"create-story-dialog"
    });

    // handler unregister when the dialog closes
    dialog.bind('Dialog.hide', function () {
        AJS.$(document).unbind(JIRA.EpicPicker.READY_EVENT);
        AJS.$(document).unbind("QuickCreateIssue.validationError");
        GH.BacklogController.unregisterIssueCreatedCallback(ISSUE_CREATE_CALLBACK_ID);
    });

    dialog.show();
};


GH.EpicView.hideDropdown = function() {
    _.each(GH.EpicView.dropdowns, function(dropdown) {
        dropdown.hide();
    });
};


GH.EpicView.toggleFiltering = function($clickedEpic) {
    // finish ongoing edits
    GH.RapidBoard.Util.InlineEditable.submitActiveEdits();

    GH.EpicView.deselectAllEpics();

    var epicKey = $clickedEpic.attr('data-epic-key'),
        selectedEpicKey;

    if (GH.EpicController.getFilteredEpicKey() === epicKey) {
        selectedEpicKey = null;
    } else {
        selectedEpicKey = epicKey;
        AJS.$('#ghx-epic-column').find('.ghx-classification-all').removeClass('ghx-selected');
        $clickedEpic.addClass('ghx-selected');
    }

    // update epic for filtering
    GH.EpicController.setFilteredEpicKey(selectedEpicKey);

    // apply epic filter
    GH.BacklogController.updateEpicFiltering(false);

    // update state
    GH.RapidBoard.State.pushState();
};

GH.EpicView.deselectAllEpics = function() {
    // deselect all epics
    var $epicsColumn = AJS.$('#ghx-epic-column');
    var $epics = $epicsColumn.find('.ghx-classification-item');
    $epics.removeClass('ghx-selected');
    $epicsColumn.find('.ghx-classification-all').addClass('ghx-selected');
};

GH.EpicView.setTwixieState = function(epicKey, value) {
    var key = 'epicTwix-' + epicKey;
    GH.BoardState.setPerViewValue(key, value);
};

GH.EpicView.getTwixieState = function(epicKey) {
    var key = 'epicTwix-' + epicKey;
    return GH.BoardState.getPerViewValue(key, 'closed');
};

// Epic toggle state used for Expand Collapse
GH.EpicView.toggleEpicDisplay = function(e) {
    var $expando = AJS.$(this);
    var $epic = $expando.parents('.ghx-classification-item');
    GH.EpicView.toggleEpicDisplayByElementWithAnimation($epic);
};

GH.EpicView.toggleEpicDisplayByElement = function($epic) {
    const epicKey = $epic.attr('data-epic-key');

    if ($epic.hasClass('ghx-open')){
        $epic.removeClass('ghx-open').addClass('ghx-closed');
        GH.EpicView.setTwixieState(epicKey, 'closed');
        $epic.find('.ghx-expanded').remove();
    } else if ($epic.hasClass('ghx-closed')){
        GH.EpicView.renderEpicsExpanded(epicKey);
        $epic.removeClass('ghx-closed').addClass('ghx-open');
        GH.EpicView.setTwixieState(epicKey, 'open');
    }
    UiUtils.updateExpanderA11yAttrs($epic);
};

GH.EpicView.toggleEpicDisplayByElementWithAnimation = function($epic) {
    const epicKey = $epic.attr('data-epic-key');
    const isOpen = $epic.hasClass('ghx-open');
    if (isOpen) {
        const expandedElement = $epic.find(".ghx-expanded");
        GH.RapidBoard.Animation.animateHeight(expandedElement, expandedElement.height(), 0).done(function() {
            $epic.removeClass('ghx-open').addClass('ghx-closed');
            GH.EpicView.setTwixieState(epicKey, 'closed');
            expandedElement.remove();
            UiUtils.updateExpanderA11yAttrs($epic);
        });
        UiUtils.updateExpanderA11yAttrs($epic, UiUtils.Aria.COLLAPSED);
    } else if ($epic.hasClass('ghx-closed')){
        GH.EpicView.renderEpicsExpanded(epicKey);
        const expandedElement = $epic.find(".ghx-expanded");
        $epic.removeClass('ghx-closed').addClass('ghx-open');
        GH.RapidBoard.Animation.animateHeight(expandedElement, 0, expandedElement.height()).done(function() {
            GH.EpicView.setTwixieState(epicKey, 'open');
            UiUtils.updateExpanderA11yAttrs($epic);
        });
        UiUtils.updateExpanderA11yAttrs($epic, UiUtils.Aria.EXPANDED);
    }
    GH.EpicController.sendAnalytics('epicToggled', {
        isOpen: !isOpen
    });
};

GH.EpicView.initializeInlineEdits = function() {

    // Validation for name, start and end date. These function also extract the value from the dom
    var validateName = function(editData) {
        var newValue = editData.editElement.val();
        if (!GH.Validation.notBlank(editData.editElement, AJS.I18n.getText('gh.epic.label.error.required'))) {
            return false;
        }
        editData.newValue = newValue;
        return true;
    };

    // Saves the changed data, we always update the complete information, not just a field
    var saveEpic = function(editData) {
        GH.EpicView.updateEpicLabel(editData.epicKey, editData.newValue);
        GH.RapidBoard.Util.InlineEditable.updateView(editData);
    };

    var getEpicData = function(viewElement) {
        // force other edits to close
        GH.RapidBoard.Util.InlineEditable.submitActiveEdits();

        var editData = GH.RapidBoard.Util.InlineEditable.defaultFieldData(viewElement);
        if (editData.fieldName === 'epicLabel') {
            editData.maxLength = 30;
        }
        // fetch the sprint id from the dialog
        editData.epicKey = viewElement.parents('.ghx-classification-item').attr('data-epic-key');
        return editData;
    };

    var getEpicLabelElement = function(event) {
        var epicKey = AJS.$(this).parents('.js-epic-actions-list').attr('data-epic-key');
        return AJS.$('.ghx-classification-item[data-epic-key="'+epicKey+'"]').find('.js-edit-epicLabel-trigger');
    };

    // edit Epic Label
    GH.RapidBoard.Util.InlineEditable.register('.js-edit-epic-label', {
        getViewElement: getEpicLabelElement,
        getData: getEpicData,
        validate : validateName,
        save : saveEpic,
        renderView: GH.RapidBoard.Util.InlineEditable.remoteTriggeredTextRenderView,
        preEdit: GH.RapidBoard.Util.InlineEditable.cancelActiveEdits
    });
};

GH.EpicView.updateEpicLabel = function(epicKey, newValue) {
    var labelFieldId = GH.EpicConfig.getEpicLabelFieldId();
    var epicsList = GH.EpicController.getEpicModel().getEpicList();
    var epic = epicsList.getIssueData(epicKey);

    var oldLabel = epic.epicLabel;
    var oldOriginalLabel = epic.originalEpicLabel;
    epic.epicLabel = newValue;
    epic.originalEpicLabel = newValue;

    epicsList.updateIssue(epic);

    // rerender all issues, as the label will have changed
    GH.BacklogView.draw();

    var successFn = function() {
        GH.EpicView._reloadDetailsView(GH.PlanController);
        GH.EpicController.sendAnalytics('updateEpicLabel', {labelLength: newValue.length});
    };

    var errorFn = function() {
        epic.epicLabel = oldLabel;
        epic.originalEpicLabel = oldOriginalLabel;
        epicsList.updateIssue(epic);
        GH.EpicView.renderEpics();
        GH.BacklogView.draw();
    };

    GH.EpicController.updateEpicField(epicKey, labelFieldId, newValue, successFn, errorFn);
};

GH.EpicView.changeEpicColor = function(event) {
    var elem = AJS.$(this);
    var colorId = elem.attr('data-color-id');
    var epicKey = elem.parents('.js-epic-actions-list').attr('data-epic-key');
    if (colorId && epicKey) {
        GH.EpicView.updateEpicColor(epicKey, colorId);
    }
};

GH.EpicView.updateEpicColor = function(epicKey, colorId) {
    var colorFieldId = GH.EpicConfig.getEpicColorFieldId();
    var epicsList = GH.EpicController.getEpicModel().getEpicList();
    var epic = epicsList.getIssueData(epicKey);
    var oldColor = epic.epicColor;
    epic.epicColor = colorId;
    epicsList.updateIssue(epic);
    GH.EpicView.renderEpics();
    GH.BacklogView.draw();

    var successFn = function() {
        // rerender the epics as well as all issues, as the label will have changed
        GH.EpicView._reloadDetailsView(GH.PlanController);
        GH.EpicController.sendAnalytics('updateEpicColor', {oldColorId: oldColor, newColorId: colorId});
    };

    var errorFn = function() {
        epic.epicColor = oldColor;
        epicsList.updateIssue(epic);
        GH.EpicView.renderEpics();
        GH.BacklogView.draw();
    };

    GH.EpicController.updateEpicField(epicKey, colorFieldId, colorId, successFn, errorFn);
};

/**
 * Scrolls the epics card controller to the bottom.
 */
GH.EpicView.scrollToBottom = function() {
    var $epicCardsScroller = AJS.$('.ghx-epic-column .ghx-classification-scrollview');
    if ($epicCardsScroller.length) {
        $epicCardsScroller.scrollTop($epicCardsScroller[0].scrollHeight);
    }
};

GH.EpicView.handleMarkEpicAsDone = function(event) {
    if (!GH.EpicConfig.canMarkEpicAsDone()) {
        return;
    }

    // cancel active edits
    GH.RapidBoard.Util.InlineEditable.cancelActiveEdits();

    var elem = AJS.$(this);
    var epicKey = elem.parents('.js-epic-actions-list').attr('data-epic-key');
    var epicsList = GH.EpicController.getEpicModel().getEpicList();
    var epic = epicsList.getIssueData(epicKey);
    var epicLabel = epic.epicLabel;
    var numIncompleteIssues = epic.epicStats.notDone;

    // pop a dialog for confirmation
    var confirmFn = function() {
        GH.EpicController.updateEpicAsDone(epicKey);
    };

    var header = AJS.I18n.getText("gh.epic.action.mark.done.dialog.header");
    var content = GH.tpl.epic.renderMarkEpicAsDoneDialog({
        epicName: AJS.escapeHTML(String(epicLabel)),
        epicKey: epicKey,
        numIncompleteIssues: numIncompleteIssues,
        rapidView:GH.RapidBoard.State.getRapidViewId()
    });
    var dialog = GH.ConfirmDialog.create("ghx-mark-epic-done-dialog", {
        width: 450,
        header: header,
        content: content,
        contentEscaping: false,
        onConfirmFn: confirmFn
    });

    dialog.show();

    // adjust height to fit content
    dialog.updateHeight();
};

GH.EpicView.showEpicMarkedAsDoneMessage = function() {
    GH.Notification.showSuccess(AJS.I18n.getText('gh.epic.action.mark.done.success'));
};

/**
 * Sets a filtered style on the epics with the given epic keys and removes any previous filtered styles applied to any epic
 *
 * @param epicKey
 */
GH.EpicView.setFilteredStyle = function(epicKeys){
    var epicColumn = AJS.$("#ghx-epic-column");
    var $epics = epicColumn.find(".ghx-classification-item").removeClass("ghx-filtered");
    $epics.each(function(index, element){
        var $epic = AJS.$(element);
        if(_.contains(epicKeys, $epic.attr("data-epic-key"))){
            $epic.addClass("ghx-filtered");
        }
    });
};

GH.EpicView.clearFilteredStyle = function(){
    var epicColumn = AJS.$("#ghx-epic-column");
    epicColumn.find(".ghx-classification-item").removeClass("ghx-filtered");
};

GH.EpicView.showEpicPagesUpdatedMessage = function(epicId, epicKey, context) {
    JIRA.IssueNavigator.setIssueUpdatedMsg({
        issueMsg: context.options.issueMsg,
        issueId: epicId,
        issueKey: epicKey
    });
};

GH.EpicView.renderNewEpics = function(epics) {
    var epicColumn = AJS.$("#ghx-epic-column");
    _.each(epics, function(epic) {
        var $epic = epicColumn.find('.ghx-classification-item[data-epic-key="' + epic.issueKey + '"]');
        if (! $epic.hasClass('ghx-open')) {
            GH.EpicView.toggleEpicDisplayByElement($epic);
        }

        GH.RapidBoard.Animation.animateHeight($epic, 0, 200, GH.RapidBoard.Animation.LONG_DURATION);
    });
};

GH.EpicView._reloadDetailsView = function(controller) {
    var EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
    controller.reloadDetailView(EditableDetailsViewReloadReason.VERSION_PANEL_CHANGED);
};
})();
