/**
 * Component taking care of the "release" feature
 */
GH.Dialogs.ReleaseVersionDialog = {};

GH.Dialogs.ReleaseVersionDialog.EVENT_VERSION_RELEASED = "versionReleased";
GH.Dialogs.ReleaseVersionDialog.EVENT_TIMEOUT = "timeout";
GH.Dialogs.ReleaseVersionDialog.dialog = undefined;

(function () {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.Dialogs.ReleaseVersionDialog.analytics = new AnalyticsTracker('gh.release');
})();

/**
 * Open the release version dialog
 * @param rapidViewId the rapid view to release from
 * @param columnId the column id to release
 */
GH.Dialogs.ReleaseVersionDialog.showDialog = function (rapidViewId, columnId) {
    AJS.dim();
    AJS.$('.aui-blanket').html('<div class="jira-spinner-container"><aui-spinner size="medium" class="jira-page-loading-indicator"></aui-spinner></div>');

    GH.Ajax.get({
        url: '/release/model.json',
        data: {
            rapidViewId: rapidViewId,
            columnId: columnId
        }
    }, "showReleaseVersionDialog").done(GH.Dialogs.ReleaseVersionDialog.renderDialog).fail(GH.Dialog.undim);
};

/**
 * Renders the "release" dialog where the details of the new version can be specified
 *
 * @param model : see ReleaseViewModel
 */
GH.Dialogs.ReleaseVersionDialog.renderDialog = function (model) {
    _.each(model.projectIssueDetails, function (project) {
        project.name = AJS.escapeHTML(String(project.name));
    });

    // create the dialog
    var dialog = GH.Dialogs.ReleaseVersionDialog.dialog = GH.Dialog.create({
        width: 500,
        height: 390,
        id: 'ghx-dialog-release',
        onCancelFn: function onCancelFn() {
            GH.Dialogs.ReleaseVersionDialog.dialog = undefined;

            // analytics tracking for clicking cancel
            GH.Dialogs.ReleaseVersionDialog.analytics.trigger('cancel', model.projectIssueDetails.length); // SAFE
        }
    });

    AJS.$('#' + dialog.id).addClass('ghx-dialog');
    dialog.addHeader(AJS.I18n.getText('gh.rapid.release.dialog.title'));
    dialog.addPanel('', 'panel');
    dialog.getCurrentPanel().html(GH.tpl.dialogs.releaseversion.renderReleaseDialog({ model: model }));
    dialog.addButton(AJS.I18n.getText('gh.rapid.release.dialog.ok'), function () {
        GH.Dialogs.ReleaseVersionDialog.release(model);
    }, 'aui-button aui-button-primary');
    dialog.addCancel(AJS.I18n.getText("gh.rapid.operations.cancel"), function () {
        dialog.cancel();
    });

    AJS.$('.aui-dialog').addClass('ghx-dialog');

    // initialise DatePicker
    var cal = new Calendar(); // GHS-5829 Need to call new calendar to set up short month names and day names when they are not defined
    Calendar.setup({
        firstDay: 0,
        inputField: 'ghx-release-date',
        button: 'ghx-release-date_trigger_c',
        align: 'Tl',
        singleClick: true,
        positionStyle: 'fixed',
        ifFormat: model.datePickerFormat
    });

    // analytics tracking for clicking release
    GH.Dialogs.ReleaseVersionDialog.analytics.trigger('start', model.projectIssueDetails.length); // SAFE

    // hook up issue nav links
    //    AJS.$('#ghx-release-navigator a').click(GH.Dialogs.ReleaseVersionDialog.prepareIssueNavigator);

    dialog.show();
    AJS.$('.aui-blanket').empty();

    AJS.$("#ghx-release-name").focus();
};

/**
 * Removes the "release" dialog
 */
GH.Dialogs.ReleaseVersionDialog.disposeDialog = function () {
    if (GH.Dialogs.ReleaseVersionDialog.dialog) {
        GH.Dialogs.ReleaseVersionDialog.dialog.dispose();
        GH.Dialogs.ReleaseVersionDialog.dialog = undefined;
    }
    return this;
};

/**
 * Release! (create versions in all projects, add issues, release versions, perform miracle).
 */
GH.Dialogs.ReleaseVersionDialog.release = function (model) {

    // analytics tracking for release
    GH.Dialogs.ReleaseVersionDialog.analytics.trigger('complete', model.projectIssueDetails.length); // SAFE

    var dialog = GH.Dialogs.ReleaseVersionDialog.dialog;
    var versionName = AJS.$('#ghx-release-name').val();

    dialog.disableControls();
    dialog.showSpinner();

    GH.Ajax.post({
        url: '/release',
        data: {
            versionName: versionName,
            releaseDate: AJS.$('#ghx-release-date').val(),
            description: AJS.$('#ghx-release-desc').val(),
            rapidViewId: model.rapidViewId,
            columnId: model.columnId
        },
        errorContextMap: {
            name: '#ghx-release-name',
            releaseDate: '#ghx-release-date_trigger_c'
        },
        onGlobalError: function onGlobalError(result) {
            if (504 == result.getGlobalErrors()[0].status) {
                dialog.addPage();
                dialog.addPanel('', 'timeout');
                dialog.addHeader(AJS.I18n.getText('gh.rapid.release.dialog.title'));
                dialog.getCurrentPanel().html(GH.tpl.dialogs.releaseversion.renderReleaseDialogTimeout({ model: AJS.$.extend({ versionName: versionName }, model) }));
                dialog.addButton(AJS.I18n.getText('gh.rapid.release.dialog.timeout.button'), function () {
                    AJS.$(GH).trigger(GH.Dialogs.ReleaseVersionDialog.EVENT_TIMEOUT);
                    GH.Dialogs.ReleaseVersionDialog.analytics.trigger('timeout', 'issueCount', model.projectIssueDetails.length); // SAFE

                    GH.Dialogs.ReleaseVersionDialog.disposeDialog();
                });

                GH.Notification.showErrors(result.error, false);
            }
        }
    }).done(function (result) {
        // no errors! first, remove dialog...
        GH.Dialogs.ReleaseVersionDialog.disposeDialog();

        // .. then show success message
        GH.Notification.showSuccess(result.success.message);

        // fire a completion event
        AJS.$(GH).trigger(GH.Dialogs.ReleaseVersionDialog.EVENT_VERSION_RELEASED);
    }).fail(function () {
        var dialog = GH.Dialogs.ReleaseVersionDialog.dialog;
        if (dialog) {
            dialog.hideSpinner();
            dialog.enableControls();
        }
    });
};