(function(){
/*
 * The following code is common to dialogs that we use around the "new" GH. Due to shortcomings in AJS.Dialog and
 * AUI, we need to do some extra things to get the right stuff we want.
 */
AJS.namespace("GH.Dialog");

const Dialog = require('aui/dialog');

/**
 * Instantiates an AJS.Dialog, but first removes any lingering DOM element for that dialog.
 * @param dialogParams the dialog parameters e.g.
 * <code>
 *     {
 *        id: 'ghx-dialog',
 *        width: 100,
 *        height: 100,
 *        onCancelFn(dialog): fn   // called in case of cancel
 *     }
 * </code>
 */
GH.Dialog.create = function(dialogParams) {
    // first remove the existing dialog element from the DOM
    AJS.$('#' + dialogParams.id).remove();

    // set default parameters
    dialogParams = _.defaults(dialogParams, {
        bindEscape: true
    });

    // create AJS.Dialog
    var dialog = new Dialog(dialogParams);

    var origShow = dialog.show;
    var show = function() {
        GH.Dialog.preShow();
        origShow.call(dialog);
        var button = AJS.$('#'+dialogParams.id).find('.dialog-button-panel .aui-button');
        if (button.is(":enabled")) {
            button.focus();
        }
        AJS.$('#'+dialogParams.id).addClass('ghx-dialog');
    };

    // add some extra goodness on
    _.extend(dialog, {
        // we want to be able to 'dispose' of the dialog cleanly in both Cancel and Submit paths
        dispose : GH.Dialog.generateCloseHandler(dialog),
        cancel : GH.Dialog.generateCloseHandler(dialog, dialogParams.onCancelFn),
        submit : GH.Dialog.generateCloseHandler(dialog, dialogParams.onConfirmFn),
        show: show,
        // disableControls can be called on click of the submit button to stop double-submitting of dialogs
        disableControls: GH.Dialog.generateDisableControlsFn(dialog),
        // enableControls can be called when the submit of a dialog fails to restore enabled state
        enableControls: GH.Dialog.generateEnableControlsFn(dialog),
        showSpinner: GH.Dialog.generateShowSpinnerFn(dialog),
        hideSpinner: GH.Dialog.generateHideSpinnerFn(dialog)
    });

    // always bind the hide handler
    GH.Dialog.bindEscapeHandler(dialog, !dialogParams.bindEscape);

    // then return the AJS.Dialog object
    return dialog;
};

/**
 * When setting up a step of a dialog, we always need to hide the video underneath and add
 * a 'Cancel' link which removes the dialog and reveals the video.
 */
GH.Dialog.addCancelButton = function(dialog) {
    dialog.addCancel(AJS.I18n.getText('gh.rapid.operations.cancel'), function(event) {
        dialog.cancel(dialog);
    });
};

/**
 * We want 'esc' event to behave the same as Cancel.
 * Until we get AUI 3.5, which fires an event on pressing 'esc', we have to manually handle the keyup
 * @param dialog the dialog we are operating on
 */
GH.Dialog.bindEscapeHandler = function(dialog, suppress) {
    // post AUI 3.5: this
    // AJS.bind("hide.dialog", handler);
    // until then...
    if (!suppress) {
        AJS.$(document).bind("keyup." + dialog.id, function(e) {
            // esc ascii code...
            if(e.keyCode == 27) {
                dialog.cancel(dialog);
            }
        });
    } else {
        JIRA.bind("Dialog.beforeHide", function (e, popup, reason, id) {
            if (id === dialog.id && reason === JIRA.Dialog.HIDE_REASON.escape) {
                e.preventDefault();
            }
        });
    }
};

/**
 * Generates a function which closes and disposes the dialog.
 *
 * @param dialog the dialog to operate on
 * @param callback an optional handler to execute extra stuff
 */
GH.Dialog.generateCloseHandler = function(dialog, callback) {
    // we bind the dialog in this closure (rather than providing it as a parameter) as the two
    //  places we use it have different prototypes. This is why we have this generate function
    return function() {
        dialog.remove();
        GH.Dialog.undim();

        // unbind the hide handler
        AJS.$(document).unbind("keyup." + dialog.id);
        AJS.unbind("remove.dialog");

        // extra callback
        if (callback) {
            callback(dialog);
        }
        // return false to stop propagation of events
        return false;
    };
};

/**
 * Generates the disableControls function for a dialog.
 * @param dialog
 */
GH.Dialog.generateDisableControlsFn = function(dialog) {
    return function () {
        AJS.$('#' + dialog.id + ' button').attr('disabled', 'disabled');
    };
};

/**
 * Generates the enableControls function for a dialog.
 * @param dialog
 */
GH.Dialog.generateEnableControlsFn = function(dialog) {
    return function () {
        AJS.$('#' + dialog.id + ' button').removeAttr('disabled');
    };
};

GH.Dialog.generateShowSpinnerFn = function(dialog) {
    return function() {
        AJS.$('#' + dialog.id)
            .find('.button-panel-cancel-link').css('visibility', 'hidden').before('<span class="ghx-spinner"/>');
    };
};

GH.Dialog.generateHideSpinnerFn = function(dialog) {
    return function() {
        AJS.$('#' + dialog.id)
            .find('.ghx-spinner').remove().end()
            .find('.button-panel-cancel-link').css('visibility', 'visible');
    };
};

/**
 * Sanitizes AJS.dim to not throw exceptions.
 *
 * AJS.dim doesn't work nicely when videos are present on a page. For more infos see for https://studio.atlassian.com/browse/AJS-703
 */
GH.Dialog.sanitizeAJSDim = function() {
    var dim = AJS.dim;
    AJS.dim = function() {
        try {
            dim.apply(this, arguments);
        } catch (err) {
            // do nothing
        }
    };
};

/**
 * Bind a dialog handler to a selector.
 *
 * Handles disabled class as well as default prevention for links
 */
GH.Dialog.registerDialog = function(selector, handler, extraData) {
    AJS.$(document)
        .undelegate(selector, 'click')
        .delegate(selector, 'click', function(event) {
            // ignore if action is disabled
            var action = AJS.$(this);
            if (action.hasClass('disabled')) {
                return;
            }
            // stop anchor causing jump
            event.preventDefault();

            // call the handler
            handler(event, extraData);
        }
    );
};

GH.Dialog.dim = function() {
    AJS.dim();
    AJS.$('.aui-blanket').html('<div class="jira-spinner-container"><aui-spinner size="medium" class="jira-page-loading-indicator"></aui-spinner></div>');
};

GH.Dialog.undim = function() {
    AJS.undim();
    AJS.$('.aui-blanket').empty();
};


/**
 * Hacks for the welcome page - hide video while having a dialog
 */
GH.Dialog.preShow = function() {
    AJS.$('.aui-blanket').empty();
};
})();