/**
 * Create View Dialog controller
 */
GH.Dialog.CreateView = {};

(function () {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.Dialog.CreateView.analytics = new AnalyticsTracker('gh.create.view');
})();

/**
 * Bind the opening of the dialog to the right elements via live event.
 */
GH.Dialog.CreateView.init = function () {
    GH.Dialog.registerDialog('.js-view-action-create', GH.StartWizardView.startWizard);
    GH.Dialog.registerDialog('.js-view-action-create-scrum', GH.Dialog.CreateView.openCreateScrum);
    GH.Dialog.registerDialog('.js-view-action-create-kanban', GH.Dialog.CreateView.openCreateKanban);
};

GH.Dialog.CreateView.openCreateScrum = function () {
    GH.Dialog.CreateView.openPresetDialog('scrum');
};

GH.Dialog.CreateView.openCreateKanban = function () {
    GH.Dialog.CreateView.openPresetDialog('kanban');
};

GH.Dialog.CreateView.open = function () {
    // fetch the view model for the create dialog and pass it on to the handler
    GH.Ajax.get({
        url: '/rapidview/createmodel'
    }, "createView").done(GH.Dialog.CreateView.openWizard);
};

/**
 * Opens the dialog to create a new project, filter and rapid view
 */
GH.Dialog.CreateView.openWizard = function (createViewModel) {
    // cancel callback
    var onCancelFn = function onCancelFn(dialog) {
        GH.Dialog.CreateView.analytics.trigger("cancel", dialog.curpage.toString()); // SAFE
    };

    var dialog = GH.Dialog.create({
        width: 600,
        height: 600,
        id: 'ghx-dialog-create-view',
        onCancelFn: onCancelFn
    });

    GH.Dialog.CreateView.analytics.trigger("start"); // SAFE

    // add two steps to the dialog
    GH.Dialog.CreateView.renderSelectPresetPage(dialog, createViewModel);
    dialog.addPage();
    GH.Dialog.CreateView.renderSelectNameAndProjectsPage(dialog, true);

    // return to the first page/panel
    dialog.gotoPage(0);
    dialog.gotoPanel(0);
    dialog.show();

    //    dialog.updateHeight();

    // this causes the user picker frother single select to be created
    // JIRA.trigger(JIRA.Events.NEW_CONTENT_ADDED, [dialog.getCurrentPanel().body]);
    AJS.$(dialog.getCurrentPanel().body).trigger("contentRefresh");
};

/**
 * Opens the dialog to create a new project, filter and rapid view
 */
GH.Dialog.CreateView.openPresetDialog = function (preset) {
    // cancel callback
    var onCancelFn = function onCancelFn(dialog) {
        GH.Dialog.CreateView.analytics.trigger("cancel", dialog.curpage.toString()); // SAFE
    };

    var dialog = GH.Dialog.create({
        width: 600,
        height: 600,
        id: 'ghx-dialog-create-view',
        onCancelFn: onCancelFn
    });
    dialog.selectedPreset = preset;

    GH.Dialog.CreateView.analytics.trigger("start", preset); // SAFE

    // add two steps to the dialog
    GH.Dialog.CreateView.renderSelectNameAndProjectsPage(dialog, false);

    // return to the first page/panel
    dialog.gotoPage(0);
    dialog.gotoPanel(0);
    dialog.show();
    dialog.updateHeight();
    // put focus on the first input element
    AJS.$('#' + dialog.id).find("input:visible:first").focus();
};

/**
 * Render the first page of the dialog. It's got two tabs, presets and advanced.
 */
GH.Dialog.CreateView.renderSelectPresetPage = function (dialog, createViewModel) {
    dialog.addHeader(AJS.I18n.getText('gh.rapid.view.create'));
    AJS.$('#' + dialog.id).addClass('ghx-dialog-create-view');

    var page = AJS.$(GH.tpl.createviewdialog.renderDialog({
        canSeeSavedFilters: createViewModel.canSeeSavedFilters
    }));
    dialog.addPanel(AJS.I18n.getText('gh.rapid.welcome.template.select'), page, 'ghx-dialog-create-view-page');

    GH.Dialog.CreateView.setupTabs(dialog.id);

    GH.Dialog.addCancelButton(dialog);

    // bind Click events of Kanban and Scrum buttons to proceed to next dialog
    var optionKanban = AJS.$('#' + dialog.id).find('.ghx-kanban button');
    var optionScrum = AJS.$('#' + dialog.id).find('.ghx-scrum button');

    function gotoSecondPage() {
        dialog.nextPage();
        dialog.updateHeight();

        // put focus on the first input element
        AJS.$('#' + dialog.id).find("input:visible:first").focus();
    }

    optionKanban.click(function () {
        dialog.selectedPreset = "kanban";
        gotoSecondPage();
    });

    optionScrum.click(function () {
        dialog.selectedPreset = "scrum";
        gotoSecondPage();
    });

    page.find("form").submit(function () {
        // this for some reason only hits the frother. the text field 'clicks the button' itself.
        GH.Dialog.CreateView.submitAdvanced();
        return false;
    });

    // create frother single select for the saved filters
    var element = AJS.$("#ghx-create-filter-select");
    if (element.size() > 0) {
        var singleSelect = new AJS.SingleSelect({
            element: element,
            width: 300,
            removeOnUnSelect: true,
            overlabel: AJS.I18n.getText('gh.rapid.view.select.filter'),
            itemAttrDisplayed: "label",
            errorMessage: '',
            ajaxOptions: {
                query: true,
                url: GH.Ajax.buildRestUrl('/savedfilter/list.json'),
                formatResponse: function formatResponse(response) {
                    if (response.errors) {
                        var errors = GH.Ajax._convertJiraErrors(response.errors);
                        var contexts = {
                            'searchName': '#ghx-create-filter-select'
                        };
                        GH.Ajax.handleContextualErrors(errors, contexts);
                        return false;
                    }

                    // clear out the errors
                    AJS.$('#ghx-create-filter-select').siblings(".ghx-error").remove();

                    var ret = [];
                    AJS.$(response.filters).each(function () {
                        ret.push(new AJS.ItemDescriptor({
                            // value of item added to select
                            value: this.id.toString(),
                            // title of lozenge -- This is already HTML-escaped by SingleSelect, so no need to do it here
                            label: this.name,
                            // html used in suggestion -- This is NOT HTML-escaped by SingleSelect, so we must do so explicitly
                            html: AJS.escapeHTML(String(this.name)),

                            // use the properties dump of the itemdecriptor to save our actual savedFilter object, since we need
                            // to render the permissions info outside the frother when the selection changes
                            savedFilter: this
                        }));
                    });
                    return ret;
                }
            }
        });

        // update the permissions info when the selection of the frother changes. The actual data has been stored against
        // a custom property of the itemdescriptor
        element.bind('selected', function (event, item) {
            GH.Dialog.CreateView.updatePermissionInfo(item.properties.savedFilter);
        });

        element.bind('unselect', function (event) {
            GH.Dialog.CreateView.updatePermissionInfo(null);
        });
    }

    // bind the Submit button for Advanced to the right handler
    var submitAdvanced = AJS.$("#ghx-create-advanced-submit");
    submitAdvanced.click(function (e) {
        dialog.disableControls();
        GH.Dialog.CreateView.submitAdvanced(dialog);
        return false;
    });
};

/**
 * Render the second page of the dialog, which is currently what you get after clicking "kanban" preset.
 */
GH.Dialog.CreateView.renderSelectNameAndProjectsPage = function (dialog, showBackButton) {
    var dialogEl = AJS.$('#' + dialog.id);

    // TODO: will only have an effect when selectedPreset has been set before creating the page
    var title = AJS.I18n.getText('gh.rapid.view.create');
    if (dialog.selectedPreset == 'scrum') {
        title = AJS.I18n.getText('gh.rapid.view.create.scrum');
    } else if (dialog.selectedPreset == 'kanban') {
        title = AJS.I18n.getText('gh.rapid.view.create.kanban');
    }
    dialog.addHeader(title);
    dialogEl.addClass('ghx-dialog');

    // render panel
    var page = AJS.$(GH.tpl.createviewdialog.renderCreateNewView());
    dialog.addPanel(title, page, 'ghx-dialog-create-view-page');

    // add a Back button
    if (showBackButton) {
        dialog.addButton(AJS.I18n.getText('gh.rapid.common.back'), function (dialog) {
            dialog.prevPage();
            dialog.updateHeight();
        }, 'aui-button');
    }

    // create an event handler which will toggle the disabled state of the next button based on errors in the form
    var nextDisableEventHandler = function nextDisableEventHandler() {
        var next = AJS.$('#' + dialog.id).find('.ghx-next');
        if (shouldNextBeDisabled()) {
            next.attr('disabled', 'disabled').addClass('js-disabled');
        } else {
            next.removeAttr('disabled').removeClass('js-disabled');
        }
    };

    var viewName = dialogEl.find('#ghx-dialog-create-view-viewname');
    // create the project MultiSelect control
    var projectIdsMultiSelect = new GH.ProjectPicker({
        selector: "#ghx-dialog-create-view-viewprojects",
        change: nextDisableEventHandler,
        currentProjectId: JIRA.API && JIRA.API.Projects ? JIRA.API.Projects.getCurrentProjectId() : null
    });

    // function to check if we have valid values in all form controls
    var shouldNextBeDisabled = function shouldNextBeDisabled() {
        return !viewName.val() || !projectIdsMultiSelect.hasValue() || projectIdsMultiSelect.hasErrors();
    };

    // bind a validation function to the Next button
    var formValidateFn = function formValidateFn(dialog) {
        if (shouldNextBeDisabled()) {
            var next = AJS.$('#' + dialog.id).find('.ghx-next');
            next.attr('disabled', 'disabled').addClass('js-disabled');
            return false;
        }
        return GH.Dialog.CreateView.submitPresets(dialog);
    };
    dialog.addButton(AJS.I18n.getText('gh.rapid.operations.create'), formValidateFn, 'ghx-next aui-button');

    // bind form submit to the next button
    page.find("form").submit(function () {
        formValidateFn(dialog);
        return false;
    });

    GH.Dialog.addCancelButton(dialog);

    // disable the Next button initially
    dialogEl.find('.ghx-next').attr('disabled', 'disabled').addClass('js-disabled');

    // bind event handler to viewName
    viewName.keyup(nextDisableEventHandler);
};

/**
 * Register AUI tabs for dialog and set up tab handling
 */
GH.Dialog.CreateView.setupTabs = function (dialogId) {
    // initialise tabs
    AJS.tabs.setup();

    // preselect tab if set
    var preferredTab = GH.Dialog.CreateView.State.getPreferredTab();
    if (preferredTab) {
        AJS.tabs.change(AJS.$("a[href=" + preferredTab + "]"));
    }

    // register click
    AJS.$('#' + dialogId + ' .menu-item').click(function () {
        var el = AJS.$(this);
        GH.Dialog.CreateView.State.setPreferredTab(el.find('a:first').attr('href'));
    });
};

GH.Dialog.CreateView.submitPresets = function (dialog) {
    // clear any existing errors
    AJS.$(dialog.getCurrentPanel().body).find('.ghx-error').remove();

    var preset = dialog.selectedPreset;
    var view = {
        name: AJS.$('#ghx-dialog-create-view-viewname').val(),
        projectIds: AJS.$('#ghx-dialog-create-view-viewprojects').val(),
        preset: preset
    };
    var errorContextMap = {
        'name': '#ghx-dialog-create-view-viewname',
        'projectIds': '#ghx-dialog-create-view-viewprojects',
        'preset': preset

    };
    var successFn = function successFn(result) {
        if (preset == "scrum") {
            GH.Dialog.CreateView.handleCreateResult(result, function (id) {
                return GH.RapidBoard.gotoRapidBoardPage(id, "planning");
            }, "preset", preset);
        } else {
            GH.Dialog.CreateView.handleCreateResult(result, GH.RapidBoard.gotoRapidBoardPage, "preset", preset);
        }
    };

    GH.Ajax.post({
        url: '/rapidview/create/presets',
        data: view,
        errorContextMap: errorContextMap
    }).done(successFn);
};

/**
 * Submits the advanced create view form
 */
GH.Dialog.CreateView.submitAdvanced = function (dialog) {
    var nameField = AJS.$('#ghx-view-name');
    var name = nameField.val();
    var filterField = AJS.$('#ghx-create-filter-select');
    if (!GH.Validation.notBlank(nameField, AJS.I18n.getText('gh.rapid.view.error.name.required'))) {
        dialog.enableControls();
        return;
    }
    if (!GH.Validation.notBlank(filterField, AJS.I18n.getText('gh.rapid.view.error.filter.required'), '#ghx-filter-error')) {
        dialog.enableControls();
        return;
    }

    var filterId = filterField.val() || [];

    GH.Ajax.post({
        url: '/rapidview/create/advanced',
        data: {
            'name': name,
            'filterId': filterId[0]
        },
        errorContextMap: {
            name: '#ghx-view-name',
            filterId: '#ghx-create-filter-select'
        }
    }).done(function (result) {
        GH.Dialog.CreateView.handleCreateResult(result, GH.RapidBoard.gotoRapidViewConfiguration, "advanced", "advanced");
    }).fail(dialog.enableControls);
};

/**
 * Fire off some analytics, set up the message and then redirect somewhere.
 */
GH.Dialog.CreateView.handleCreateResult = function (result, redirectFn, type, analyticsLabel) {
    GH.Dialog.CreateView.analytics.trigger("complete", analyticsLabel); // SAFE

    // infer preset "subType" from analytics label
    var subType = type != analyticsLabel ? analyticsLabel : false;
    GH.PersistentMessages.CreateViewMessage.setView(result.success.id, result.success.name, type, subType);

    redirectFn(result.success.id);
};

/**
 * Update the display of the JIRA filter permissions once the filter frother selection changes. If there is no filter selected,
 * the view is cleared.
 */
GH.Dialog.CreateView.updatePermissionInfo = function (savedFilter) {
    var html = savedFilter != null ? GH.tpl.rapid.view.renderPermissionInfo({ savedFilter: savedFilter }) : '';
    AJS.$('#ghx-create-permissioninfo-container').html(html);
};