GH.Dialogs.CreateStatus = {};
GH.Dialogs.CreateStatus.dialog = null;

GH.Dialogs.CreateStatus.EVENT_STATUS_CREATED = "statusCreated";
GH.Dialogs.CreateStatus.statusCategoryPicker = null;
GH.Dialogs.CreateStatus.statusNamePicker = null;
GH.Dialogs.CreateStatus.statusNameChangedTimeout = null;

(function () {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.Dialogs.CreateStatus.analytics = new AnalyticsTracker('gh.config.add.status.dialog.change');
})();

GH.Dialogs.CreateStatus.showDialog = function (rapidViewId) {

    GH.Dialogs.CreateStatus.renderDialog(rapidViewId);
    GH.Dialogs.CreateStatus.analytics.trigger("start"); // SAFE
};

/**
 * Renders the "create status" dialog
 */
GH.Dialogs.CreateStatus.renderDialog = function (rapidViewId) {
    // cancel callback
    var cancelFn = function cancelFn() {
        GH.Dialogs.CreateStatus.analytics.trigger("cancel"); // SAFE
    };

    // create the dialog
    var dialog = GH.Dialogs.CreateStatus.dialog = GH.Dialog.create({
        width: 575,
        height: 200,
        id: 'ghx-dialog-create-status',
        onCancelFn: cancelFn
    });
    dialog.addHeader(AJS.I18n.getText('gh.rapid.config.workflow.add.status.dialog.title'));
    dialog.addPanel('', 'panel');

    var $page = AJS.$(GH.tpl.dialogs.createstatus.renderCreateStatusDialog({}));
    dialog.getCurrentPanel().html($page);
    var submitFunction = function submitFunction() {
        dialog.disableControls();
        GH.Dialogs.CreateStatus.submit(dialog, rapidViewId);
        return false;
    };
    dialog.addButton(AJS.I18n.getText('gh.rapid.config.workflow.add.status.dialog.submit.label'), submitFunction, 'aui-button');
    // make sure that hitting Enter submits this form and not something else on the page
    $page.submit(submitFunction);
    GH.Dialog.addCancelButton(dialog);

    GH.Dialogs.CreateStatus.registerStatusCategoryControls(dialog);

    AJS.$('.aui-dialog').addClass('ghx-dialog');

    dialog.show();
    AJS.$('.aui-blanket').empty();

    GH.Dialogs.CreateStatus.registerStatusNameField();

    // give focus to first input element
    AJS.$('#ghx-status-name-field').focus();
};

/**
 * Create and add Single Select for the status name field
 */
GH.Dialogs.CreateStatus.registerStatusNameField = function () {

    // add the single select
    GH.Dialogs.CreateStatus.statusNamePicker = new AJS.SingleSelect({
        removeOnUnSelect: false,
        element: "#ghx-status-name",
        submitInputVal: true,
        userEnteredOptionsMsg: AJS.I18n.getText('gh.rapid.config.workflow.add.status.dialog.create.new'),
        ajaxOptions: {
            url: GH.Ajax.buildBareRestUrl('/rest/api/2/status'),
            query: true,
            formatResponse: function formatResponse(statuses) {
                var descriptor = new AJS.GroupDescriptor({
                    weight: 0,
                    label: AJS.I18n.getText("gh.rapid.config.workflow.add.status.dialog.picker.label")
                });

                _.each(statuses, function (status) {
                    descriptor.addItem(new AJS.ItemDescriptor({
                        value: status.name,
                        label: status.name,
                        html: AJS.escapeHTML(String(status.name))
                    }));
                });

                return [descriptor];
            }
        }
    });
};

/**
 * Set up the event listeners for the status name single select and text field
 */
GH.Dialogs.CreateStatus.bindStatusNameEvents = function () {
    var $statusName = AJS.$("#ghx-status-name-field");

    // bind to the select change
    AJS.$("#ghx-status-name").change(GH.Dialogs.CreateStatus.handleStatusNameChanged);

    $statusName.on('input', function () {
        clearTimeout(GH.Dialogs.CreateStatus.statusNameChangedTimeout);
        GH.Dialogs.CreateStatus.statusNameChangedTimeout = setTimeout(GH.Dialogs.CreateStatus.handleStatusNameChanged, 275);
    });
};

/**
 * Respond to events fired when a the user types into the status name field or selects a value from the status name
 * picker
 */
GH.Dialogs.CreateStatus.handleStatusNameChanged = function () {
    var val = AJS.$("#ghx-status-name-field").val() || "";
    var newStatusName = AJS.$.trim(val.toLowerCase());

    var result;
    if (!val) {
        result = AJS.$.Deferred().resolve([]);
    } else {
        result = AJS.$.get(GH.Ajax.buildBareRestUrl("/rest/api/2/status"));
    }

    // fetch the existing statuses from JIRA
    result.done(function (statuses) {

        // check to see if what we have typed matches any
        var existingStatus = _.find(statuses, function (status) {
            return status.name.toLowerCase() === newStatusName;
        });

        // if the status matches an already existing one, lock the category picker
        // with the already defined category and add errors to explain
        if (existingStatus) {
            GH.Dialogs.CreateStatus.statusCategoryPicker.disableWithValue(existingStatus.statusCategory.id);
            GH.Dialogs.CreateStatus.addStatusExistsErrors();

            // otherwise reset the picker and clear the error fields
        } else {
            GH.Dialogs.CreateStatus.statusCategoryPicker.enableWithValueIfDisabled(GH.StatusCategoryPicker.defaultStatusCategoryId);
            GH.Dialogs.CreateStatus.clearErrors();
        }
    });
};

/**
 * Display an error to the user that the status already exists and the cateogry cannot be selected
 */
GH.Dialogs.CreateStatus.addStatusExistsErrors = function () {
    AJS.$("#js-status-category-error").text(AJS.I18n.getText('gh.rapid.config.workflow.add.status.dialog.status.category.defined'));
};

/**
 * Clear existing errors from the status category picker
 */
GH.Dialogs.CreateStatus.clearErrors = function () {
    AJS.$("#js-status-category-error").text(AJS.I18n.getText('gh.rapid.config.workflow.add.status.dialog.status.category.description'));
};

/**
 * Add the status category picker if status categories are available on the user's JIRA
 */
GH.Dialogs.CreateStatus.registerStatusCategoryControls = function () {
    GH.Dialogs.CreateStatus.fetchStatusCategoryConfig().done(function (isEnabled) {
        if (isEnabled) {
            GH.Dialogs.CreateStatus.renderStatusCategoryPicker().done(function () {
                GH.Dialogs.CreateStatus.dialog.updateHeight();
            });
        }
    });
};

/**
 * Render the status category picker
 * @returns Deferred which is resolved when the picker renders successfully
 */
GH.Dialogs.CreateStatus.renderStatusCategoryPicker = function () {
    return AJS.$.get(GH.Ajax.buildBareRestUrl('/rest/api/2/statuscategory')).done(function (data) {
        // Don't allow selection of the 'No Category' status category.
        var categories = _.reject(data, function (category) {
            return category.key === "undefined";
        });

        AJS.$("#js-status-category-picker-container").html(GH.tpl.dialogs.createstatus.renderCategoryPicker());
        GH.Dialogs.CreateStatus.bindStatusNameEvents();
        GH.Dialogs.CreateStatus.statusCategoryPicker = GH.StatusCategoryPicker.build({
            element: AJS.$("#ghx-status-category-picker"),
            statusCategories: categories
        });
    });
};

/**
 * Query JIRA and test whether or not status categories are available
 */
GH.Dialogs.CreateStatus.fetchStatusCategoryConfig = function () {
    return AJS.$.get(GH.Ajax.buildBareRestUrl('/rest/api/2/status')).andThen(function (result) {
        return _.find(result, function (status) {
            return _.isObject(status.statusCategory);
        });
    });
};

/**
 * Submit the dialog
 */
GH.Dialogs.CreateStatus.submit = function (dialog, rapidViewId) {
    clearTimeout(GH.Dialogs.CreateStatus.statusNameChangedTimeout);

    // create the data we send to the server
    var statusName = AJS.$('#ghx-status-name').val() || [];
    var statusCategory = AJS.$('#ghx-status-category-picker').val();
    var data = {
        rapidViewId: rapidViewId,
        name: AJS.$.trim(statusName.pop()),
        statusCategoryId: parseInt(statusCategory, 10) || GH.StatusCategoryPicker.defaultStatusCategoryId
    };

    GH.Ajax.post({
        url: '/workflow/status/create.json',
        data: data,
        errorContextMap: {
            'name': '#ghx-status-name'
        }
    }).done(function (result) {
        // analytics for state change and also meta data
        GH.Dialogs.CreateStatus.analytics.trigger("complete"); // SAFE

        // close our dialog
        dialog.dispose();

        // tell parties about the success
        AJS.$(GH).trigger(GH.Dialogs.CreateStatus.EVENT_STATUS_CREATED, result.success);
    }).fail(function () {
        GH.Dialogs.CreateStatus.dialog.updateHeight();
        dialog.enableControls();
    });
};