/** globals
 * GH, GH.Ajax, GH.Notification, AJS
 **/
/**
 * Complete Sprint Dialog
 * @module jira-agile/rapid/ui/dialog/complete-sprint-dialog
 * @requires module:jquery
 * @requires module:jira-agile/rapid/ui/component/sprint-select
 */
define('jira-agile/rapid/ui/dialog/complete-sprint-dialog', ['require'], function(require) {
    'use strict';

    var $ = require('jquery');
    var SprintSelect = require('jira-agile/rapid/ui/component/sprint-select');

    // GLOBALS... FIX ME
    var Ajax = GH.Ajax;
    var GHDialog = GH.Dialog;
    var Notification = GH.Notification;
    var tpl = GH.tpl;

    var CompleteSprintDialog = {};

    CompleteSprintDialog.sprintSelect = null;

    CompleteSprintDialog.EVENT_SPRINT_COMPLETED = "sprintCompleted";

    CompleteSprintDialog.BACKLOG_ID = -1;

    CompleteSprintDialog.analyticsMetaData = function(metaData) {
        AJS.trigger('analyticsEvent', {name: 'gh.sprint.complete', data : metaData});
    };

    CompleteSprintDialog.showDialog = function(rapidViewId, sprintId) {
        AJS.dim();
        $('.aui-blanket').html('<div class="jira-spinner-container"><aui-spinner size="medium" class="jira-page-loading-indicator"></aui-spinner></div>');

        Ajax.get({
            url: '/sprint/' + sprintId + '/complete',
            data: {
                rapidViewId: rapidViewId,
                sprintId: sprintId
            }
        }, "completesprintdialog")
            .done(CompleteSprintDialog.renderDialog)
            .fail(function () {
                AJS.undim();
                $('.aui-blanket').empty();
            });
    };


    /**
     * Renders the "complete sprint" dialog where the details of the new version can be specified
     *
     * @param model : see CompleteSprintViewModel
     */
    CompleteSprintDialog.renderDialog = function(model) {
        var hasUnfinishedSubtasks = model.finishedParentsWithUnfinishedSubtasks !== undefined && model.finishedParentsWithUnfinishedSubtasks.length > 0;

        // create the dialog
        var dialog = GHDialog.create({
            width : 630,
            height : 300,
            id : 'ghx-dialog-complete-sprint'
        });

        dialog.addHeader(AJS.I18n.getText('gh.sprint.complete.dialog.title.with.name', model.sprint.name));
        dialog.addPanel('', 'panel');

        // if there are no offending issues, allow the user to complete the sprint
        if(!hasUnfinishedSubtasks)
        {
            var cal = new Calendar(); // GHS-5829 Need to call new calendar to set up short month names and day names when they are not defined
            dialog.getCurrentPanel().html(tpl.dialogs.completesprint.renderCompleteSprintDialog({ model : getViewModel(model) }));
            dialog.addButton(AJS.I18n.getText('gh.sprint.complete.dialog.ok.button.label'), function() {
                dialog.disableControls();
                dialog.showSpinner();
                CompleteSprintDialog.complete(dialog, model);
            },'aui-button ghx-complete-button');
            CompleteSprintDialog.$completeButton = $("#ghx-dialog-complete-sprint").find(".ghx-complete-button");

            if (shouldShowSprintSelector(model)) {
                initSprintSelect(model);
            }
        }
        else
        {
            dialog.getCurrentPanel().html(tpl.dialogs.completesprint.renderCanNotCompleteSprintDialog({ issues: model.finishedParentsWithUnfinishedSubtasks }));
        }

        GHDialog.addCancelButton(dialog);

        $('.aui-dialog').addClass('ghx-dialog');

        dialog.show();
        $('.aui-blanket').empty();

        $("#ghx-release-name").focus();
    };

    /**
     * Complete sprint
     * @param model : see CompleteSprintViewModel
     */
    CompleteSprintDialog.complete = function(dialog, model) {
        Ajax.put({
            url: '/sprint/' + model.sprint.id + '/complete',
            data: {
                rapidViewId: model.rapidViewId,
                sprintId: model.sprint.id,
                incompleteIssuesDestination: model.incompleteIssuesDestination
            },
            errorContextMap : {
                name : '#ghx-release-name',
                releaseDate : '#ghx-release-date'
            }
        })
            .done(function(result) {

                CompleteSprintDialog.analyticsMetaData({
                    sprintId: model.sprint.id,
                    completeIssueCount: model.completeIssues,
                    incompleteIssueCount: model.incompleteIssues,
                    partiallyCompleteIssueCount: model.partiallyCompleteIssues,
                    hasGoal: !!model.sprint.goal
                }); // SAFE

                // no errors! first, remove dialog...
                dialog.dispose();

                // .. then show success message
                Notification.showSuccess(result.success.message);

                // fire an event
                $(GH).trigger(CompleteSprintDialog.EVENT_SPRINT_COMPLETED, model.sprint.id);
            })
            .fail(function () {
                dialog.hideSpinner();
                dialog.enableControls();
            });
    };

    function initSprintSelect(model) {
        CompleteSprintDialog.sprintSelect = new SprintSelect();
        CompleteSprintDialog.sprintSelect.initDropdown();
        CompleteSprintDialog.sprintSelect.on('change', handleItemChanged.bind(null, model));
    }

    function shouldShowSprintSelector(model) {
        return model.incompleteIssues + model.partiallyCompleteIssues > 0;
    }

    function handleItemChanged(model, value) {
        if (value) {
            model.incompleteIssuesDestination = value;
            enableComplete();
        } else {
            disableComplete();
        }
    }

    function enableComplete() {
        CompleteSprintDialog.$completeButton.enable(true);
    }

    function disableComplete() {
        CompleteSprintDialog.$completeButton.enable(false);
    }

    function getViewModel(model) {
        var viewModel = {
            completeIssues : model.completeIssues,
            totalIncompleteIssues : model.incompleteIssues + model.partiallyCompleteIssues,
            unfinishedParentsWithFinishedSubtasks : model.unfinishedParentsWithFinishedSubtasks ? model.unfinishedParentsWithFinishedSubtasks.length : 0,
            incompleteIssueDestinations : model.incompleteIssueDestinations,
            shouldShowSprintSelector : shouldShowSprintSelector(model),
            defaultIncompleteIssueDestination : model.defaultIncompleteIssueDestination
        };
        return viewModel;
    }

    return CompleteSprintDialog;
});
