AJS.test.require(['com.pyxis.greenhopper.jira:gh-rapid-tests', 'com.pyxis.greenhopper.jira:gh-rapid-inline-editable-deps'], function() {
    require([
        'jira-agile/rapid/ui/detail/inlineedit/issue-editor-fake-server',
    ], function(
        FakeServer
    ) {
        var JIRAFormDialog = require('jira/dialog/form-dialog');
        var Flag = require('jira/flag');

        module('Issue editor wrapper tests', {
            setup: function() {
                this.sandbox = sinon.sandbox.create();
                this.context = AJS.test.mockableModuleContext();

                this.JIRAFormDialog = this.sandbox.spy(JIRAFormDialog);
                this.sandbox.stub(this.JIRAFormDialog.prototype, "show");
                this.context.mock('jira/dialog/form-dialog', this.JIRAFormDialog);

                this.sandbox.stub(Flag);
                this.context.mock('jira/flag', Flag);

                this.editor = this.context.require('jira-agile/rapid/ui/detail/inlineedit/issue-editor-wrapper');

                FakeServer.setup();
                this.editor.init({
                    $detailContainer : AJS.$(),
                    loadIssueDetailCallback : this.editor.loadIssueToEditor, // because the DetailsView.load will call to this method
                    getIssueIdCallback : function() {
                        return FakeServer.issueId;
                    },
                    getIssueKeyCallback : function() {
                        return FakeServer.issueKey;
                    },
                    getInlineEditableFieldsCallback: sinon.stub().returns([])
                });
                this.sandbox.stub(this.editor.getWrappedIssueEditor(), 'canDismissComment').returns(true);

                this.sandbox.stub(this.editor, 'getMode');
                this.focusOnIssue = this.sandbox.stub(this.editor, 'focusOnIssue');
                this.redirectTo = this.sandbox.stub(this.editor, 'redirectTo');
                this.linksCapturer = this.editor.getLinksCapturer();
            },

            teardown: function() {
                FakeServer.teardown();
                this.sandbox.restore();
                this.editor.destroy();
            }
        });


        asyncTest('issueLink moves focus to issue on board', function() {
            expect(2);
            this.sandbox.stub(this.editor, 'issueDisplayedOnBoard').returns(true);

            const linksCapturer = this.editor.getLinksCapturer();
            linksCapturer.trigger('linkToIssue', {issueKey: "GUGU-1"});

            window.setTimeout(() => {
                sinon.assert.calledOnce(this.focusOnIssue);
                sinon.assert.notCalled(this.redirectTo);
                start();
            }, 0);
        });

        asyncTest('issueLink moves focus to issue on board', function() {
            expect(2);
            this.sandbox.stub(this.editor, 'issueDisplayedOnBoard').returns(false);
            this.linksCapturer.trigger('linkToIssue', {issueKey: "GUGU-1"});

            window.setTimeout(() => {
                sinon.assert.notCalled(this.focusOnIssue);
                sinon.assert.calledOnce(this.redirectTo);
                start();
            }, 0);
        });

        test('When loading issues, should not call to server', function() {
            this.editor.loadIssueToEditor();
            FakeServer.respond();
            equal(FakeServer.getNumberOfServerCallsToLoadIssue(), 0, 'First time load an issue, must NOT hit the server');

            this.editor.loadIssueToEditor();
            FakeServer.respond();
            equal(FakeServer.getNumberOfServerCallsToLoadIssue(), 0, 'Reload the same issue, must NOT hit the server');

            FakeServer.issueId = 2;
            FakeServer.issueKey = 'ISS-2';
            this.editor.loadIssueToEditor();
            FakeServer.respond();
            equal(FakeServer.getNumberOfServerCallsToLoadIssue(), 0, 'Issue was changed, must NOT hit the server');

            this.editor.loadIssueToEditor();
            FakeServer.respond();
            equal(FakeServer.getNumberOfServerCallsToLoadIssue(), 0, 'Reload the same issue, must NOT hit the server');

            this.editor.getWrappedIssueEditor().updateIssueWithQuery({attachmentOrder : 'desc'});
            FakeServer.respond();
            equal(FakeServer.getNumberOfServerCallsToLoadIssue(), 0, 'View issue query has changed, must HIT the server');
        });

        test('After saving a field, do not call to server to load issue data. Instead, directly use the save response', function() {
            this.editor.loadIssueToEditor();
            FakeServer.respond();

            // trigger save on a field
            this.editor.getWrappedIssueEditor().editIssueController.save(FakeServer.createSavingFieldModel());
            FakeServer.respond();
            equal(FakeServer.getNumberOfServerCallsToSaveIssue(), 1, 'Expect a POST to AjaxIssueAction');
            equal(FakeServer.getNumberOfServerCallsToLoadIssue(), 0, 'After saved, directly use the save response, must NOT hit the server to load issue');
        });

        test('When saving and the server returns HTML, it fails gracefully', function () {
            this.editor.loadIssueToEditor();
            FakeServer.respond();
            FakeServer.setResponse(FakeServer.responses["200_html"]);
            this.editor.getWrappedIssueEditor().editIssueController.save(FakeServer.createSavingFieldModel());
            FakeServer.respond();

            sinon.assert.calledOnce(this.JIRAFormDialog);
            equal(this.JIRAFormDialog.args[0][0].content.text(), "Probably not logged in", "Expect correct Form Dialog content");
            sinon.assert.calledOnce(this.JIRAFormDialog.prototype.show);
        });

        test('When saving and the server returns a HTTP error, it fails gracefully', function () {
            this.editor.loadIssueToEditor();
            FakeServer.respond();
            FakeServer.setResponse(FakeServer.responses["500"]);
            this.editor.getWrappedIssueEditor().editIssueController.save(FakeServer.createSavingFieldModel());
            FakeServer.respond();

            sinon.assert.calledOnce(Flag.showMsg);
            sinon.assert.calledWith(Flag.showMsg, "", "<p>common.forms.ajax.error.dialog.heading</p><p>common.forms.ajax.error.dialog</p>", { type: "error" });
        });

        module('getFieldsData test', {
            setup: function() {
                this.context = AJS.test.mockableModuleContext();
                this.editor = this.context.require('jira-agile/rapid/ui/detail/inlineedit/issue-editor-wrapper');

                this.getInlineEditableFieldsCallback = sinon.stub().returns([]);

                this.editor.init({
                    getInlineEditableFieldsCallback: this.getInlineEditableFieldsCallback
                });
                sinon.stub(this.editor.getWrappedIssueEditor(), 'canDismissComment').returns(true);
                this.clock = sinon.useFakeTimers();
            },

            teardown: function() {
                this.editor.destroy();
                this.clock.restore();
            }
        });

        asyncTest('Should resolve the callback if the field data is defined', function() {
            var promise = this.editor._getFieldsData();
            promise.done(function() {
                start();
                ok(true, "Expected promise to be resolved.");
            });
            this.clock.tick();
        });

        asyncTest('Should not resolve callback if the field data is not defined', function() {
            this.getInlineEditableFieldsCallback.returns(null);
            var promise = this.editor._getFieldsData();
            promise.done(function() {
                start();
                ok(false, "Did not expect promise to resolve.");
            });

            this.clock.tick();
            start();
            ok(true);
        });

        module('Issue editor wrapper method tests', {
            setup: function() {
                this.context = AJS.test.mockableModuleContext();
                this.detailsModelStub = sinon.stub(GH.DetailsObjectFactory, 'getDetailsModel').returns({
                    isFieldInSection: function(fieldId, sectionId) {
                        return false;
                    }
                });
                this.editor = this.context.require('jira-agile/rapid/ui/detail/inlineedit/issue-editor-wrapper');
                this.loadIssueDetailCallback = sinon.spy();

                this.hasOtherEditsInProgress = sinon.stub();
                this.editor.init({
                    $detailContainer : AJS.$(),
                    loadIssueDetailCallback : this.loadIssueDetailCallback,
                    getInlineEditableFieldsCallback: function() {},
                    getIssueIdCallback : function() {},
                    getIssueKeyCallback : function() {},
                    hasOtherEditsInProgress: this.hasOtherEditsInProgress
                });

                var issueEditor = this.editor.getWrappedIssueEditor();
                this.canDismissCommentStub = sinon.stub(issueEditor, 'canDismissComment').returns(true);
                this.getEditsInProgressStub = sinon.stub(issueEditor.editIssueController, 'getEditsInProgress').returns([]);
                this.getDirtyEditsInProgressStub = sinon.stub(issueEditor.editIssueController, 'getDirtyEditsInProgress').returns([]);
            },

            teardown: function() {
                this.editor.destroy();
                this.canDismissCommentStub.restore();
                this.getEditsInProgressStub.restore();
                this.getDirtyEditsInProgressStub.restore();
                this.detailsModelStub.restore();
            }
        });

        test('saveSuccess event handler calls the loadIssueDetailCallback with the correct fieldId', function() {
            this.editor.getWrappedIssueEditor().trigger('saveSuccess', {
                savedFieldIds: ['magic-field']
            });

            var expectedArg = 'magic-field';

            ok(this.loadIssueDetailCallback.calledOnce, 'The callback was called');

            var actualArg = this.loadIssueDetailCallback.lastCall.args[0];
            equal(actualArg, expectedArg);
        });

        test('hasEditsInProgress with no edits', function() {
            var result = this.editor.hasEditsInProgress();

            ok(!result, 'Should not have edits in progress');
        });

        test('hasEditsInProgress with some edits', function() {
            this.getEditsInProgressStub.returns(['some-field', 'some-other']);

            var result = this.editor.hasEditsInProgress();

            ok(result, 'Should have edits in progress');
        });

        test('hasEditsInProgress with some dirty edits', function() {
            this.getDirtyEditsInProgressStub.returns(['some-field', 'some-other']);

            var result = this.editor.hasEditsInProgress();

            ok(result, 'Should have edits in progress');
        });
    });
});
