/**
 * Overrides module jira-agile/rapid/ui/detail/details-field-renderer to provide inline-editable capability for the detail view.
 */
define('jira-agile/rapid/ui/detail/inlineedit/inline-editable-details-field-renderer', [
    'require'
], function(
    require
) {
    'use strict';

    var _ = require('underscore');
    var DetailsFieldRenderer = require('jira-agile/rapid/ui/detail/details-field-renderer');
    var DetailsModel = GH.DetailsObjectFactory.getDetailsModel();
    var DetailsFieldUtils = require('jira-agile/rapid/ui/detail/editable/details-field-utils');

    var InlineEditableDetailsFieldRenderer = {};

    InlineEditableDetailsFieldRenderer.buildTabs = function(viewData, detailsSections) {
        var tabs = [];

        if (viewData.tabs.defaultTabs.length > 0) {
            this._addTabs(tabs, viewData.tabs.defaultTabs, viewData);
        }
        return tabs;
    };

    InlineEditableDetailsFieldRenderer._addTabs = function(tabs, tabsData, viewData) {
        _.each(tabsData, function(tabInstance) {
            if (tabInstance.tabId !== 'HEADER') {
                tabs.push({
                    tabId: tabInstance.tabId,
                    sections: []
                });
            }
        }.bind(this));

        _.each(tabsData, function (tabInstance) {
            var isStandardTab = tabInstance.tabId !== 'THIRD_PARTY_TAB';
            var tab = _.findWhere(tabs, {tabId: tabInstance.tabId});

            if (tab) {
                if (tabInstance.tabId === 'SUB_TASKS') {
                    this.addSubtaskTab(tab.sections, tabInstance, viewData.trackingStatisticFieldId, viewData.key);
                } else if (tabInstance.sections) {
                    _.each(tabInstance.sections, function(section) {
                        if (section.providerKey && (section.iconURL || section.iconFont || isStandardTab)) {
                            var tabViewId;
                            if (isStandardTab) {
                                tabViewId = section.providerKey.split(":")[1];
                            } else {
                                tabViewId = "ghx-tab-" + section.providerKey;
                            }

                            // set the title count for each tab
                            section.titleCount = this._getTitleCount(section, tabInstance);

                            var tabSection = _.extend(section, {
                                id: tabViewId.replace(/\./g, '-').replace(/:/g, '-'),
                                title: section.label,
                                hasTitleCount: !_.isUndefined(section.titleCount),
                                thirdPartyTab: !isStandardTab,
                                isLazyLoading: false,
                                subpanelHtmls: []
                            });

                            tab.sections.push(tabSection);
                        }
                    }.bind(this));
                }
            }
        }.bind(this));
    };

    InlineEditableDetailsFieldRenderer.addSubtaskTab = function(tabs, subtaskTabInstance, trackingStatisticFieldId, parentIssueKey) {
        if (!subtaskTabInstance.subtaskEntries) {
            return;
        }
        // we show the actions column if there is at least one action to show
        var showActionsColumn = !!_.find(subtaskTabInstance.subtaskEntries, function (subtask) {
            return subtask.canEdit || subtask.canDelete;
        });

        // fetch the subtasks aggregate estimate
        var showTrackingField = !_.isUndefined(trackingStatisticFieldId);
        var trackingFieldSum;
        if (showTrackingField) {
            var subtaskAggregateEstimateField = DetailsModel.getViewField('aggregatetimeestimate_subtasks');
            if (subtaskAggregateEstimateField) {
                trackingFieldSum = subtaskAggregateEstimateField.html;
            }
        }

        tabs.push({
            id: 'subtasks',
            title: AJS.I18n.getText('gh.boards.subtasks'),
            titleCount: subtaskTabInstance.subtaskEntries.length,
            hasTitleCount: true,
            subtasks: subtaskTabInstance.subtaskEntries,
            canCreateSubtask: DetailsModel.hasIssueOperation('create-subtask'),
            showActionsColumn: showActionsColumn,
            showTrackingField: showTrackingField,
            trackingFieldSum: trackingFieldSum,
            filteredSubtaskCount: subtaskTabInstance.subtaskEntries ? subtaskTabInstance.totalSubtaskCount - subtaskTabInstance.subtaskEntries.length : 0,
            parentIssueKey: parentIssueKey,
            iconFont: 'subtask'
        });
    };

    /**
     * Renders an entire 'module' (grouping) of fields
     * @param viewData the view data of the current issue
     * @param detailContainer is the detail view container
     * @param detailTabsContainer a list of fields which will be rendered inside this module
     * @param tabs to be rendered
     * @return the DOM element of the entire module structure (to be appended back into the issueFields)
     */
    InlineEditableDetailsFieldRenderer.renderTabs = function (viewData, detailContainer, detailTabsContainer, tabs) {
        // render tab structure
        var tabStructure = AJS.$(GH.tpl.inlineEditableDetailview.renderTabs({
            tabs: tabs
        }));
        var tabMenu = AJS.$(GH.tpl.inlineEditableDetailview.renderTabMenuStructure({
            tabs: tabs
        }));

        detailContainer.append(tabMenu);
        detailTabsContainer.append(tabStructure);

        this.renderTabsContent(tabs, detailContainer.find('.ghx-detail-nav-menu'), detailTabsContainer);
    };

    InlineEditableDetailsFieldRenderer.renderTabsContent = function (tabs, $detailTabsMenu, $detailTabsContainer) {
        // render fields into tabs
        _.each(tabs, function (tab) {
            DetailsFieldUtils.replaceTabNavigation(tab, $detailTabsMenu);
            var $tabPanel = $detailTabsContainer.find('#' + tab.tabId);
            $tabPanel.empty();
            // get a reference to the container for fields within this module
            _.each(tab.sections, function(section) {
                if (section.subtasks) {
                    $tabPanel.append(this.renderSubtasks(section));
                } else {
                    $tabPanel.append(this.renderTab(section));
                }
            }.bind(this));
        }.bind(this));
    };

    InlineEditableDetailsFieldRenderer.renderSubtasks = function (section) {
        var tabHtmlNodes = this.renderSubtaskAsHtmlNode(section);

        var headerLinks = {
            links: [],
            groups: []
        };

        if (section.canCreateSubtask && section.subtasks.length > 0) {
            headerLinks.links.push({
                id: "createSubtask",
                title: AJS.I18n.getText('createsubtaskissue.title'),
                label: AJS.I18n.getText('createsubtaskissue.title'),
                styleClass: "icon issueaction-create-subtask aui-icon aui-icon-small aui-iconfont-add issueaction-aui-icon"
            });
        }

        AJS.$.extend(section, {
            html: _.map(tabHtmlNodes, function (node) {
                return AJS.$('<DIV></DIV>').append(AJS.$(node).clone()).html();
            }).join(""),
            headerLinks: headerLinks,
            renderHeader: true,
            subpanelHtmls: []
        });
        return this.renderTab(section);
    };

    InlineEditableDetailsFieldRenderer.renderTab = function (section) {
        var sectionPanel = [];
        if (section.headerLinks) {
            var $tab = AJS.$(JIRA.Components.IssueViewer.Templates.Body.issuePanel({
                html: section.html,
                id: section.id,
                label: section.title,
                renderHeader: section.renderHeader,
                headerLinks: section.headerLinks,
                subpanelHtmls: section.subpanelHtmls
            }));
            $tab.addClass('ghx-detail-section');
            sectionPanel = $tab;
        } else {
            // This will be deleted when all tabs are migrated
            var $tabTitle = AJS.$(GH.tpl.detailview.renderTabTitle({tabTitle: section.title}));
            sectionPanel.append($tabTitle);
            sectionPanel.append(AJS.$(section.html));
        }

        if (!section.thirdPartyTab && !section.iconFont && !section.iconURL) {
            sectionPanel.addClass('ghx-no-tab-menu');
        }

        if (section.tabIconId) {
            sectionPanel.attr('ghx-tab-icon-id', section.tabIconId);
        } else if (!section.iconFont && !section.iconURL) {
            sectionPanel.addClass('ghx-hidden-tab');
        }

        return sectionPanel;
    };

    InlineEditableDetailsFieldRenderer._getTitleCount = function (section, tabInstance) {
        var result;

        if (!_.isUndefined(section.titleCount)) {
            result = section.titleCount;
        } else {
            switch (tabInstance.tabId) {
                case "ATTACHMENT":
                    result = tabInstance.totalCount;
                    break;
                case "COMMENT":
                    result = tabInstance.totalComments;
                    break;
            }
        }

        return result;
    };

    InlineEditableDetailsFieldRenderer.renderEstimationAndTimeTracking = function (viewData, detailHead) {
        var $detailStatisticGroup = detailHead.find('.ghx-statistic-group');

        var estimateStatisticField = DetailsModel.getPrimaryEstimateStatisticField();
        var timeTrackingFields = DetailsModel.getTimeTrackingFields();
        if (!estimateStatisticField && !timeTrackingFields) {
            $detailStatisticGroup.empty();
        } else {
            this.renderEstimateStatisticField(estimateStatisticField, viewData, $detailStatisticGroup);

            if (!timeTrackingFields) {
                $detailStatisticGroup.find('.ghx-remaining-group').remove();
            } else {
                this.renderTimeTrackingFields(timeTrackingFields, viewData, $detailStatisticGroup);
            }
        }
    };

    InlineEditableDetailsFieldRenderer.renderEstimateStatisticField = function (estimateStatisticField, viewData, $detailStatisticGroup) {
        if (!estimateStatisticField) {
            $detailStatisticGroup.find('.js-estimate').remove();
        } else {
            var fieldToRender = _.extend({}, estimateStatisticField, {
                dataTooltip: estimateStatisticField.label,
                label: AJS.I18n.getText('gh.issue.estimate')
            });
            this.renderStatisticFields([fieldToRender], '.js-estimate', viewData, $detailStatisticGroup);
        }
    };

    InlineEditableDetailsFieldRenderer.renderTimeTrackingFields = function (timeTrackingFields, viewData, $detailStatisticGroup) {
        var trackingFields = [];
        trackingFields.push(_.extend({}, timeTrackingFields.currentIssueRemainingEstimateField, {
            dataTooltip: timeTrackingFields.currentIssueRemainingEstimateField.label,
            label: AJS.I18n.getText('gh.issue.remaining')
        }));

        // case 01: Sum everything
        if (timeTrackingFields.aggregateRemainingField) { // only available if subtask exists
            var tooltip = GH.tpl.detailview.renderAggregateRemainingEstimateTooltip({
                data: [
                    {
                        label: AJS.I18n.getText('gh.rapid.detail.view.fields.remainingestimate.parent'),
                        value: timeTrackingFields.currentIssueRemainingEstimateField.text
                    },
                    {
                        label: timeTrackingFields.aggregateRemainingSubtaskField ? timeTrackingFields.aggregateRemainingSubtaskField.label : null,
                        value: timeTrackingFields.aggregateRemainingSubtaskField ? timeTrackingFields.aggregateRemainingSubtaskField.text : null,
                    }
                ]
            });

            trackingFields.push(_.extend({}, timeTrackingFields.aggregateRemainingField, {
                dataTooltip: tooltip,
                label: AJS.I18n.getText('gh.issue.remaining.sum')
            }));
        }

        this.renderStatisticFields(trackingFields, '.js-remainings', viewData, $detailStatisticGroup);
    };

    InlineEditableDetailsFieldRenderer.renderStatisticFields = function (fields, containerSelector, viewData, $detailStatisticGroup) {
        var $detailStatisticContainer = $detailStatisticGroup.find(containerSelector);
        $detailStatisticContainer.empty();
        _.each(fields, function (theField) {
            var $statisticDom = this.renderField(viewData, theField);
            $detailStatisticContainer.append($statisticDom);
            // if the field is "empty" then we add an extra class to the container to call out the editing of it.
            if (_.isUndefined(theField.text)) {
                $detailStatisticContainer.addClass('ghx-empty');
            } else {
                $detailStatisticContainer.removeClass('ghx-empty');
            }
        }.bind(this));
    };

    return _.extend(DetailsFieldRenderer, InlineEditableDetailsFieldRenderer);
});
