/* global _ */
/**
 * Details view component.
 */
define('jira-agile/details-view', [
    'jira-agile/rapid/analytics-tracker'
], function (AnalyticsTracker) {

    const DetailsView = new GH.Events();

    DetailsView.minWidth = 400;

    /**
     * Fired when the details pane is finished updating
     */
    DetailsView.API_EVENT_DETAIL_VIEW_UPDATED = "GH.DetailView.updated";

    DetailsView.EVENT_DETAIL_VIEW_CLOSED = "GH.DetailView.closed";

    DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED = "GH.DetailView.update.started";

    DetailsView.selectedIssueKey = undefined;

    DetailsView.rapidViewId = undefined;

    DetailsView.rapidViewConfig = undefined;

    DetailsView.containerSelector = undefined;

    DetailsView.TAB_SUBTASKS = 'subtasks';

    DetailsView.$lastSectionSpacer = undefined;

    /**
     * Options for this detail view
     */
    DetailsView.opts = {
        canClose: true,
        showActionsCog: true,
        showSubtaskTab: true,
        showLogWorkTrigger: false,
    };

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    DetailsView.analytics = new AnalyticsTracker('gh.issueaction.issuedetail');

    DetailsView.triggerAnalyticsOpenDetailView = function () {
        let data = {mode: DetailsView.getMode()};
        AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.open', data: data});
        DetailsView.analytics.openDetailViewTime = Date.now();
    };

    DetailsView.triggerAnalyticsCloseDetailView = function () {
        // Trigger analytics event [SW-2193]
        var openTime = DetailsView.analytics.openDetailViewTime;
        var now = Date.now();
        // Open the detail view more than 3s
        if (!openTime || now - openTime > 3000) {
            AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.close'});
        } else {
            AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.dismiss'});
        }
        DetailsView.analytics.openDetailViewTime = null;
    };

    DetailsView.triggerAnalyticsEventForAction = function (action, data) {
        AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.' + action, data: data});
    };

    /**
     * initializes the detail view stuff
     */
    DetailsView.init = function () {
        GH.DetailsObjectFactory.mergeDetailsViewTemplate();

        // tab selection
        AJS.$(document).bind("tabSelect", DetailsView.handleTabSelection);

        // statistic editing might change when issue is updated
        AJS.$(GH).bind("issueUpdated", DetailsView.handleIssueUpdated);

        AJS.$(document).delegate('.js-detailclose', "click", function (e) {
            if (DetailsView.closeHandler) {
                DetailsView.closeHandler(e);
                DetailsView.triggerAnalyticsCloseDetailView();
            }
        });
        DetailsView._postInit();

        // field edit functionality
        GH.DetailsFieldEdit.init();
    };

    DetailsView.getMode = function () {
        return GH.RapidBoard.ViewController.getMode();
    };

    DetailsView.setContainerSelector = function (selector) {
        DetailsView.containerSelector = selector;
        DetailsView._onContainerSelectorSet(selector);
    };

    DetailsView._onContainerSelectorSet = function (selector) {
    };

    DetailsView.setOptions = function (opts) {
        DetailsView.opts = opts;
        DetailsView.closeHandler = DetailsView.opts.closeHandler;
    };

    /**
     * Get the container element for the detail view panel
     * @returns {jQuery}
     */
    DetailsView.getContainer = function () {
        if (!DetailsView.containerSelector) {
            // This error case happens on initial page load because ViewController calls hide()
            // on all modes except for the active one.
            // Both PlanController and WorkController will then attempt to hide the detail view.
            GH.log('tried to get container of DetailsView before selector was set');
            return AJS.$();
        }

        return AJS.$(DetailsView.containerSelector);
    };

    /**
     * Get the container element for all content inside the detail view panel
     * @returns {jQuery}
     */
    DetailsView._getContentContainer = function () {
        return DetailsView.getContainer().find('.ghx-detail-contents');
    };

    /**
     * Get the root element representing the issue in detail view.
     * @returns {jQuery}
     */
    DetailsView._getIssueElement = function () {
        return DetailsView.getContainer().find('.ghx-detail-issue');
    };

    DetailsView._replaceUpdatedFields = function ($issueElement, $newIssueElement, fieldId) {
        // Replace the field that was saved
        DetailsView._replaceField($issueElement, $newIssueElement, fieldId);

        function isNotEditing(field) {
            return !_.contains(DetailsView.getEditsInProgress(), field);
        }

        function isTimeTracking(field) {
            return _.contains(['timeestimate', 'timeoriginalestimate'], field);
        }

        if (fieldId === 'timeoriginalestimate' && isNotEditing('timeestimate')) {
            DetailsView._replaceField($issueElement, $newIssueElement, 'timeestimate');
        }

        if (fieldId === 'timeestimate' && isNotEditing('timeoriginalestimate')) {
            DetailsView._replaceField($issueElement, $newIssueElement, 'timeoriginalestimate');
        }

        if (isTimeTracking(fieldId)) {
            DetailsView._replaceField($issueElement, $newIssueElement, 'aggregatetimeestimate');
        }
    };

    DetailsView._replaceField = function ($oldIssueElement, $newIssueElement, fieldId) {
        var fieldSelector = DetailsView._InlineEditor.getFieldSelector(fieldId);
        var $oldField = $oldIssueElement.find(fieldSelector);
        var $newField = $newIssueElement.find(fieldSelector);

        if ($newField.length === 0) {
            $oldField.closest('li, dl').remove();
        } else {
            $oldField.replaceWith($newField);
            DetailsView._InlineEditor.triggerFieldUpdate(fieldId, $oldIssueElement, $newField);
        }
    };

    DetailsView._setIssueElement = function ($newIssueElement, savedFieldId) {
        var $issueElement = DetailsView._getIssueElement();

        if ($issueElement.length === 0) {
            DetailsView._getContentContainer().prepend($newIssueElement);
            return;
        }

        if (savedFieldId && DetailsView.hasEditsInProgress()) {
            // If there are edits in progress, we can't just replace the whole details view, because the user
            // will lose their unsaved changes. So we only replace the saved field and any other fields which we think may have updated.
            DetailsView._replaceUpdatedFields($issueElement, $newIssueElement, savedFieldId);
        } else {
            $issueElement.replaceWith($newIssueElement);
        }
    };

    /**
     * Set the current rapid view id
     */
    DetailsView.setRapidViewId = function (rapidViewId) {
        DetailsView.rapidViewId = rapidViewId;
    };

    /**
     * Set the current rapid view config
     */
    DetailsView.setRapidViewConfig = function (rapidViewConfig) {
        DetailsView.rapidViewConfig = rapidViewConfig;

        // also update DetailsView.opts to consider information from config
        // showLogWorkTrigger also depends on whether or not the tracking statistic is enabled for board
        // and it ALSO depends on whether or not the issue/user is allowed to log work, but that comes later
        if (DetailsView.opts.showLogWorkTrigger) {
            DetailsView.opts.showLogWorkTrigger = !_.isUndefined(rapidViewConfig.trackingStatistic) && rapidViewConfig.trackingStatistic.isEnabled;
        }
    };

    /**
     * Set the displayed issue
     */
    DetailsView.setSelectedIssueKey = function (issueKey) {
        DetailsView.selectedIssueKey = issueKey;
    };
    DetailsView.clearSelectedIssueKey = function () {
        DetailsView.selectedIssueKey = undefined;
    };

    /**
     * Renders the details view
     * @fires GH.DetailsView#before:render if the view panel is about to be rendered
     */
    DetailsView.show = function (detailsViewLoadReason) {
        AJS.$.when(WRM.require('wrc!gh-rapid-detailsview')).done(function () {
            // finish any previous edits
            DetailsView._preShow();
            GH.DetailsFieldEdit.completeEdits();

            // TODO: this is incorrect! we delete current view data before having new view data around!
            // In case loading of new data fails we still show the current details, but the underlying data has gone.
            // ensure we don't keep stale data around
            if (!detailsViewLoadReason) {
                GH.DetailsObjectFactory.getDetailsModel().clearViewData();
            }

            // set up DOM
            var container = DetailsView.getContainer();
            if (container.find('#ghx-detail-contents').length === 0) {
                var skeleton = GH.tpl.inlineEditableDetailview.renderIssueDetailsSkeleton();
                container.empty().append(skeleton);
                container.width(DetailsView.getWidth());
                DetailsView.initTooltips();

                DetailsView.triggerAnalyticsOpenDetailView();
            }

            if (!DetailsView.selectedIssueKey) {
                DetailsView._getIssueElement().remove();
                return;
            }

            DetailsView.trigger('before:render');

            DetailsView.load(null, detailsViewLoadReason);

            DetailsView._postShow();
        });
    };

    DetailsView.getWidth = function () {
        var storedWidthPercentage = GH.RapidBoard.State.getDetailViewWidth();
        var totalWidth = DetailsView.getContentElementWidth() + AJS.$("#ghx-detail-view").width();
        var calculatedWidth = storedWidthPercentage * totalWidth;

        var width = Math.max(calculatedWidth, DetailsView.minWidth);

        return Math.min(DetailsView.getMaxWidth(), width);
    };

    DetailsView.initTooltips = function () {
        // for showing tooltip (Tipsy) on any element in the detail view.
        // how to use:
        //  - add the tooltip text in the element's DOM under attribute "data-tooltip"
        //  - Add the selector of that element into the array below
        // that's it
        //
        // The reason why we don't use the generic selector [data-tooltip] is: JIRA platform's editable fields may have
        // its tooltip (for example "status" field). If we use [data-tooltip] as a all-in-one selector, we will show a
        // duplicated tooltip
        var targetElementSelectors = [
            '.ghx-statistic-group [data-tooltip]',
            // add more selectors here
        ];

        GH.Tooltip.tipsify({
            selector: targetElementSelectors.join(', '),
            context: '#ghx-detail-contents',
            html: true
        });
    };

    /**
     * Hides the details view
     */
    DetailsView.hide = function () {
        // abort any existing requests
        if (DetailsView.currentPromise && DetailsView.currentPromise.abort) {
            DetailsView.currentPromise.abort();
        }
        // finish any ongoing edits
        GH.DetailsFieldEdit.completeEdits();

        // clear the container
        DetailsView.getContainer().empty();
        AJS.$('#gh').removeClass('js-ghx-detail-loading').removeClass('js-ghx-detail-loaded');
        AJS.$(GH).trigger(DetailsView.EVENT_DETAIL_VIEW_CLOSED);

        DetailsView._postHide();
    };

    DetailsView.resizeLastSectionSpacer = function () {
        var $detailsNavContent = AJS.$('#ghx-detail-nav');
        var $detailsSections = $detailsNavContent.find('.ghx-detail-section');
        var $lastSectionSpacer;

        // add extra space in the last section if the view port is bigger than the content to allow to scroll to the bottom

        // get the current spacer height, if existing, to include it in the calculation
        var lastSectionSpacerPreviousHeight = 0;
        $lastSectionSpacer = $detailsNavContent.data('$lastSectionSpacer');
        if ($lastSectionSpacer) {
            lastSectionSpacerPreviousHeight = $lastSectionSpacer.height();
        }

        var viewPortHeight = $detailsNavContent.height();
        var $lastSection = $detailsSections.last();
        var lastSectionHeight = $lastSection.height() - lastSectionSpacerPreviousHeight;
        if (viewPortHeight > lastSectionHeight) {
            if (!$lastSectionSpacer) {
                $detailsNavContent.data('$lastSectionSpacer', $lastSectionSpacer = AJS.$('<div>', {css: {clear: 'both'}}).appendTo($lastSection));
            }

            // seems that IE calculates the size we want, but the others browser add an extra of 32px, we need to remove
            // that from the spacer height to not get extra scrolling space
            var empiricalExtraHeight = !AJS.$.browser.msie ? 40 : 0;

            $lastSectionSpacer.height(viewPortHeight - lastSectionHeight - empiricalExtraHeight);
        }
    };

    DetailsView.selectNavItem = function (sectionId) {
        var triggerAnalyticsEventOnScrollEnd = _.debounce(function (sectionId) {
            // Trigger analytics event [SW-2194]
            DetailsView.triggerAnalyticsEventForAction('scroll', {sectionId: sectionId});
        }, 150);

        var $detailsNavItems = AJS.$('.ghx-detail-nav-menu .ghx-detail-nav-item');

        var $selectedNavItem = $detailsNavItems.filter('.ghx-selected');
        if ($selectedNavItem.find("a").attr("href") !== '#' + sectionId) {
            triggerAnalyticsEventOnScrollEnd(sectionId);
        }

        $selectedNavItem.removeClass('ghx-selected');

        var $currentSection = AJS.$('#' + sectionId);

        // If tab specifies tabIconId, select by tabIconId instead
        var tabIconId = $currentSection.attr('ghx-tab-icon-id');
        if (tabIconId) {
            sectionId = tabIconId;
        }

        $detailsNavItems.find('a').filter(function () {
            return this.hash === '#' + sectionId;
        }).closest('.ghx-detail-nav-item').addClass('ghx-selected');
    };

    // fixedElement stuff start

    /**
     * Resize the column size.
     */
    DetailsView.updateSize = function () {
        DetailsView.resizeLastSectionSpacer();
        // delegate to our selected renderer
        DetailsView.getContainer().width(DetailsView.getWidth());
        // update max width for details view when sizes change
        if (AJS.$("#ghx-detail-view").data("ui-resizable")) {
            AJS.$("#ghx-detail-view").resizable("option", "maxWidth", DetailsView.getMaxWidth());
        }
        AJS.$(GH).trigger('pool-column-updated');
    };

    /**
     * Returns the fixed element data (the fixed element size is managed by the ViewController)
     */
    DetailsView.getFixedElement = function () {
        return {
            element: DetailsView.getContainer(),
            resizeCallback: DetailsView.updateSize
        };
    };

// fixedElement stuff end

    DetailsView.reload = function (detailsViewLoadReason) {
        var model = GH.DetailsFieldEdit.model;
        // TODO: Figure out a smarter way to reload the details view
        // if we aren't in the middle of an edit
        if (!model || !model.getEditingCount()) {
            DetailsView.show(detailsViewLoadReason);
        }
    };

    /**
     * Loads the detail view content, displaying a spinner while loading
     */
    DetailsView.load = function (savedFieldId, loadReason) {
        AJS.$('#gh').addClass('js-ghx-detail-loading').removeClass('js-ghx-detail-loaded');

        if (!loadReason) {
            DetailsView.loadContent(savedFieldId, DetailsView.renderDetails);
        } else if (savedFieldId && DetailsView.hasEditsInProgress()) {
            DetailsView._Reloader.reload(loadReason, savedFieldId, DetailsView.renderDetailsSingleField);
        } else {
            DetailsView._Reloader.reload(loadReason);
        }
    };

    DetailsView.hasEditsInProgress = function () {
        return DetailsView.getEditsInProgress().length > 0
    };

    DetailsView.getEditsInProgress = function () {
        var edits = DetailsView._InlineEditor && DetailsView._InlineEditor.getEditsInProgress();
        return edits || [];
    };

    /**
     * Updates the issue details view.
     */
    DetailsView.loadContent = function (savedFieldId, onSuccess) {
        // display a blanket with a spinner in the middle of Detail View
        // Unless this is a load due to a field being saved.
        if (!savedFieldId) {
            DetailsView.showSpinner("load-content-spinner");
        }

        var requestData = {
            rapidViewId: DetailsView.rapidViewId,
            issueIdOrKey: DetailsView.selectedIssueKey,
            loadSubtasks: DetailsView.opts.showSubtaskTab
        }

        if (DetailsView._InlineEditor.getIssueAttachmentQuery()) {
            requestData = _.extend(requestData, DetailsView._InlineEditor.getIssueAttachmentQuery());
        }

        // issue an ajax request
        DetailsView.currentPromise = GH.Ajax.get({
            url: '/xboard/issue/details.json',
            data: requestData
        }, 'detailView');

        DetailsView.currentPromise.done(function (data) {
            // ignore content if it doesn't match expected key
            if (data.key != DetailsView.selectedIssueKey) {
                // ignore
                return;
            }

            // setup the model
            GH.DetailsObjectFactory.getDetailsModel().setViewData(data);

            if (!_.isUndefined(onSuccess) && _.isFunction(onSuccess)) {
                onSuccess(savedFieldId);
            }

            DetailsView.currentPromise = undefined;

            DetailsView.makeResizable();

            //AJS.$('#ghx-detail-view').on('simpleClick', function () {
            //alert('ok')
            //});

        }).always(function () {
            // hide a blanket with a spinner in the middle of Detail View as long as no other code wants to take
            // ownership of the spinner.
            DetailsView.hideSpinner("load-content-spinner");
        });
    };

    /**
     * @param spinnerOwnerId similar to event namespace, by providing a value, you could say: I will hide the spinner as long as
     *      it's still mine
     */
    DetailsView.showSpinner = function (spinnerOwnerId) {
        AJS.$('.ghx-detail-view-blanket').show().spin("large").attr('spinnerOwnerId', !spinnerOwnerId ? '' : spinnerOwnerId);
    };

    /**
     * @param spinnerOwnerId when provided, the spinner can only be hidden if its id has the same value. If undefined, the spinner
     *      will be hidden anyway.
     */
    DetailsView.hideSpinner = function (spinnerOwnerId) {
        var $blanket = AJS.$('.ghx-detail-view-blanket');
        if (!spinnerOwnerId || $blanket.attr('spinnerOwnerId') === spinnerOwnerId) {
            $blanket.hide().spin(false).attr('spinnerOwnerId', '');
        }
    };

    function getElementWidthOptimised($el) {
        // We are assuming that hidden element has 0 width. This is optimized to avoid layout trashing when rendering
        // big boards because normally jQuery will show and hide an element to measure its dimensions.
        if ($el.css('display') === 'none') {
            return 0;
        } else {
            return $el.width();
        }
    };

    DetailsView.getContentElementWidth = function () {
        const backlog = AJS.$('#ghx-backlog');
        const board = backlog.length > 0 ? backlog : AJS.$('#ghx-pool-column');
        return getElementWidthOptimised(board);
    };

    DetailsView.getDetailViewElementWidth = function () {
        return getElementWidthOptimised(AJS.$('#ghx-detail-view'));
    };

    DetailsView.getMaxWidth = function () {
        return Math.max(DetailsView.minWidth, (DetailsView.getDetailViewElementWidth() + DetailsView.getContentElementWidth()) / 2);
    };

    DetailsView.makeResizable = function () {
        const tooltip = AJS.I18n.getText('gh.rapid.detail.resize.tip');

        if (AJS.$('#ghx-detail-view').is('.ui-resizable')) {
            AJS.$('#ghx-detail-view').resizable('destroy');
            AJS.$('.ui-resizable-outer').append('<div id="js-sizer" class="ui-sidebar ui-resizable-handle ui-resizable-w" data-tooltip="' + tooltip + '"></div>');
        }
        AJS.$("#ghx-detail-view").resizable({
            handles: {
                w: AJS.$("#js-sizer")
            },
            stop: DetailsView.storeWidth,
            maxWidth: DetailsView.getMaxWidth(),
            minWidth: DetailsView.minWidth
        });

        GH.Tooltip.tipsify({
            selector: '#js-sizer',
            gravity: 'e'
        });
    };

    DetailsView.storeWidth = function () {
        var detailViewWidth = DetailsView.getDetailViewElementWidth();
        var contentElementWidth = DetailsView.getContentElementWidth();

        var detailViewPercentage = detailViewWidth <= DetailsView.minWidth ? 0 : (detailViewWidth / (detailViewWidth + contentElementWidth));
        GH.RapidBoard.State.setDetailViewWidth(detailViewPercentage);

        var isSmallWidth = function (detailViewWidth) {
            return detailViewWidth >= DetailsView.minWidth && detailViewWidth <= 500;
        };
        var isMediumWidth = function (detailViewWidth) {
            return detailViewWidth > 500 && detailViewWidth <= 600;
        };
        var isLargeWidth = function (detailViewWidth) {
            return detailViewWidth > 600;
        };
        // Trigger analytics event [SW-2194]
        let data = {
            targetWidth: isSmallWidth(detailViewWidth) ? 'small' : isMediumWidth(detailViewWidth) ? 'medium' : isLargeWidth(detailViewWidth) ? 'large' : '',
            mode: DetailsView.getMode()
        };
        DetailsView.triggerAnalyticsEventForAction('resize', data);
        AJS.$(GH).trigger('pool-column-updated');
    };

    DetailsView.renderDetailsSingleField = function (savedFieldId) {
        var viewData = GH.DetailsObjectFactory.getDetailsModel().viewData;

        // render the contents
        var detailContents = AJS.$(GH.tpl.inlineEditableDetailview.renderIssueDetails({
            issueDetails: viewData,
            canClose: DetailsView.opts.canClose,
            showActionsCog: DetailsView.opts.showActionsCog && GH.UserData.hasUser() // only show cog to logged in users
        }));
        // render all fields
        GH.DetailsObjectFactory.getDetailsFieldRenderer().renderFields(viewData, detailContents, DetailsView.opts);

        // replace the content
        DetailsView._setIssueElement(detailContents, savedFieldId);
    };

    /**
     * Renders the details view given the provided issue data.
     * @fires GH.DetailsView#after:render
     */
    DetailsView.renderDetails = function () {
        AJS.$(GH).trigger(DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED);

        var viewData = GH.DetailsObjectFactory.getDetailsModel().viewData;

        // render the contents
        var detailContents = AJS.$(GH.tpl.inlineEditableDetailview.renderIssueDetails({
            issueDetails: viewData,
            canClose: DetailsView.opts.canClose,
            showActionsCog: DetailsView.opts.showActionsCog && GH.UserData.hasUser() // only show cog to logged in users
        }));

        // render all fields
        GH.DetailsObjectFactory.getDetailsFieldRenderer().renderFields(viewData, detailContents, DetailsView.opts);

        // register events
        DetailsView.registerAllListeners(detailContents);

        // replace the content
        DetailsView._setIssueElement(detailContents);

        // image gallery support, only setup if we got image attachments
        if (viewData.attachments && !_.isEmpty(viewData.attachments.imageAttachments)) {
            GH.DetailFancyBox.initFancyBox();
        }

        // updating the size
        DetailsView.updateSize();

        // hide the loading
        AJS.$('#gh').removeClass('js-ghx-detail-loading').addClass('js-ghx-detail-loaded');

        GH.DetailsViewScrollTracker.init({
            onSelectedNavChanged: DetailsView.selectNavItem
        });

        GH.DetailsViewScrollTracker.afterDetailsViewLoaded({
            issueId: viewData.id
        });

        DetailsView.trigger('after:render');
        // tell third party tabs that the details are drawn and they
        // are free to make their content more "live"
        JIRA.trigger(DetailsView.API_EVENT_DETAIL_VIEW_UPDATED, {
            issueId: viewData.id,
            issueKey: viewData.key,
            rapidViewId: DetailsView.rapidViewId,
            mode: DetailsView.getMode()
        });

        DetailsView._postRenderDetails(detailContents);
    };

    DetailsView.registerAllListeners = function (detailContents) {
        // create drop down for issue actions cog
        if (detailContents.find("#ghx-detail-head .aui-list").length > 0) {
            var actionsDropdown = AJS.Dropdown.create({
                trigger: detailContents.find("#ghx-detail-head .ghx-actions"),
                content: detailContents.find("#ghx-detail-head .aui-list"),
                alignment: AJS.RIGHT
            });

            function removeActionsDropdown() {
                AJS.$(actionsDropdown).each(function (index, dropdown) {
                    dropdown.hide();
                    //remove the dropdown from the page
                    if (dropdown.layerController.$content) {
                        dropdown.layerController.$content.remove();
                    }
                });
                AJS.$(GH).unbind(DetailsView.EVENT_DETAIL_VIEW_CLOSED, removeActionsDropdown);
                AJS.$(GH).unbind(DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED, removeActionsDropdown);
            }

            // Remove actions menu when details view is closed
            AJS.$(GH).bind(DetailsView.EVENT_DETAIL_VIEW_CLOSED, removeActionsDropdown);
            // Remove actions menu when we refresh the contents of the panel
            AJS.$(GH).bind(DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED, removeActionsDropdown);
        }

        // bind the "More Actions" item to show the Operations Dialog
        detailContents.find('#ghx-more-actions').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.showOperationsDialog();
        });

        // bind click for analytics
        detailContents.find('.js-issueaction-container-cog .js-issueaction').click(function (e) {
            var actionId = AJS.$(e.currentTarget).attr('id');
            DetailsView.analytics.trigger(actionId); // SAFE
        });

        // bind create attachment actions
        detailContents.find('.js-create-attachment').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.attachFilesSelectedIssue();
        });

        DetailsView.registerListenersForSubtask(detailContents);

        // bind link issue
        detailContents.find('.js-link-issue').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.linkSelectedIssue();
        });

        detailContents.find('.js-remove-issue-from-sprint').click(function (e) {
            e.preventDefault();
            // Do a thing
            var issueData = GH.DetailsObjectFactory.getDetailsModel().viewData;
            // TODO: fixes edge case where user navigates to a visible but deleted issue and then invokes the action.
            //       at that point the detail view has forgotten its issueData
            if (!issueData) {
                return;
            }
            var issueKey = issueData.key;

            GH.IssueActions.removeIssuesFromSprint(DetailsView.rapidViewId, issueKey);
        });

        detailContents.find('.js-flag-issue').click(function (e) {
            e.preventDefault();
            DetailsView.flagCurrentIssue(true, false);
        });

        detailContents.find('.js-flag-issue-and-comment').click(function (e) {
            e.preventDefault();
            DetailsView.flagCurrentIssue(true, true);
        });


        detailContents.find('.js-unflag-issue').click(function (e) {
            e.preventDefault();
            DetailsView.flagCurrentIssue(false, false);
        });

        detailContents.find('.js-unflag-issue-and-comment').click(function (e) {
            e.preventDefault();
            DetailsView.flagCurrentIssue(false, true);
        });

        DetailsView.registerHeaderFieldsListeners(detailContents);

        // Trigger analytics event [SW-2193]
        detailContents.find("#ghx-detail-head .ghx-actions").click(function () {
            AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.open.more.actions'});
        });

        DetailsView._postRegisterAllListeners(detailContents);
    };

    DetailsView.registerListenersForSubtask = function ($detailsViewContents) {
        $detailsViewContents.find('.js-create-subtask').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.createSubtask();
        });
        $detailsViewContents.find('.js-edit-subtask').click(function (e) {
            e.preventDefault();
            if (!DetailsView.setSubtaskIssueOverwrite(AJS.$(this))) {
                return;
            }
            GH.IssueOperationShortcuts.editSelectedIssue();
        });
        $detailsViewContents.find('.js-delete-subtask').click(function (e) {
            e.preventDefault();
            if (!DetailsView.setSubtaskIssueOverwrite(AJS.$(this))) {
                return;
            }
            GH.IssueOperationShortcuts.deleteSelectedIssue();
        });
    }

    /**
     * Register listeners for fields which appear in the Header. Since the header might be reloaded independently, we
     * need to be able to re-register listeners just for this section.
     */
    DetailsView.registerHeaderFieldsListeners = function (detailContents) {
        // bind log work action
        detailContents.find('.js-log-work').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.logWorkOnSelectedIssue();
        });
    };

    DetailsView.setSubtaskIssueOverwrite = function (actionElem) {
        var subtaskIssue = actionElem.parents('.js-subtask-issue');
        var issueKey = subtaskIssue.attr('data-issue-key');
        var issueId = parseInt(subtaskIssue.attr('data-issue-id'), 10);
        if (issueKey && issueId) {
            // set the overwrite
            GH.IssueOperationShortcuts.setOverrideSelectedIssue({
                id: issueId,
                key: issueKey
            });
            return true;
        } else {
            return false;
        }
    };

    DetailsView.handleTabSelection = function (event, data) {
        // return if not correct tab group
        if (data.pane.parents('#ghx-detail-nav').length === 0) {
            return;
        }

        // clicked tab ID
        var tabId = data.pane.attr('id');
        GH.DetailsViewScrollTracker.setSelectedTab(tabId);
    };

// set the Detail View tab
    DetailsView.setTab = function (tabId) {
        var tab = AJS.$('a[href=#' + tabId + ']');
        if (tab.length > 0) {
            AJS.tabs.change(tab);
        }
    };

    /**
     * When an issue is updated, it could have come from inline edit of the primary statistic. In this case, we might
     * need to tweak the UI.
     */
    DetailsView.handleIssueUpdated = function (event, data) {
        GH.log(event.type + " from source " + data.source + " handled", GH.Logger.Contexts.event);

        // ensure the updated issue is still the one shown currently. Ignore update otherwise
        var updatedIssueKey = GH.DetailsObjectFactory.getDetailsModel().getIssueKeyForId(data.issueId);
        if (!updatedIssueKey) {
            GH.log('Unable to find key for updated issue id ' + data.issueId + ". Ignoring update", GH.Logger.Contexts.event);
            return;
        }
        if (updatedIssueKey !== DetailsView.selectedIssueKey) {
            return;
        }
    };

    /**
     * Renders the details view given the provided issue data.
     */
    DetailsView.renderDetailsHeader = function () {
        var viewData = GH.DetailsObjectFactory.getDetailsModel().viewData;

        // get the existing contents
        var detailContents = AJS.$('#ghx-detail-issue');

        // re-render only header fields
        GH.DetailsObjectFactory.getDetailsFieldRenderer().renderDetailsHeaderFields(viewData, detailContents, DetailsView.opts);

        // register event listeners
        DetailsView.registerHeaderFieldsListeners(detailContents);

        // updating the size
        DetailsView.updateSize();

        // hide the loading
        AJS.$('#gh').removeClass('js-ghx-detail-loading').addClass('js-ghx-detail-loaded');
    };

    DetailsView.hasCurrentError = function () {
        var detailContents = AJS.$('#ghx-detail-issue');
        return (detailContents.find(".ghx-error").text().length > 0);
    };

    /**
     * Updates the flag of the currently displayed issue and trigger the "issueUpdated" event.
     *
     * @param {boolean} flag the flag value
     */
    DetailsView.flagCurrentIssue = function (flag, withComment) {
        function updateView() {
            // will trigger the issue list referesh
            AJS.$(GH).trigger('issueUpdated', {issueId: issueData.id, source: 'detailView'});
            // reload the details view to update the menu, we don't have a way to be more granular currently
            DetailsView.reload();
        }

        var issueData = GH.DetailsObjectFactory.getDetailsModel().viewData;
        if (issueData) {
            if (withComment) {
                GH.IssueFlagAndCommentAction.execute(issueData.key, flag, updateView);
            } else {
                GH.IssueActions.flag(issueData.key, flag).done(updateView);
            }
        }

        var eventName;
        if (flag) {
            eventName = withComment ? "gh.issue.ctx.menu.action.flag.with.comment" : "gh.issue.ctx.menu.action.flag";
            AJS.trigger("analytics", {name: eventName, data: {selectedIssueCount: 1}});
        } else {
            eventName = withComment ? "gh.issue.ctx.menu.action.unflag.with.comment" : "gh.issue.ctx.menu.action.unflag";
            AJS.trigger("analytics", {name: eventName, data: {selectedIssueCount: 1}});
        }
    };

    DetailsView._preShow = function () {
        DetailsView._InlineEditor = require('jira-agile/rapid/ui/detail/inlineedit/issue-detail-view-inline-editor');
        DetailsView._Reloader = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reloader');
    };

    DetailsView._postInit = function () {
    };

    DetailsView._postShow = function () {
        DetailsView._InlineEditor.init({
            $detailContainer: DetailsView.getContainer(),
            getRapidViewIdCallback: function () {
                return DetailsView.rapidViewId;
            },
            loadIssueDetailCallback: DetailsView.load,
            getInlineEditableFieldsCallback: function () {
                var detailsModel = GH.DetailsObjectFactory.getDetailsModel();
                if (detailsModel.viewData && detailsModel.viewData.editable) {
                    return GH.DetailsObjectFactory.getDetailsModel().getInlineEditableFields();
                }
                return [];
            },
            getIssueIdCallback: function () {
                return GH.DetailsObjectFactory.getDetailsModel().viewData.id;
            },
            getIssueKeyCallback: function () {
                return GH.DetailsObjectFactory.getDetailsModel().viewData.key;
            },
            getIssueSprintStatusCallback: function () {
                return GH.DetailsObjectFactory.getDetailsModel().viewData.sprint ? GH.DetailsObjectFactory.getDetailsModel().viewData.sprint.state : null;
            }
        });
    };

    DetailsView._postRegisterAllListeners = function () {
    };

    DetailsView._postHide = function () {
        if (DetailsView._InlineEditor) {
            DetailsView._InlineEditor.destroy();
        }
    };

    DetailsView._postRenderDetails = function ($detailContents) {
        DetailsView._InlineEditor.fixUpDetailView($detailContents);

        DetailsView._InlineEditor.loadIssueToEditor();
    };

    DetailsView._handleHeaderChanged = function () {
        DetailsView.load();
    };

    DetailsView._onContainerSelectorSet = function (selector) {
        if (selector) {
            AJS.$(selector).addClass('gh-editable-detail-view');
        }
    };

    DetailsView.getAttachmentViewQuery = function () {
        return DetailsView._InlineEditor.getIssueAttachmentQuery();
    };

    DetailsView.registerListenersForTab = function (tabId, $detailsViewContents) {
        DetailsView._InlineEditor.registerListenersForTab(tabId, $detailsViewContents);
        // Because we render subtasks by ourselves, not reuse from View Issue plugin.
        // Therefore, the InlineEditor should not have any knowledge about these listeners.
        if (tabId === 'sub_tasks') {
            DetailsView.registerListenersForSubtask($detailsViewContents);
        }
    };

    return DetailsView;

});

AJS.namespace('GH.DetailsView', null, require('jira-agile/details-view'));
