define('jira-agile/rapid/ui/detail/details-model', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var RapidBoard = GH.RapidBoard;
    var DetailsView = GH.DetailsView;

    var ISSUE_ID_REPLACEMENT_REGEX = /\?id=\d+/;

    var DetailsModel = {};

    /**
     * Defines the mapping of tabs -> sections
     */
    DetailsModel.TAB_SECTIONS = {
        'details': ['details', 'people', 'dates'],
        'description': ['description']
    };

    /**
     * Defines the mapping of fields -> sections
     */
    DetailsModel.FIELD_SECTION_PRESETS = {
        'assignee': 'people',
        'created': 'dates',
        'components': 'details',
        'description': 'description',
        'fixVersions': 'details',
        'issuekey': 'header',
        'labels': 'details',
        'parentissuekey': 'header',
        'reporter': 'people',
        'summary': 'header',
        'status': 'details',
        'updated': 'dates',
        'versions': 'details',
        'votes': 'people'
    };

    /**
     * Defines the mapping of fields -> custom error selectors.
     * If no mapping is present then regular error selectors are used.
     */
    DetailsModel.fieldErrorSelectors = {
        // populated during processFields()
    };

    /**
     * Defines categories of fields. These are semantic, and not used to group fields in the UI.
     */
    DetailsModel.fieldCategories = {
        'timetracking': ['timeestimate', 'timeoriginalestimate', 'aggregatetimeestimate', 'aggregatetimeestimate_subtasks']
    };

    // Dumb. we operate with issue keys, but issue update events operate with issue ids.
    // Store a lookup from id to key, so we have a way to compare issue update ids against displayed issue keys.
    // the id won't be available until the issue details data has been loaded
    DetailsModel.issueIdToKeyCache = {};

    DetailsModel.getIssueKeyForId = function (issueId) {
        return this.issueIdToKeyCache[issueId] || false;
    };

    /**
     * Holds the view data object (see IssueEntry.java)
     */
    DetailsModel.viewData = null;

    /**
     * Clear the view data out of the model
     */
    DetailsModel.clearViewData = function () {
        this.viewData = null;
    };

    /**
     * Set the view data in the model and process data
     * @param viewData
     */
    DetailsModel.setViewData = function (viewData) {
        // store id against key
        this.issueIdToKeyCache[viewData.id] = viewData.key;

        // assign returned fields to sections
        this.processFields(viewData);

        // fixes xrfs token and sets correct return url
        this.applyOperationUrlPatches(viewData);

        // escape the comment's restriction role name
        this.escapeCommentsRestriction(viewData.comments);

        // filter out some of the issue operations
        this.filterIssueOperations(viewData);

        this.viewData = viewData;
    };

    DetailsModel.setViewDataForField = function (fieldId, fieldData) {
        // update the view data for the given field
        var viewData = this.viewData;
        var i;
        for (i = 0; i < viewData.fields.length; i++) {
            if (viewData.fields[i].id === fieldId) {
                viewData.fields[i] = fieldData;
                break;
            }
        }
    };

    /**
     * Returns the field from the viewData with the specified id.
     */
    DetailsModel.getViewField = function (fieldId) {
        return _.find(this.viewData.fields, function (field) {
            return field.id === fieldId;
        });
    };

    DetailsModel.isFieldInCategory = function (theFieldId, categoryId) {
        var category = this.fieldCategories[categoryId];
        if (!_.isUndefined(category)) {
            return _.any(category, function (fieldId) {
                return fieldId === theFieldId;
            });
        }
        return false;
    };

    /**
     * @param theFieldId the field in question
     * @param sectionId the section in question
     * @return boolean true if the field is a part of the section; false otherwise
     */
    DetailsModel.isFieldInSection = function (theFieldId, sectionId) {
        var section = this.fieldSections[sectionId];
        if (!_.isUndefined(section)) {
            return _.any(section, function (fieldId) {
                return fieldId === theFieldId;
            });
        }
        return false;
    };

    /**
     * Returns the Primary Estimate Statistic field object, if one exists; false otherwise
     */
    DetailsModel.getPrimaryEstimateStatisticField = function () {
        if (this.fieldSections['estimate'].length === 0) {
            return false;
        }
        return this.getFieldsForSection('estimate')[0];
    };

    /**
     * Returns a list of the field objects which are in the specified section
     * @param section the section
     * @return array of field objects; empty array if there are none
     */
    DetailsModel.getFieldsForSection = function (section) {
        var fieldIdsForSection = this.fieldSections[section];
        if (_.isUndefined(fieldIdsForSection)) {
            return [];
        }

        return _.map(fieldIdsForSection, function (fieldId) {
            return this.getViewField(fieldId);
        }.bind(this));
    };

    /**
     * Returns the custom error selector to use when rendering errors for the field; <tt>false</tt> to indicate defaults
     * should be used
     * @param fieldId the field ID
     */
    DetailsModel.getErrorSelectorForField = function (fieldId) {
        return this.fieldErrorSelectors[fieldId] || false;
    };

    /**
     * Process the fields that were returned in the view data and assign them to their correct field sections
     * @param viewData
     */
    DetailsModel.processFields = function (viewData) {
        GH.log('unimplemented function processFields', GH.Logger.Contexts.ui);
    };

    /**
     * Replaces atl_token= by atl_token={tokenValue}.
     * The reason for the token not being there is that we generate the links through a rest call, but SimpleLinkFactory uses
     * the webwork Action context to figure out the current request in order to generate the token... Thanks but no thanks!
     */
    DetailsModel.applyOperationUrlPatches = function (viewData) {
        // fetch the return url to set
        var returnUrl = RapidBoard.getCurrentPageReturnUrl();
        // fetch the xrfs token to add
        var token = AJS.$('#atlassian-token').attr('content');

        // go through all operations, which are inside sections
        if (viewData.operations) {
            _.each(viewData.operations.sections, function (section) {
                _.each(section.operations, function (operation) {
                    // patch the URL and label
                    operation.url = this.patchIssueOperationUrl(operation.url, returnUrl, token);
                    operation.label = this.patchRankOperationName(operation.styleClass, operation.label);
                }.bind(this));
            }.bind(this));
        }
    };

    DetailsModel.patchRankOperationName = function (styleClass, label) {
        if (styleClass === 'issueaction-greenhopper-rank-top-operation') {
            return AJS.I18n.getText('gh.keyboard.shortcut.send.to.top.desc');
        } else if (styleClass === 'issueaction-greenhopper-rank-bottom-operation') {
            return AJS.I18n.getText('gh.keyboard.shortcut.send.to.bottom.desc');
        } else {
            return label;
        }
    };

    DetailsModel.patchIssueOperationUrl = function (url, returnUrl, token) {

        // if there is a fragment in the given URL cut it and append it when all the modifications are done
        var fragment = '';
        if (url.indexOf('#') !== -1) {
            fragment = url.substring(url.indexOf('#'));
            url = url.substring(0, url.indexOf('#'));
        }

        // set the returnUrl
        url += url.indexOf('?') < 0 ? '?' : '&';
        url += 'returnUrl=' + returnUrl;

        // fill in atl_token, but only if not already set
        var newToken = 'atl_token=' + token;
        if (url.indexOf(newToken) < 0) {
            url = url.replace('atl_token=', newToken);
        }

        // remove the hardcoded issue id from all workflow transitions, replace by {0}
        // we have to do this because the transition dialog otherwise "remembers" the first executed issue id
        if (url.indexOf('WorkflowUIDispatcher.jspa') > -1) {
            url = url.replace(ISSUE_ID_REPLACEMENT_REGEX, '?id={0}');
        }

        return url + fragment;
    };

    /**
     * Helper function to escape the comments restrictions roles
     */
    DetailsModel.escapeCommentsRestriction = function (comments) {
        GH.log('unimplemented function escapeCommentsRestriction', GH.Logger.Contexts.ui);
    };

    /**
     * For display in the Detail View, we don't care about certain issue operations.
     * So we will re-process the operations and re-organise them.
     * @param viewData
     */
    DetailsModel.filterIssueOperations = function (viewData) {
        // define which operations we don't want to display
        var excludedOpIds = ['view-voters', 'greenhopper-planningboard-operation', 'greenhopper-taskboard-operation', 'attach-screenshot', 'toggle-issue/detailswatch-issue', 'manage-watchers', 'move-issue', 'edit-labels', 'issue-to-subtask', 'clone-issue', 'toggle-vote-issue', 'greenhopper-rapidboard-operation'];
        // define which sections we don't want to display
        var excludedGroupIds = ['opsbar-transitions'];

        // do the filtering
        var filteredSections = [];
        _.each(viewData.operations.sections, function (section) {

            // first do a section check based on the groupId
            var isSectionExcluded = _.find(excludedGroupIds, function (groupId) {
                return groupId === section.groupId;
            });
            if (isSectionExcluded) {
                return;
            }

            // filter the operations for this section
            var f = _.reject(section.operations, function (operation) {
                // do an ID check
                var isOpExcluded = _.find(excludedOpIds, function (id) {
                    return id === operation.id;
                });
                return _.isString(isOpExcluded);
            });

            // SW-3178 view issue page prevents default on the click handler for this when we pull in its resources.
            // Remove in SW-3685
            DetailsModel.renameWatchIssueId(f);

            // if there are still operations, add this section
            if (f.length > 0) {
                filteredSections.push({
                    operations: f
                });
            }
        });

        var canSectionBeMerged = function canSectionBeMerged(section) {
            return section.operations.length === 1 && !_.contains(['archive-issue', 'delete-issue'], section.operations[0].id);
        };

        // merge consecutive sections with only one item
        for (var i = 0, previousSection, sectionToAdd; i < filteredSections.length; i++) {
            var currentSection = filteredSections[i];
            if (!_.isUndefined(previousSection)) {
                if (canSectionBeMerged(currentSection)) {
                    if (_.isUndefined(sectionToAdd) && canSectionBeMerged(previousSection)) {
                        sectionToAdd = previousSection;
                    }

                    if (!_.isUndefined(sectionToAdd)) {
                        // add the operations from the current section
                        sectionToAdd.operations = sectionToAdd.operations.concat(currentSection.operations);

                        // remove operations from currentSection
                        currentSection.operations = [];
                    }
                } else if (currentSection.operations.length > 1) {
                    // if we come across a section that has more than one operation, unset sectionToAdd
                    sectionToAdd = undefined;
                }
            }
            previousSection = currentSection;
        }

        // delete all sections with no operations
        filteredSections = _.reject(filteredSections, function (section) {
            return section.operations.length === 0;
        });

        // set the results onto the viewData
        viewData.operations.filteredSections = filteredSections;

        // following conditions need to be met to remove an issue from sprint
        // 1. scrum board
        // 2. issue editable
        // 3. in a sprint
        // 4. not a subtask
        if (DetailsView.rapidViewConfig.sprintSupportEnabled && viewData.editable && viewData.sprint && !viewData.isSubtask) {
            this.addRemoveFromSprintOperation(viewData.operations);
        }

        this.addFlagOperation(viewData.operations, !viewData.flagged);
    };

    /**
     * Renames the toggle-watch-issue operation's id to gh-toggle-watch-issue
     * @param operations
     */
    DetailsModel.renameWatchIssueId = function (operations) {
        _.find(operations, function (operation) {
            if (operation.id === 'toggle-watch-issue') {
                operation.id = 'gh-toggle-watch-issue';
                return true;
            }
        });
    };

    DetailsModel.addRemoveFromSprintOperation = function (issueOps) {
        issueOps.filteredSections.push({
            operations: [{
                id: 'removeIssueFromSprint',
                label: AJS.I18n.getText('gh.remove.issue.from.sprint.label'),
                styleClass: 'issueaction-remove-issue-from-sprint js-remove-issue-from-sprint',
                title: AJS.I18n.getText('gh.remove.issue.from.sprint.title'),
                url: ''
            }]
        });

        return issueOps;
    };

    /**
     * Append the flag / unflag operation to the operations list according to the given "flag" parameter.
     *
     * @param {{filteredSections: Array}} issueOps the operations list
     * @param {boolean} flag true for flag, flase for unflag
     */
    DetailsModel.addFlagOperation = function (issueOps, flag) {
        if (flag) {
            issueOps.filteredSections.push({
                operations: [{
                    id: 'flagIssue',
                    label: AJS.I18n.getText('gh.rapid.operations.flag'),
                    styleClass: 'issueaction-flag js-flag-issue',
                    title: AJS.I18n.getText('gh.rapid.operations.flag'),
                    url: ''
                }, {
                    id: 'flagIssueAndComment',
                    label: AJS.I18n.getText('gh.rapid.operations.flag.and.comment'),
                    styleClass: 'issueaction-flag-and-comment js-flag-issue-and-comment',
                    title: AJS.I18n.getText('gh.rapid.operations.flag.and.comment'),
                    url: ''
                }]
            });
        } else {
            issueOps.filteredSections.push({
                operations: [{
                    id: 'unflagIssue',
                    label: AJS.I18n.getText('gh.rapid.operations.unflag'),
                    styleClass: 'issueaction-unflag js-unflag-issue',
                    title: AJS.I18n.getText('gh.rapid.operations.unflag'),
                    url: ''
                }, {
                    id: 'unflagIssueAndComment',
                    label: AJS.I18n.getText('gh.rapid.operations.unflag.and.comment'),
                    styleClass: 'issueaction-unflag-and-comment js-unflag-issue-and-comment',
                    title: AJS.I18n.getText('gh.rapid.operations.unflag.and.comment'),
                    url: ''
                }]
            });
        }
    };

    /**
     * Find out whether the current issue has the specified operation available to it. Checks the operations that were returned
     * from the server as part of the model. Useful for checking if an operation is allowed (e.g. due to permission checks).
     */
    DetailsModel.hasIssueOperation = function (operationId) {
        return _.find(this.viewData.operations.sections, function (section) {
            return _.find(section.operations, function (operation) {
                return operation.id === operationId;
            });
        });
    };

    return DetailsModel;
});