/**
 * Details view field edit manager. Manages edits, keeping all necessary information in a model object.
 */
GH.DetailsFieldEdit = {};

GH.DetailsFieldEdit.model = false;

/**
 * initializes the detail view stuff
 */
GH.DetailsFieldEdit.init = function () {

    // editing of fields
    AJS.$(document).delegate('.js-issue-fields .js-editable-field', 'click', GH.DetailsFieldEdit.handleDetailsFieldClick);
};

/**
 * Tries completing all ongoing edits
 */
GH.DetailsFieldEdit.completeEdits = function () {
    var model = GH.DetailsFieldEdit.model;
    GH.DetailsFieldEdit.model = false;
    if (model) {
        model.completeEdits();
    }
};

/**
 * Discards all edits and throws away the edit model
 */
GH.DetailsFieldEdit.discardEdits = function () {
    var model = GH.DetailsFieldEdit.model;
    GH.DetailsFieldEdit.model = false;
    if (model) {
        model.discardEdits();
    }
};

GH.DetailsFieldEdit.reset = function () {
    // TODO
    //GH.DetailsFieldEditModel.editIsLoading = false;
    GH.DetailsFieldEdit.discardEdits();
};

/**
 * Get the number of currently edited fields
 */
GH.DetailsFieldEdit.getEditingCount = function () {
    return GH.DetailsFieldEdit.model && GH.DetailsFieldEdit.model.getEditingCount() || 0;
};

/**
 * Get the number of currently edited fields
 */
GH.DetailsFieldEdit.getEditingFieldIds = function () {
    return GH.DetailsFieldEdit.model && GH.DetailsFieldEdit.model.getEditingFieldIds() || [];
};

/**
 * Is a given field currently being edited?
 */
GH.DetailsFieldEdit.isEditing = function (fieldId) {
    return GH.DetailsFieldEdit.model && GH.DetailsFieldEdit.model.isEditing(fieldId) || false;
};

//
// Editing kickoff stuff (field id or event based)
//

/**
 * Handles the click on a field.
 */
GH.DetailsFieldEdit.handleDetailsFieldClick = function (event) {

    // check whether we are an editable field, do nothing otherwise
    var clickedElem = AJS.$(this);
    var fieldElem = clickedElem.closest('dd.js-editable-field');
    if (fieldElem.length < 1) {
        return;
    }

    // check whether we are a link - we want the link behavior rather than start editing
    if (AJS.$(event.target).is('a')) {
        return;
    }

    // fetch required values
    var fieldId = fieldElem.attr('data-field-id');
    var issueId = parseInt(AJS.$('#ghx-detail-issue').attr('data-issueid'), 10);

    // kick off editing.
    GH.DetailsFieldEdit.editField(issueId, fieldId);

    // make sure the default for the event does not happen, otherwise we loose focus right as the click is fired not against the replaced edit
    event.preventDefault();
};

/**
 * Passes the edit to the sibling field, specified by delta.
 */
GH.DetailsFieldEdit.editRelativeField = function (editInfo, delta) {
    var model = editInfo.model;
    var focusedFieldId = editInfo.fieldData.id;

    // TODO: make sure the model is still the same as stored here
    if (!GH.DetailsFieldEdit.model || GH.DetailsFieldEdit.model.getIssueId() != model.getIssueId()) {
        return;
    }

    var issueId = model.getIssueId();
    var orderedFields = model.getFieldInTabOrder();
    var fieldIndex = null;
    var i;
    for (i = 0; i < orderedFields.length; i++) {
        if (orderedFields[i].id === focusedFieldId) {
            fieldIndex = i;
            break;
        }
    }
    // return if we can't find the current field
    if (_.isNull(fieldIndex)) {
        return false;
    }

    // select next field
    if (delta > 0) {
        // search forward
        for (i = fieldIndex + 1; i < orderedFields.length; i++) {
            if (orderedFields[i] && orderedFields[i].editable && GH.DetailsFieldEdit.isFieldVisible(orderedFields[i])) {
                GH.DetailsFieldEdit.editField(issueId, orderedFields[i].id);
                return true;
            }
        }
    } else {
        // search backwards
        for (i = fieldIndex - 1; i >= 0; i--) {
            if (orderedFields[i] && orderedFields[i].editable && GH.DetailsFieldEdit.isFieldVisible(orderedFields[i])) {
                GH.DetailsFieldEdit.editField(issueId, orderedFields[i].id);
                return true;
            }
        }
    }
    return false;
};

/**
 * Is the field currently visible
 */
GH.DetailsFieldEdit.isFieldVisible = function (fieldData) {
    return AJS.$('dd[data-field-id="' + fieldData.id + '"]').is(':visible');
};

/**
 * Initiates field editing for a given field. Either forwards the work to the DetailFields or triggers a data load
 * if not yet loaded
 */
GH.DetailsFieldEdit.editField = function (issueId, fieldId) {
    // make sure that the issue to be edited is still the one that is/should be displayed
    // TODO: we really should compare here against the issueKey in the detail view, as the key
    // is available way before the id (for which the viewData needs to be loaded first
    if (!GH.DetailsObjectFactory.getDetailsModel().viewData || GH.DetailsObjectFactory.getDetailsModel().viewData.id != issueId) {
        return;
    }

    // check whether we are already editing, just focus in this case
    var model = GH.DetailsFieldEdit.model;

    // discard model if for a different issue
    if (model.issueId != issueId) {
        model = GH.DetailsFieldEdit.model = false;
    }

    // are we already editing the field, if so just focus
    if (model && model.isEditing(fieldId)) {
        var editInfo = model.getEditInfo(fieldId);
        if (editInfo) {
            GH.DetailsFieldEdit.focusFieldEditor(editInfo);
            return;
        }
    }

    // if we got other fields edited we can just start a second edit without reloading the edit data
    if (model && model.getEditingCount() > 0) {
        GH.DetailsFieldEdit.startFieldEdit(model, fieldId);
        return;
    }

    // create a new model and start editing
    if (!model || !model.isLoading) {
        GH.DetailsFieldEdit.model = model = new GH.DetailsFieldEditModel(issueId);
    }

    // load the data
    GH.DetailsFieldEdit.loadEditData(model, fieldId);
};

/**
 * Loads the edit data and kicks off editing once loaded.
 */
GH.DetailsFieldEdit.loadEditData = function (model, fieldToEditUponCompletion) {
    // set which field we want to edit
    model.fieldToEditUponCompletion = fieldToEditUponCompletion;

    // Double click multiple loads prevention
    if (model.isLoading) {
        return;
    }
    model.isLoading = true;

    GH.Ajax.get({
        url: '/xboard/issue/edit-data.json',
        data: {
            'issueIdOrKey': model.getIssueId(),
            'rapidViewId': GH.RapidViewConfig.currentData.id
        }
    }, 'loadEditData').done(function (data) {
        model.isLoading = false;
        model.setEditData(data);
        var fieldToEdit = model.fieldToEditUponCompletion;
        delete model.fieldToEditUponCompletion;
        GH.DetailsFieldEdit.startFieldEdit(model, fieldToEdit);
    });
};

//
// Field update management
//

/**
 * Called on a successful field update
 */
GH.DetailsFieldEdit.handleFieldUpdateSuccess = function (editInfo, data) {
    //var updatedFieldId = fieldData.view.id;
    var updatedFieldId = editInfo.fieldData.id;
    var issueId = editInfo.model.getIssueId();
    var updatedFieldData = data.view;

    // make sure we are still on the same detail view
    if (GH.DetailsObjectFactory.getDetailsModel().viewData && GH.DetailsObjectFactory.getDetailsModel().viewData.id == issueId) {
        // update the view data
        GH.DetailsObjectFactory.getDetailsModel().setViewDataForField(updatedFieldId, updatedFieldData);

        // update the ui
        GH.DetailsFieldEdit.updateFieldViewAndCloseEdit(editInfo, updatedFieldData);

        // update the edit data
        editInfo.model.updateEditDataForField(updatedFieldId, data.edit);
    } else {
        // show success message
        var message = GH.tpl.detailview.fieldUpdateSuccess();
        GH.Notification.showSuccess(message);
    }

    // then fire an event to update the rest of the ui
    GH.log("issueUpdated triggered", GH.Logger.Contexts.ui);
    AJS.$(GH).trigger('issueUpdated', { issueId: issueId, source: 'detailView', fieldId: updatedFieldId });
};

/**
 * Called on a failed field update
 */
GH.DetailsFieldEdit.handleFieldUpdateFailure = function (editInfo, fieldId, ctxError) {
    // make sure we are still on the same view
    var model = editInfo.model;
    if (GH.DetailsObjectFactory.getDetailsModel().viewData && GH.DetailsObjectFactory.getDetailsModel().viewData.id == model.getIssueId() && model.isEditing(fieldId)) {
        // show contextual errors
        if (ctxError.errors) {
            GH.DetailsFieldEdit.showFieldErrors(editInfo, ctxError.contextId, ctxError.errors);
        }
    } else {
        // display warning message
        var message = GH.tpl.detailview.fieldUpdateFailure();
        GH.Notification.showWarning(message);
    }
};

//
// Actual editing dom interaction
//


/**
 * Opens a field editor
 */
GH.DetailsFieldEdit.startFieldEdit = function (model, fieldId) {
    // fetch the edit data for that specific field
    var fieldEditData = model.getFieldEditData(fieldId);
    if (!fieldEditData) {
        return;
    }

    // put together a data carrier object, which holds the info throughout the edit cycle
    var editInfo = {};
    editInfo.model = model;
    editInfo.fieldData = fieldEditData;
    editInfo.enabled = true;

    // find the correct view dd for the edit
    var viewDd = AJS.$('dd.ghx-fieldname-' + fieldId);
    if (viewDd.length < 1) {
        return;
    }

    // add view and edit dd
    editInfo.viewDd = viewDd;
    editInfo.editDd = GH.DetailsFieldEdit.createEditDD(fieldEditData);

    // find correct editor for the field
    switch (fieldEditData.type) {
        case 'textarea':
            // TextArea control can be set to either submit on Return or not; we don't want this for Description field
            var doesReturnSubmit = fieldId != 'description';
            editInfo.editor = new GH.DetailsFieldTextArea(doesReturnSubmit);
            break;

        case 'number':
        case 'text':
            editInfo.editor = new GH.DetailsFieldTextInput();
            break;
    }

    // start editing if we got a valid editInfo object.
    if (editInfo.editor) {
        // register info with the model
        model.addEditInfo(editInfo);

        // kick off editor
        editInfo.editor.startEdit(editInfo);
    }
};

/**
 * Creates the edit dd dom
 */
GH.DetailsFieldEdit.createEditDD = function (fieldData) {
    var elem = AJS.$('<dd></dd>');
    elem.addClass('ghx-fieldtype-' + fieldData.type).addClass('ghx-fieldname-' + fieldData.id).addClass('js-editing ghx-editing').attr({ 'data-field-id': fieldData.id });
    return elem;
};

/** Switches the view for the edit. */
GH.DetailsFieldEdit.showEdit = function (editInfo) {
    // hide view and insert edit after view
    editInfo.viewDd.hide().after(editInfo.editDd);
};

/** Focuses the edited field */
GH.DetailsFieldEdit.focusFieldEditor = function (editInfo) {
    editInfo.editor.focus(editInfo);
};

/**
 * Removes the edit and shows the previous view again.
 */
GH.DetailsFieldEdit.hideEdit = function (editInfo) {
    // remove edit, then show view
    editInfo.editDd.remove();
    editInfo.viewDd.show();

    // hide errors if necessary
    if (editInfo.fieldData.errorSelector) {
        AJS.$(editInfo.fieldData.errorSelector).empty();
    }

    // remove the edit info from the model as it is no longer valid
    editInfo.model.removeEditInfo(editInfo);
};

/**
 * Displays an error for a given field.
 */
GH.DetailsFieldEdit.showFieldErrors = function (editInfo, fieldId, errors) {
    // TODO: ensure the issue is still the same
    if (GH.DetailsFieldEdit.model && GH.DetailsFieldEdit.model.getIssueId() == editInfo.model.getIssueId()) {
        // show the errors
        editInfo.editor.setErrors(editInfo, errors);
        // re-enable the field
        editInfo.editor.returnToEdit(editInfo);
    } else {
        // unknown field or field removed already
        AJS.log('Unable to find field for id ' + fieldId);
        GH.Notification.showErrors({ errors: errors });
    }
};

/**
 * Update the view for an updated field
 */
GH.DetailsFieldEdit.updateFieldViewAndCloseEdit = function (editInfo, updatedFieldData) {
    // TODO: ensure model is still up to date

    // we only update the content of the field
    var fieldContent = GH.DetailsObjectFactory.getDetailsFieldRenderer().renderFieldContent(updatedFieldData);
    editInfo.viewDd.html(fieldContent);

    // finally hide the edit and show the view
    GH.DetailsFieldEdit.hideEdit(editInfo);

    // BFP: in case of summary edits the header height might change, and as a result the details container height needs
    // to be adapted. Unfortunately this might already have scrolled the container, so we have to scroll back to
    // the top.
    if (editInfo.fieldData.id == 'summary') {
        AJS.$('#ghx-detail-issue').scrollTop(0);
        GH.DetailsView.updateSize();
    }
};