/**
 * A specialized inline dialog that displays the linked confluence pages of an entity
 *
 * The options object has the following structure
 *
 * options = {
 *    entityType: the type of entity. Types are defined in GH.LinkedPagesController.
 *    entityId: the id of the entity for which to display the linked pages
 *    trigger: the trigger that opens up this dialog. Must be a jquery object
 * }
 * @param options
 * @constructor
 */
GH.LinkedPagesInlineDialog = function(options){
    this.entityType = options.entityType;
    if (this.entityType == 'sprint') {
        this.entityTypeParam = 'sprintId';
    } else if (this.entityType == 'epic') {
        this.entityTypeParam = 'issueKey';
    } else {
        throw new Error("Did not understand entity type: " + this.entityType);
    }
    this.entityId = options.entityId;
    this.agileMode = options.agileMode;
    this.$trigger = options.trigger;
    this.$content = null;
    this.dialog = null;

    // create a options object for underlying dialog based on the default options
    this.dialogOptions = _.extend({}, GH.LinkedPagesInlineDialog.DIALOG_OPTIONS);
};

/**
 * @const
 * @type {number}
 */
GH.LinkedPagesInlineDialog.INITIAL_SHOWN_PAGES_COUNT = 3;

/**
 * Base options for the underlying AUI inline dialog
 *
 * @const
 * @type {{fadeTime: number, noBind: boolean, width: number, cacheContent: boolean, hideDelay: number}}
 */
GH.LinkedPagesInlineDialog.DIALOG_OPTIONS = {
    fadeTime: 0,
    noBind: true,
    width: 300,
    cacheContent: false, // don't cache the dialog content
    hideDelay: 60000 // set longer timeout (default is 10 seconds)
};

GH.LinkedPagesInlineDialog.prototype.isVisible = function() {
    return AJS.$('#inline-dialog-viewPagesDialog').is(':visible');
};

/**
 * Shows the inline dialog.
 */
GH.LinkedPagesInlineDialog.prototype.show = function(){
    this.showDeferred = AJS.$.Deferred();

    this.dialog = GH.SideInlineDialog.create("viewPagesDialog", this.$trigger, _.bind(this._render, this), this.dialogOptions);
    this.dialog.show();

    return this.showDeferred.promise();
};

/**
 * Hides the inline dialog.
 */
GH.LinkedPagesInlineDialog.prototype.hide = function(){
    this.dialog.hide();
};

/**
 * Repositions the inline dialog.
 */
GH.LinkedPagesInlineDialog.prototype.refresh = function(){
    this.dialog.refresh();
};

/**
 * Renders and initializes the content of the inline dialog
 * @private
 */
GH.LinkedPagesInlineDialog.prototype._render = function($content, $trigger, showPopup){
    this.$content = $content;

    this.showDeferred.notify("EmptyBegin");
    $content.html(GH.tpl.component.linkedpages.renderEmptyDialog({
        entityId: this.entityId
    }));
    this.showDeferred.notify("EmptyRender");

    // prevent any click inside dialog from closing it
    // Note: AUI's InlineDialog actually checks if the click originates from inside the dialog, and won't close the dialog if it does.
    // The problem comes from JIRA's InlineLayer code conflicting. There's no easy solution other than to stopPropagation.
    $content.click(function(event){
        event.stopPropagation();
    });

    // mark this for GH styling
    $content.closest('.aui-inline-dialog').addClass('ghx-inline-dialog');

    // show a spinner while we fetch the linked pages
    $content.find('.ghx-progress-indicator.ghx-initial').spin("medium");
    if (_.isFunction(showPopup)) {
        showPopup();
    }
    this.showDeferred.notify("EmptyDone");

    this.showDeferred.notify("AjaxCall");
    var deferred = GH.LinkedPagesController.getLinkedPagesForEntity(this.entityType, this.entityId);
    deferred.done(_.bind(function (data) {
        this.showDeferred.notify("AjaxDone");
        var primaryLink = _.findWhere(data.applicationLinks, {
            primary: true
        });
        var createPageAction = primaryLink ? primaryLink.url + GH.LinkedPagesController.CONFLUENCE_CREATE_PAGE_ACTION : false;

        $content.find('.ghx-progress-indicator.ghx-initial').spin("medium");

        var pages = data.pages;
        this.canModify = data.canModifyPages;
        this.applicationLinks = data.applicationLinks;

        // If there's only one server and all the pages are unavailable, then assume it's unavailable
        // If there are multiple applinked servers, we'll assume at least one is available
        var serverUnavailable = pages.length > 0 && data.applicationLinks.length <= 1 && _.all(pages, function (page) {
            return page.serverUnavailable;
        });

        var unAuthenticatedPages = _.where(pages, {authenticationRequired: true});
        var unAuthenticatedPagesGroups = _.groupBy(unAuthenticatedPages, "applicationLinkId");
        var needAuthenticationApplicationIds = _.keys(unAuthenticatedPagesGroups);
        var needAuthenticationApplications = _.map(needAuthenticationApplicationIds, function (id) {
            var authenticateUrl = unAuthenticatedPagesGroups[id][0].authenticationUrl;
            var applicationLink = _.findWhere(data.applicationLinks, {id: id});
            return {
                applicationLink: applicationLink,
                authenticateUrl: authenticateUrl
            };
        });

        // If page count is just 1 more than the initially show page count, just show all pages
        var pagesCount = pages.length;
        var initiallyShownCount = GH.LinkedPagesInlineDialog.INITIAL_SHOWN_PAGES_COUNT;
        if(initiallyShownCount + 1 == pagesCount){
            initiallyShownCount++;
        }

        this.showDeferred.notify("ContentRender");

        $content.html(GH.tpl.component.linkedpages.renderDialog({
            entityId: this.entityId,
            entityType: this.entityType,
            linkedPages: pages,
            initiallyShownCount: initiallyShownCount,
            canEdit: this.canModify,
            appLink: primaryLink,
            serverUnavailable: serverUnavailable,
            needAuthenticationApplications: needAuthenticationApplications
        }));

        this._updateViewLinkedPagesText(pages.length);

        // setup click handler for authentication
        $content.find("a.applink-authenticate").click(function (e) {
            window.open(this.href, "com_atlassian_applinks_authentication");
            e.preventDefault();
        });

        // mark this for GH styling
        $content.closest('.aui-inline-dialog').addClass('ghx-inline-dialog');

        this._initCreateAndLink({
            createPageAction: createPageAction,
            entityId: this.entityId,
            entityType: this.entityType,
            entityTypeParam: this.entityTypeParam,
            agileMode: this.agileMode,
            jiraApplicationId: data.jiraApplicationId,
            jiraUrl: data.jiraBaseUrl
        });

        // setup click handlers for remove page link
        $content.find(".js-wiki-remove").click(_.bind(this._handleRemovePageClick, this));

        // setup click handler for show more link
        $content.find(".js-wiki-view-more-pages-link").click(_.bind(this._handleShowMoreClick, this));

        // setup key handler for search input
        var pagesSearch = $content.find(".js-page-search");
        pagesSearch.click(function (event) {
            event.preventDefault();
            return false;
        });
        pagesSearch.keydown(_.bind(this._handleSelectNextItem, this));
        pagesSearch.keydown(_.bind(this._handleSelectPreviousItem, this));
        pagesSearch.keydown(_.bind(this._handleAddSelectedItem, this));
        pagesSearch.keyup(_.bind(this._handlePageSearchQueryChanged, this));

        // setup click handler for close search panel icon
        $content.find(".js-search-panel-hide").on('click', _.bind(this._handleCloseSearchPanelClick, this));

        // setup click handler for cancel/clear search icon
        $content.find(".js-search-page-icon").click(_.bind(this._handleClearSearchIconClick, this));

        this._showLinkedPagesPanel();

        if (_.isFunction(showPopup)) {
            showPopup();
        }
        this.showDeferred.notify("ContentDone");

        this.dialog.refresh();

        this.showDeferred.resolve();
    }, this));

    return false;
};

GH.LinkedPagesInlineDialog.prototype._renderPageSearchResults = function (searchResult) {
    this._hideSearchingSpinner();
    var $searchResults = this.$content.find(".js-entity-view-pages-search-result");
    if(searchResult.authenticationRequired) {
        var appLink = _.findWhere(this.applicationLinks, {id: searchResult.applicationLinkId}) || {};
        $searchResults.html(GH.tpl.component.linkedpages.renderSearchAuthenticationRequest({
            authenticationUrl: searchResult.authenticationUrl,
            authenticationLinkUrl: appLink.url,
            authenticationLinkName: appLink.name
        }));
        $searchResults.find("a.applink-authenticate").click(function (e) {
            window.open(this.href, "com_atlassian_applinks_authentication");
            e.preventDefault();
        });
    } else {
        $searchResults.html(GH.tpl.component.linkedpages.renderPageResults({
            pages: searchResult.pages,
            isComplete: searchResult.isComplete
        }));

        if (searchResult.pages.length) {
            // Select first result
            this._reselectItem('first', _.identity);
        }

        $searchResults.find('.ghx-doco-list-item').click(_.bind(this._handlePageSearchResultClick, this));
    }
};

GH.LinkedPagesInlineDialog.prototype._renderPageSearchFailed = function(res) {
    this._hideSearchingSpinner();

    var messages = res.getGlobalErrors();
    AJS.messages.warning(AJS.$(".js-link-page-panel"), {
        title:  AJS.I18n.getText('gh.rapid.pages.server.search.error', ''),
        body: messages && messages[0].message
    });
};

/***********************************************
 *               EVENT HANDLERS                *
 ***********************************************/

GH.LinkedPagesInlineDialog.prototype._handleCloseSearchPanelClick = GH.Util.wrapPreventDefault(function(event) {
    this._showLinkedPagesPanel();
});

GH.LinkedPagesInlineDialog.prototype._handleClearSearchIconClick = function(event){
    GH.PageSearchRequestService.cancel();
    this._emptySearchResults();
    this._hideSearchingSpinner();

    this.$content.find(".js-search-existing-pages-input").val("");

    event.preventDefault();
    return false;
};

GH.LinkedPagesInlineDialog.prototype._handleRemovePageClick = function(event) {
    var $deleteLink = AJS.$(event.target);
    var pageId = $deleteLink.attr("data-page-id");

    // hide the remove icon and show a spinner
    $deleteLink
        .hide()
        .closest('li').find('.js-wiki-spin').removeClass('ghx-hidden')
        .find('.ghx-progress-indicator.ghx-small').spin("small");

    // Call the remove link api
    var deferred = GH.LinkedPagesController.removeLinkedPage(this.entityType, this.entityId, pageId);
    deferred.done(_.bind(function(){
        // remove the html of the linked page
        $deleteLink.closest("li").remove();

        var pages = GH.PageIssueLinksModel.list(this.entityId);
        if(pages.length === 0){
            // show the no pages linked yet message of no linked pages
            this.$content.find(".js-no-linked-pages").removeClass('hidden');
        }
        this._updateViewLinkedPagesText(pages.length);
    }, this));

    this._showMoreLinkedPages();

    event.preventDefault();
    return false;
};

GH.LinkedPagesInlineDialog.prototype._handleShowMoreClick = function(event){
    event.preventDefault();
    event.stopPropagation();
    this._showMoreLinkedPages();
    return false;
};

GH.LinkedPagesInlineDialog.prototype._handlePageSearchQueryChanged = function (event) {
    if(event.keyCode == 13 || event.keyCode == 38 || event.keyCode == 40) {
        return;
    }

    var query = AJS.$(event.target).val();
    this._performSearch(query);
};

GH.LinkedPagesInlineDialog.prototype._performSearch = function(query) {
    this._emptySearchResults();
    if (query.length > 0) {
        GH.LinkedPagesController.searchPages(this.entityType, this.entityId, query)
            .done(_.bind(this._renderPageSearchResults, this))
            .fail(_.bind(this._renderPageSearchFailed, this));
        this._showSearchingSpinner();
        this._showClearSearchIcon();
    } else {
        this._hideClearSearchIcon();
    }
};

GH.LinkedPagesInlineDialog.prototype._handleSelectNextItem = function(event) {
    if(event.keyCode != 40) {
        return;
    }

    event.preventDefault();
    return this._reselectItem('first', function(index) {
        return index + 1;
    });
};

GH.LinkedPagesInlineDialog.prototype._handleSelectPreviousItem = function(event) {
    if(event.keyCode != 38) {
        return;
    }

    event.preventDefault();
    return this._reselectItem('last', function(index) {
        return index - 1;
    });
};

GH.LinkedPagesInlineDialog.prototype._reselectItem = function(emptyFn, indexFn) {
    var results = AJS.$('.ghx-doco-list-result');
    // No items in the list
    if(results.length === 0) {
        return;
    }

    var selected = results.filter('.ghx-doco-list-selected');
    var newSelection = null;
    if(selected.length === 0) {
        newSelection = results[emptyFn]();
    } else {
        var index = results.index(selected);
        selected.removeClass('ghx-doco-list-selected');

        var newIndex = indexFn(index);
        if(newIndex < 0 || newIndex > results.length - 1) {
            newSelection = results[emptyFn]();
        } else {
            newSelection = AJS.$(results[newIndex]);
        }
    }

    if(newSelection) {
        newSelection.addClass('ghx-doco-list-selected');

        var parent = newSelection.parent();
        var lowerBound = parent.position().top;
        var higherBound = lowerBound + parent.height();

        var newSelectionPosition = newSelection.position().top;
        var newSelectionHeight = newSelection.height();
        if(newSelectionPosition < lowerBound || higherBound < newSelectionPosition + newSelectionHeight) {
            newSelection.get(0).scrollIntoView();
        }
    }
};

GH.LinkedPagesInlineDialog.prototype._handleAddSelectedItem = function(event) {
    if(event.keyCode != 13) {
        return;
    }
    event.preventDefault();

    var results = AJS.$('.ghx-doco-list-result');
    var selected = results.filter('.ghx-doco-list-selected');
    if(selected.length !== 0) {
        this._addSearchResult(selected);
    }
};

GH.LinkedPagesInlineDialog.prototype._handlePageSearchResultClick = function(event) {
    event.preventDefault();

    var $pageSearchListItem = AJS.$(event.target).closest("[data-page-id]");
    this._addSearchResult($pageSearchListItem);
};

GH.LinkedPagesInlineDialog.prototype._addSearchResult = function($pageSearchListItem) {
    var pageId = $pageSearchListItem.attr('data-page-id');

    var page = GH.PagesSearchModel.get(this.entityId, pageId);
    var $linkedPagesList = this.$content.find(".js-linked-pages-list");
    var pageTitle = page.title;

    // expand the list of linked pages if the 'show more' link is shown
    this._showMoreLinkedPages();

    // show a stub link with a spinner to indicate that the link is in the process of being added
    var $stubIssueLinkListItem = AJS.$(GH.tpl.component.linkedpages.renderStubLinkedPageListItem({
        pageTitle: pageTitle
    }));
    $linkedPagesList.append($stubIssueLinkListItem);

    // Add link on server and when response received, replace stub link with real one
    var deferred = GH.LinkedPagesController.addLink(this.entityType, this.entityId, pageId, pageTitle);
    deferred.done(_.bind(function (issueLink) {
        var $issueLinkListItem = AJS.$(GH.tpl.component.linkedpages.renderLinkedPageListItem({
            page: issueLink,
            entityId: this.entityId,
            canEdit: this.canModify,
            hidden: false
        }));

        $stubIssueLinkListItem.replaceWith($issueLinkListItem);
        $issueLinkListItem.find(".js-wiki-remove").click(_.bind(this._handleRemovePageClick, this));
        $issueLinkListItem.hide().fadeIn(200);
        this._updateViewLinkedPagesText(GH.PageIssueLinksModel.list(this.entityId).length);
    }, this));

    // fade out the selected page result
    $pageSearchListItem.parent().fadeOut(200, function () {
        $pageSearchListItem.parent().remove();
    });

    // hide no linked pages message
    this.$content.find('.js-no-linked-pages').addClass('hidden');

    // show linked pages panel
    this._showLinkedPagesPanel();
};

/***********************************************
 *                UI MODIFIERS                 *
 ***********************************************/

GH.LinkedPagesInlineDialog.prototype._emptySearchResults = function(){
    this.$content.find(".js-entity-view-pages-search-result").empty();
    this._hideClearSearchIcon();

    this.$content.find(".js-link-page-panel").find(".aui-message").remove();
};

GH.LinkedPagesInlineDialog.prototype._showClearSearchIcon = function(){
    this.$content.find(".js-search-page-icon")
        .removeClass("aui-iconfont-search-small ghx-iconfont-inactive")
        .addClass("aui-iconfont-remove");
};

GH.LinkedPagesInlineDialog.prototype._hideClearSearchIcon = function(){
    this.$content.find(".js-search-page-icon")
        .removeClass("aui-iconfont-remove")
        .addClass("ghx-iconfont-inactive aui-iconfont-search-small");
};

GH.LinkedPagesInlineDialog.prototype._initCreateAndLink = function (options) {

    this.$content.find(".js-create-page-link").click(_.bind(function () {

        // we need to open the window from the call stack of the click handler to work around popup blocker
        var pageCreationWindow = window.open('', '_blank');

        GH.LinkedPagesController.startRemoteLinkCreationConversation(this)
            .done(function (token) {

                GH.LinkedPagesController.analytics.action.trigger([options.entityType, 'createPage'].join('.'));

                var requestParams = {
                    applinkId: options.jiraApplicationId,
                    agileMode: options.agileMode,
                    fallbackUrl: options.jiraUrl,
                    creationToken: token
                };

                requestParams[options.entityTypeParam] = options.entityId;

                pageCreationWindow.location = options.createPageAction + '?' + AJS.$.param(requestParams);
            });

        return false;
    }, this));

    this.$content.find(".js-link-pages-link").click(_.bind(function () {
        this._showLinkPagePanel();
        return false;
    }, this));
};

GH.LinkedPagesInlineDialog.prototype._updateViewLinkedPagesText = function(pagesCount) {
    // update the title in the inline dialog
    var title = (pagesCount > 0) ? AJS.I18n.getText('gh.entity.pages.title', pagesCount) : AJS.I18n.getText('gh.entity.pages.header');
    this.$content.find(".js-entity-pages-header").text(title);
};

GH.LinkedPagesInlineDialog.prototype._showLinkPagePanel = function(){
    this.$content.find(".js-link-page-panel").show();
    this.$content.find(".js-search-panel-hide").show();
    this.$content.find(".js-linked-pages-actions").hide();
    this.$content.find(".js-linked-pages-panel").hide();
    this.$content.find(".js-search-existing-pages-input").focus();
    this.dialog.refresh();
};

GH.LinkedPagesInlineDialog.prototype._showLinkedPagesPanel = function(){
    this.$content.find(".js-linked-pages-panel").show();
    this.$content.find(".js-search-panel-hide").hide();
    this.$content.find(".js-linked-pages-actions").show();
    this.$content.find(".js-link-page-panel").hide();

    // clear out search results
    this.$content.find(".js-entity-view-pages-search-result").empty();
    this.$content.find(".js-search-existing-pages-input").val("");
    this._hideClearSearchIcon();
    this.dialog.refresh();
};

GH.LinkedPagesInlineDialog.prototype._showMoreLinkedPages = function() {
    this.$content.find('.ghx-doco-list-item').show();
    this.$content.find('.ghx-doco-list-view-more').hide();
};

GH.LinkedPagesInlineDialog.prototype._showSearchingSpinner = function () {
    var spinner = this.$content.find(".js-entity-view-pages-searching");
    if(spinner.hasClass('hidden')) {
        spinner.removeClass('hidden').spin("medium");
    }
};

GH.LinkedPagesInlineDialog.prototype._hideSearchingSpinner = function () {
    this.$content.find(".js-entity-view-pages-searching").addClass("hidden");
};

GH.LinkedPagesInlineDialog.prototype._hideClearSearchIcon = function(){
    this.$content.find(".js-search-page-icon")
        .removeClass("aui-iconfont-remove")
        .addClass("ghx-iconfont-inactive")
        .addClass("aui-iconfont-search-small");
};

GH.LinkedPagesInlineDialog.prototype.isSearchMode = function() {
    return this.$content.find(".js-link-page-panel").is(":visible");
};

GH.LinkedPagesInlineDialog.prototype.getSearchQuery = function() {
    return this.$content.find(".js-search-existing-pages-input").val();
};

GH.LinkedPagesInlineDialog.prototype.showSearch = function(searchQuery) {
    this._showLinkPagePanel();
    this.$content.find(".js-search-existing-pages-input").val(searchQuery).focus();
    this._performSearch(searchQuery);
};
