/* globals GH, _ */

/**
 * View Actions Component
 */

var GlobalEvents = require('jira-agile/rapid/global-events');

GH.ViewActions = {};

GH.ViewActions.PLAN_MODE = 'plan';
GH.ViewActions.WORK_MODE = 'work';
GH.ViewActions.REPORT_MODE = 'report';

GH.ViewActions.currentViewId = undefined;
GH.ViewActions.currentViewName = undefined;
GH.ViewActions.currentMode = 'work';

AJS.$(function initEvents() {
    GH.Events(GH.ViewActions); // add events functions into into GH.ViewActions scope
    GH.ViewActions.on('boardToolsRendered', GH.ViewActions.afterBoardToolsLoaded); // after load event handler for standard Board dropdown
});

GH.ViewActions.processData = function (renderFunction) {
    return function (rapidViewData) {
        var canDelete = false;

        if (rapidViewData) {
            GH.ViewActions.currentViewId = rapidViewData.id;
            GH.ViewActions.currentViewName = rapidViewData.name;
            GH.ViewActions.canEdit = rapidViewData.canEdit;
            canDelete = rapidViewData.canEdit;
        }
        renderFunction();

        GH.ViewActions.setView(GH.ViewActions.currentViewId, GH.ViewActions.currentViewName, canDelete);
    };
};

GH.ViewActions.initViewActions = GH.ViewActions.processData(function () {
    AJS.$('#ghx-view-presentation').empty();
    GH.PresentationMode.render(false, '#ghx-view-presentation');
});

GH.ViewActions.refreshBoardTools = function () {
    function safeRegisterPluggableDropdown(section) {
        function getEscapedSelectorForId(id) {
            // these need to be escaped ! " # $ % & ' ( ) * + , . / : ; < = > ? @ [ \ ] ^ ` { | } ~
            // because section id may comes from module descriptor key and it may contains not allowed characters for css class or id.
            return id.replace(/(!|"|#|\$|%|&|'|\(|\)|\*|\+|,|\.|\/|:|;|<|=|>|\?|@|\[|\\|\]|\^|`|\{|\||\}|~)/g, "\\$1");
        }

        var sectionId = getEscapedSelectorForId(section.id);
        var buttonId = sectionId + '-button';
        var contentId = sectionId + '-content';

        function registerLazyDropdownRendering(dropdowns) {
            // Dropdown won't be rendered if no one clicks the button to show it.
            // It's not big improvement for static sections but it has matter for connect add-ons' web-panels.
            // They use iframes so request is done if the html appears in the DOM.
            // So lazy rendering prevents of doing requests to add-ons' servers during page loading.
            dropdowns[0].bind('showLayer', _.once(function () {
                var content = AJS.$(GH.tpl.board.x.renderSectionContent(section));
                AJS.$('#' + contentId).html(content);
                GH.ViewActions.trigger('boardToolsRendered', {
                    'sectionId': sectionId,
                    'dropdown': dropdowns
                });
            }));
        }

        try {
            var dropdown = GH.ViewActions.registerDropdown(buttonId, contentId);
            registerLazyDropdownRendering(dropdown);
        } catch (err) {
            console.error("Error rendering dropdown for section" + section.id + ":" + err.message);
        }
    }

    AJS.$('#ghx-view-pluggable').css('visibility', 'hidden');
    if (GH.ViewActions.currentViewId) {
        GH.Ajax.get({
            url: '/xboard/toolSections',
            data: {
                mode: GH.ViewActions.mode,
                rapidViewId: GH.ViewActions.currentViewId,
                selectedProjectKey: GH.RapidBoard.projectKey
            },
            deferErrorHandling: false
        }).done(function (result) {
            var toolSections = GH.tpl.board.x.renderToolSections(result);
            AJS.$('#ghx-view-pluggable').html(toolSections);
            _.each(result.sections, function (section) {
                safeRegisterPluggableDropdown(section);
            });
            AJS.$('#ghx-view-pluggable').css('visibility', 'visible');
        });
    }
};

GH.ViewActions.registerDropdown = function (buttonId, contentId) {
    var $button = AJS.$('#' + buttonId);

    // resize window after displaying pop-up contents so that the pop-up will be placed in the right location on the screen
    $button.click(_.wrap(function () {
        AJS.$(window).resize();
    }, _.defer));

    return AJS.Dropdown.create({
        trigger: $button,
        content: AJS.$('#' + contentId),
        alignment: AJS.RIGHT,
        width: "auto"
    });
};

GH.ViewActions.renderOnConfiguration = GH.ViewActions.processData(function () {
    AJS.$('#ghx-modes-tools').append(GH.tpl.viewactions.renderActions());
    GH.ViewActions.createdDropdownOnConfigurationPage();
});

GH.ViewActions.renderOnManage = GH.ViewActions.processData(function () {
    AJS.$('#ghx-modes-tools').html(GH.tpl.viewactions.renderManageViewActions());
});

/**
 * Initializes the actions with a given viewId and permission setting
 */
GH.ViewActions.setView = function (rapidViewId, rapidViewName, canDelete) {
    // sensitive web items are disabled in plugin descriptor.
    // When you are working on this section bear in mint that it may contains also web-items from external plugins.

    // button only version in Manage page
    AJS.$('#ghx-modes-tools').find('.aui-button.js-view-action-create').attr('disabled', 'disabled');

    // enable configure if we have a view id
    if (rapidViewId) {
        AJS.$('.js-view-action-configure').removeClass('disabled');
    }
    // enable delete if we are allowed to delete
    if (canDelete) {
        AJS.$('.js-view-action-delete').removeClass('disabled');
    }
    // enable new if the user is logged in
    if (GH.UserData.hasUser() && GH.UserData.userConfig.canCreateBoard && (GH.UserData.userConfig.canCreateProject || GH.UserData.userConfig.hasProjectsAccessible || GH.UserData.userConfig.hasFiltersAccessible)) {

        AJS.$('.js-view-action-create, .js-view-action-copy').removeClass('disabled');
        // button only version in Manage page
        AJS.$('#ghx-modes-tools').find('.aui-button.js-view-action-create').removeAttr('disabled');
    }
};

/**
 * Sets the tools menu mode
 */
GH.ViewActions.setMode = function (mode) {
    if (mode !== GH.ViewActions.mode) {
        GH.ViewActions.mode = mode;
        GH.ViewActions.refreshBoardTools();
    }
};

GH.ViewActions.adjustButtonsToMode = function (mode) {
    // show/hide the correct actions
    var $planActionsParent = AJS.$('.js-view-actions-plan');
    var $workActionsParent = AJS.$('.js-view-actions-work');
    var $createActionParent = AJS.$('.js-view-action-create').parent();
    var $printCardsActionParent = AJS.$('.js-view-actions-print');

    // set up, no additional actions by default
    $planActionsParent.hide();
    $workActionsParent.hide();
    $printCardsActionParent.hide();
    $createActionParent.addClass('aui-last');

    switch (mode) {
        case GH.ViewActions.PLAN_MODE:
            $createActionParent.removeClass('aui-last');
            $planActionsParent.show();
            $printCardsActionParent.show().find('a').removeClass('disabled');

            // in kanplan, hide 'Hide/Show versions panel' and 'Hide/Show epics panel' buttons
            // from the 'Board' dropdown, based on a dark feature flag and board configuration
            if (GH.RapidBoard.State.isKanbanBoard()) {
                if (!GH.PlanController.isVersionsPanelEnabled()) {
                    AJS.$('.js-view-action-toggle-releases-column').hide();
                }
                if (!GH.PlanController.isEpicsPanelEnabled()) {
                    AJS.$('.js-view-action-toggle-epics-column').hide();
                }
            }
            break;
        case GH.ViewActions.WORK_MODE:
            $createActionParent.removeClass('aui-last');
            $workActionsParent.find('.js-view-action-toggle-detail-view').removeClass('disabled');
            $workActionsParent.show();
            $printCardsActionParent.show().find('a').removeClass('disabled');
            break;

        case GH.ViewActions.REPORT_MODE:
            break;
        default:
    }
};

/**
 * Modify the tools menu's swimlanes options, based on the configuration of the board and the currently selected mode
 * @param hasSwimlanes whether there are any swimlanes
 */
GH.ViewActions.setSwimlanes = function (hasSwimlanes) {
    var $workActionsParent = AJS.$('.js-view-actions-work');
    if (hasSwimlanes) {
        $workActionsParent.find('.js-swimlane-action').removeClass('disabled');
    } else {
        $workActionsParent.find('.js-swimlane-action').addClass('disabled');
    }
};

GH.ViewActions.setVersionsPanelVisible = function (visible) {
    var $releaseToggleLink = AJS.$('.js-view-action-toggle-releases-column');
    if (visible) {
        $releaseToggleLink.text(AJS.I18n.getText('gh.version.panel.collapse'));
    } else {
        $releaseToggleLink.text(AJS.I18n.getText('gh.version.panel.expand'));
    }
};

GH.ViewActions.setEpicsPanelVisible = function (visible) {
    var $epicsToggleLink = AJS.$('.js-view-action-toggle-epics-column');
    if (visible) {
        $epicsToggleLink.text(AJS.I18n.getText('gh.epic.panel.collapse'));
    } else {
        $epicsToggleLink.text(AJS.I18n.getText('gh.epic.panel.expand'));
    }
};

GH.ViewActions.setEpicLozengesVisible = function (visible) {
    // should apply to both work mode and plan mode sections
    var $detailViewToggleLink = AJS.$('.js-view-action-toggle-epics-lozenge');
    if (visible) {
        $detailViewToggleLink.text(AJS.I18n.getText('gh.rapid.issue.epic.hide'));
    } else {
        $detailViewToggleLink.text(AJS.I18n.getText('gh.rapid.issue.epic.show'));
    }
};

GH.ViewActions.setDetailViewVisible = function (visible) {
    var $detailViewToggleLink = AJS.$('.js-view-action-toggle-detail-view');
    if (visible) {
        $detailViewToggleLink.text(AJS.I18n.getText('gh.detail.view.hide'));
    } else {
        $detailViewToggleLink.text(AJS.I18n.getText('gh.detail.view.show'));
    }
};

/**
 * Renders the dropdown menu on the board configuration page.
 *
 * It's used only to render a dropdown menu on the board configuration page.
 *
 * @see GH.Dialog.CreateView.init
 */
GH.ViewActions.createdDropdownOnConfigurationPage = function () {
    // render the dropdown choices
    AJS.$('#js-view-actions').after(GH.tpl.viewactions.renderConfigurationViewActions());
    // initially hide the list
    AJS.$('#board-tools-section-content').hide();

    // create the dropdown
    var dropdown = GH.ViewActions.registerDropdown("js-view-actions", "board-tools-section-content");

    GH.ViewActions.bindBoardToolsClickEvents(dropdown);
};

GH.ViewActions.afterBoardToolsLoaded = function (context) {
    if (context.sectionId === 'board-tools-section') {
        GH.ViewActions.bindBoardToolsClickEvents(context.dropdown);
        AJS.$(context.dropdown[0]).bind("showLayer", GH.ViewActions.enableShowDetailIfIssueSelected);
        AJS.$(context.dropdown[0]).bind("showLayer", GH.ViewActions.enableShowEpicLozenge);

        // this function is invoked after drop-down is displayed so we have to refresh issue details and epic lozenge now.
        GH.ViewActions.enableShowDetailIfIssueSelected();
        GH.ViewActions.enableShowEpicLozenge();

        GH.ViewActions.setView(GH.ViewActions.currentViewId, GH.ViewActions.currentViewName, GH.ViewActions.canEdit);
        GH.ViewActions.adjustButtonsToMode(GH.ViewActions.mode);
        GH.PlanView.updateViewActions();
    }
};

GH.ViewActions.bindBoardToolsClickEvents = function (dropdown) {
    AJS.$('#board-tools-section-content .aui-list-item a').bind('click', function (event) {
        var action = AJS.$(this);

        // ignore if action is disabled
        if (action.hasClass('disabled')) {
            event.stopPropagation();
            return false;
        }

        AJS.$(dropdown).hide();
    });

    // Note: the 'create' action is handled by GH.Dialog.CreateView
    // Note: expand/collapse all is handled by SwimlaneView
    // Note: action prevent default event propagation, as href="#" would add a hash to the url/url change
    // Note: configuration and web items from external plugins are static links
    AJS.$('#board-tools-section-content .js-view-action-delete').bind('click', function (event) {
        GH.ViewActions.actionDelete();
        event.preventDefault();
    });

    AJS.$('#board-tools-section-content .js-view-action-copy').bind('click', function (event) {
        GH.ViewActions.actionCopy();
        event.preventDefault();
    });

    AJS.$('#board-tools-section-content .js-view-action-toggle-detail-view').bind('click', function (event) {
        if (GH.WorkController.isActive()) {
            GH.WorkController.toggleDetailsView();
        } else if (GH.PlanController.isActive()) {
            GH.PlanController.toggleDetailsView();
        }
        event.preventDefault();
    });

    AJS.$('#board-tools-section-content .js-view-action-print-cards').bind('click', function (event) {
        GH.ViewActions.printCards();
        event.preventDefault();
    });
};

GH.ViewActions.actionDelete = function () {
    if (!GH.ViewActions.currentViewId) {
        return;
    }
    GH.RapidBoard.DeleteDialog.showDeleteDialog(GH.ViewActions.currentViewId, GH.ViewActions.currentViewName);
};

GH.ViewActions.actionCopy = function (rapidViewId, rapidViewName) {
    if (!rapidViewId) {
        rapidViewId = GH.ViewActions.currentViewId;
        rapidViewName = GH.ViewActions.currentViewName;
    }
    GH.RapidBoard.copyRapidViewAndGotoConfiguration(rapidViewId, rapidViewName);
};

GH.ViewActions.printCards = function () {
    if (!GH.ViewActions.currentViewId) {
        return;
    }
    GlobalEvents.trigger('action.printCardsInRapidBoard');
};

GH.ViewActions.enableShowDetailIfIssueSelected = function () {
    if (GH.WorkController.isActive()) {
        var $workActionsParent = AJS.$('.js-view-actions-work');
        GH.ViewActions.setDetailViewVisible(GH.WorkController.isDetailsViewOpened());
        if (GH.WorkSelectionController.getSelectedIssueKey()) {
            $workActionsParent.find('.js-view-action-toggle-detail-view').removeClass('disabled');
        } else {
            $workActionsParent.find('.js-view-action-toggle-detail-view').addClass('disabled');
        }
    } else if (GH.PlanController.isActive()) {
        GH.ViewActions.setDetailViewVisible(GH.PlanController.isDetailsViewOpened());
    }
};

GH.ViewActions.enableShowEpicLozenge = function () {
    GH.ViewActions.setEpicLozengesVisible(GH.RapidBoard.State.getEpicShownOnRapidBoard());
};

/**
 * Sets the epic shown value and updates menus.
 */
GH.ViewActions.toggleEpicsShowOnRapidBoard = function () {
    if (GH.RapidBoard.State.getEpicShownOnRapidBoard()) {
        GH.RapidBoard.State.setEpicShownOnRapidBoard(false);
        GH.ViewActions.setEpicLozengesVisible(false);
    } else {
        GH.RapidBoard.State.setEpicShownOnRapidBoard(true);
        GH.ViewActions.setEpicLozengesVisible(true);
    }
};