if (!GH.Components) {
    GH.Components = {};
}

/**
 * Create a new SearchFilter component
 *
 * @param searchQueryCallback the function callback to execute when performing a search
 */
GH.Components.SearchFilter = function(searchQueryCallback) {
    this.searchQueryCallback = searchQueryCallback;
    this.regexReserved = new RegExp("[.*+?|()\\[\\]{}\\\\]", "g"); // .*+?|()[]{}\
};

GH.Components.SearchFilter.prototype.getExpandedSearchFieldWidth = function() {
    var $searchInput = AJS.$('#ghx-backlog-search-input');
    var $quickFilterLabel = AJS.$('#js-quickfilters-label');
    var $quickFilterLabelStart = AJS.$('#js-plan-quickfilters').position().left;
    var searchLeft = $searchInput.position().left;
    var quickFilterLabelRight = $quickFilterLabelStart + $quickFilterLabel.width();
    var activeSearchWidth = quickFilterLabelRight - searchLeft;
    return activeSearchWidth;
};

GH.Components.SearchFilter.prototype.render = function() {
    var html = GH.tpl.component.searchFilter.renderSearchBox();
    var $elem = AJS.$(html);
    var self = this;
    var $searchInput = $elem.find('#ghx-backlog-search-input');


    // when the field is focused do stuff
    $searchInput.focus(function(){
        $elem.addClass('ghx-active');
        $searchInput.css({'width':self.getExpandedSearchFieldWidth()});
        $elem.parent().addClass('ghx-search-active');
    });

    // when the field is blurred and empty, remove active state
    $searchInput.blur(function(){
        var rawQuery = AJS.$.trim($searchInput.val());
        if (_.isEmpty(rawQuery)) {
            setTimeout(function () {
                $searchInput.val('');
                $elem.removeClass('ghx-active');
                // change the icon to the search icon
                $elem.find('.aui-iconfont-remove').removeClass('aui-iconfont-remove').addClass('aui-iconfont-search-small');
                $searchInput.css({'width':''});
                $elem.parent().removeClass('ghx-search-active');
            }, 400);
        }
    });

    // bind a keyhandler for esc
    $searchInput.bind('keydown', function(event){
        if(event.keyCode == 27) {
            // clear the text
            $searchInput.val('');
            // blur the input
            $searchInput.blur();
            // re-execute search
            self.searchQuery();
        }
    });

    // when the icon is clicked do stuff
    $elem.find('.js-search-trigger').click(function(){
        // if already active close it else open it
        if($elem.hasClass('ghx-active')){
            $elem.removeClass('ghx-active');
            $searchInput.css({'width':''});
            $elem.parent().removeClass('ghx-search-active');
            // change the icon to the search icon
            $elem.find('.aui-iconfont-remove').removeClass('aui-iconfont-remove').addClass('aui-iconfont-search-small');
            // clear the text
            $searchInput.val('');
            // And re-execute the search
            self.searchQuery();
        } else {
            $elem.addClass('ghx-active');
            $searchInput.css({'width':self.getExpandedSearchFieldWidth()});
            $elem.parent().addClass('ghx-search-active');
            $searchInput.focus();
        }
    });

    // instant search on key type
    $searchInput.keyup(function(event) {
        if (event.keyCode == 13) {
            $searchInput.blur();
            return;
        }
        // testing if the input has no text for stylin'
        if ($searchInput.val() !== ''){
            // change the icon to the remove icon
            $elem.find('.aui-iconfont-search-small').removeClass('aui-iconfont-search-small').addClass('aui-iconfont-remove');
        } else {
            // change the icon to the search icon
            $elem.find('.aui-iconfont-remove').removeClass('aui-iconfont-remove').addClass('aui-iconfont-search-small');
        }
        self.searchQuery();
    });

    // ensure form is not submitted
    $elem.submit(function(event) {
        event.preventDefault();
    });

    return $elem;
};

GH.Components.SearchFilter.prototype.clearSearchBox = function() {
    var $formElement = AJS.$("#ghx-backlog-search");
    $formElement.removeClass('ghx-active');
    $formElement.parent().removeClass('ghx-search-active');
    $formElement.find(".ghx-search").val('').css({'width':''});
};

GH.Components.SearchFilter.prototype._queryToRegex = function(rawQuery) {
    var finalQuery;
    if (_.isEmpty(rawQuery)) {
        finalQuery = '.*';
    } else {
        finalQuery = _.map(rawQuery.split(" "),
            function (queryItem) {
                return "(?=.*(" + queryItem + "))";
            }).join("");
    }

    GH.Logger.log("Executing search " + finalQuery, GH.Logger.Contexts.ui);
    return new RegExp(finalQuery, "i");
};

GH.Components.SearchFilter.prototype.clearLastSearch = function() {
    this.lastSearch = "";
};

/**
 * Stops the search (does NOT cancel the already typed text but blurs the input field)
 */
GH.Components.SearchFilter.prototype.stopEdit = function(container) {
    var $searchInput = container.find('#ghx-backlog-search-input');
    $searchInput.blur();
};

GH.Components.SearchFilter.prototype.searchQuery = function() {
    GH.Logger.timeStart("GH.Components.SearchFilter.searchQuery");

    var rawQuery = AJS.$.trim(AJS.$("#ghx-backlog-search-input").val());
    // Don't rerun the search if it hasn't changed, or if it is empty
    if (rawQuery === this.lastSearch) {
        return;
    }
    this.lastSearch = rawQuery;
    // TODO Pop a spinner here in the search box?

    if (rawQuery === '') {
        this.searchQueryCallback(null);
    } else {
        var pattern;
        try {
            // Now we munge the query into an appropriate regexp
            pattern = this._queryToRegex(rawQuery);
            this.searchQueryCallback(pattern);
        } catch (e) {
            GH.Logger.log("Regex search threw exception", GH.Logger.Contexts.ui);
            pattern = this._queryToRegex(rawQuery.replace(this.regexReserved, "\\$&"));
            this.searchQueryCallback(pattern);
        }
    }

    GH.Logger.timeStop("GH.Components.SearchFilter.searchQuery");

    // TODO Remove spinner here in the search box?
};
