/**
 * Dynamic colors module
 * @module jira-agile/rapid/ui/component/dynamic-colors
 */
define('jira-agile/rapid/ui/component/dynamic-colors', ['wrm/context-path', 'jira/featureflags/feature-manager', 'jquery', 'underscore', 'jira-agile/rapid/ajax'], function (contextPath, featureManager, $, _, Ajax) {
    var DynamicColors = {
        storedColors: {},
        pendingColors: []
    };

    $(GH).bind("issueUpdated", function (event, data) {
        return DynamicColors.resetColor(data.issueId);
    });
    $(GH).bind("issueTransitioned", function (event, data) {
        return DynamicColors.resetColor(data.issueId);
    });
    $(GH).bind("issueMoved", function (event, data) {
        return DynamicColors.refreshColorByIssueId(data.issueId);
    });

    DynamicColors.loadColors = function (container) {
        container.addClass("ghx-grabber-loading");
        this.fetchColors(container.find('.js-issue[data-issue-id], .ghx-card[data-issue-id]').toArray()).then(function () {
            container.removeClass("ghx-grabber-loading");
        });
    };

    DynamicColors.flushNode = function (node) {
        var color = this.getColor(node.dataset.issueId);
        if (!_.isUndefined(color)) {
            this.updateColor(node, color);
            return;
        }

        this.pendingColors.push(node);
        node.classList.add("ghx-grabber-loading");
        this.flushColors();
    };

    DynamicColors.setupObserver = function () {
        var _this = this;

        if (this.observer) {
            return;
        }

        this.observer = new IntersectionObserver(function (entries) {
            entries.forEach(function (entry) {
                if (entry.isIntersecting) {
                    _this.flushNode(entry.target);
                } else {
                    entry.target.classList.remove("ghx-grabber-loading");
                    _this.pendingColors = _.without(_this.pendingColors, entry.target);
                }
            });
        }, {
            root: $('#ghx-content-main')[0],
            rootMargin: '0px',
            thresholds: [0, 1.0]
        });
    };

    DynamicColors.update = function () {
        var _this2 = this;

        var container = $('#ghx-content-main').find('[data-card-color-strategy="custom"]');

        if (typeof IntersectionObserver === "undefined" || featureManager.isFeatureEnabled('com.atlassian.jira.agile.darkfeature.dynamic.colors.simple')) {
            // old browser, IE, or old FF on bamboo, just load everything
            this.loadColors(container);
            return;
        } else {
            this.setupObserver();
        }

        container.find('.js-issue[data-issue-id]:not(.ghx-observed)').each(function (idx, el) {
            _this2.observeColors(el);
        });
    };

    DynamicColors.refreshColorByIssueId = function (issueId) {
        var _this3 = this;

        var container = $('#ghx-content-main').find('[data-card-color-strategy="custom"]');
        container.find('.js-issue[data-issue-id=' + issueId + '], .ghx-card[data-issue-id=' + issueId + ']').each(function (idx, el) {
            _this3.updateColor(el, null);
            if (_this3.observer) {
                _this3.refreshColor(el);
            } else {
                _this3.putColor(issueId, undefined);
                _this3.fetchColors([el]);
            }
        });
    };

    DynamicColors.refreshColor = function (el) {
        this.resetColor(el.dataset.issueId);
        this.observeColors(el);
    };

    DynamicColors.observeColors = function (el) {
        if (this.observer && !el.classList.contains("ghx-observed")) {
            el.classList.add("ghx-observed");
            this.observer.observe(el);
        } else {
            this.flushNode(el);
        }
    };

    DynamicColors.flushColors = _.debounce(function () {
        var _this4 = this;

        if (this.pendingColors.length === 0 || this.currentFetch && this.currentFetch.state() === "pending") {
            return;
        }

        var nodes = this.pendingColors.splice(0);
        nodes.forEach(function (node) {
            return _this4.observer.unobserve(node);
        });

        // single fetch per session, so that Jira doesn't self DDOS :)
        this.currentFetch = this.fetchColors(nodes).always(function () {
            delete _this4.currentFetch;
            if (_this4.pendingColors.length > 0) {
                _this4.flushColors();
            }
        });
    }, 100);

    DynamicColors.updateColor = function (node, color) {
        node.classList.remove("ghx-grabber-loading");
        if (node.classList.contains("js-issue")) {
            node.querySelector('.ghx-grabber').style.backgroundColor = color;
        } else if (node.classList.contains("ghx-card")) {
            node.querySelector('.ghx-card-color').style.borderColor = color;
        }
    };

    DynamicColors.fetchColors = function (nodes) {
        var _this5 = this;

        // try getting stored colors, instead of fetching them from server
        nodes = nodes.filter(function (node) {
            var color = _this5.getColor(node.dataset.issueId);
            if (!_.isUndefined(color)) {
                _this5.updateColor(node, color);
            } else {
                return true;
            }
        });

        var issueIds = nodes.map(function (node) {
            return node.dataset.issueId;
        });
        if (issueIds.length === 0) {
            return new $.Deferred().resolve();
        }

        // Use Ajax lib instead of regular jQuery because of usage in gadgets
        return Ajax.makeRequest({
            method: "post",
            url: contextPath() + '/rest/greenhopper/1.0/xboard/work/' + GH.RapidBoard.State.getRapidViewId() + '/colors.json',
            accept: "application/json",
            contentType: "application/json",
            dataType: "json",
            // it doesn't make sense performance-wise to query for large number of issues
            // instead we will use board query and return colors for all issues
            data: JSON.stringify({ issues: issueIds.length > 30 ? null : issueIds })
        }).then(function (res) {
            if (res.isSuccess()) {
                Object.keys(res.success.colors).forEach(function (issueId) {
                    _this5.putColor(issueId, res.success.colors[issueId]);
                });

                issueIds.forEach(function (issueId, idx) {
                    var color = res.success.colors[issueId] || null;
                    _this5.putColor(issueId, color);
                    var node = nodes[idx];
                    _this5.updateColor(node, color);
                });
            }
        });
    };

    DynamicColors.getColor = function (issueId) {
        return this.storedColors[issueId];
    };

    DynamicColors.resetColor = function (issueId) {
        delete this.storedColors[issueId];
    };

    DynamicColors.putColor = function (issueId, color) {
        this.storedColors[issueId] = color;
    };

    return DynamicColors;
});