/**
 * Dynamic colors module
 * @module jira-agile/rapid/ui/component/dynamic-colors
 */
define('jira-agile/rapid/ui/component/dynamic-colors', [
    'wrm/context-path',
    'jira/featureflags/feature-manager',
    'jquery',
    'underscore',
    'jira-agile/rapid/ajax',
], function (
    contextPath,
    featureManager,
    $,
    _,
    Ajax
) {
    const DynamicColors = {
        storedColors: {},
        pendingColors: []
    };

    $(GH).bind("issueUpdated", (event, data) => DynamicColors.resetColor(data.issueId));
    $(GH).bind("issueTransitioned", (event, data) => DynamicColors.resetColor(data.issueId));
    $(GH).bind("issueMoved", (event, data) => DynamicColors.refreshColorByIssueId(data.issueId));

    DynamicColors.loadColors = function (container) {
        container.addClass("ghx-grabber-loading");
        this.fetchColors(container.find('.js-issue[data-issue-id], .ghx-card[data-issue-id]').toArray()).then(() => {
            container.removeClass("ghx-grabber-loading");
        });
    };

    DynamicColors.flushNode = function (node) {
        let color = this.getColor(node.dataset.issueId);
        if (!_.isUndefined(color)) {
            this.updateColor(node, color);
            return;
        }

        this.pendingColors.push(node);
        node.classList.add("ghx-grabber-loading");
        this.flushColors();
    };

    DynamicColors.setupObserver = function() {
        if (this.observer) {
            return;
        }

        this.observer = new IntersectionObserver(entries => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    this.flushNode(entry.target);
                } else {
                    entry.target.classList.remove("ghx-grabber-loading");
                    this.pendingColors = _.without(this.pendingColors, entry.target);
                }
            });
        }, {
            root: $('#ghx-content-main')[0],
            rootMargin: '0px',
            thresholds: [0, 1.0]
        });
    };

    DynamicColors.update = function () {
        const container = $('#ghx-content-main').find('[data-card-color-strategy="custom"]');

        if (typeof IntersectionObserver === "undefined" || featureManager.isFeatureEnabled('com.atlassian.jira.agile.darkfeature.dynamic.colors.simple')) {
            // old browser, IE, or old FF on bamboo, just load everything
            this.loadColors(container);
            return;
        } else {
            this.setupObserver();
        }

        container.find('.js-issue[data-issue-id]:not(.ghx-observed)').each((idx, el) => {
            this.observeColors(el);
        });
    };

    DynamicColors.refreshColorByIssueId = function(issueId) {
        const container = $('#ghx-content-main').find('[data-card-color-strategy="custom"]');
        container.find(`.js-issue[data-issue-id=${issueId}], .ghx-card[data-issue-id=${issueId}]`).each((idx, el) => {
            this.updateColor(el, null);
            if (this.observer) {
                this.refreshColor(el);
            } else {
                this.putColor(issueId, undefined);
                this.fetchColors([el]);
            }
        });
    };

    DynamicColors.refreshColor = function (el) {
        this.resetColor(el.dataset.issueId);
        this.observeColors(el);
    };

    DynamicColors.observeColors = function (el) {
        if (this.observer && !el.classList.contains("ghx-observed")) {
            el.classList.add("ghx-observed");
            this.observer.observe(el);
        } else {
            this.flushNode(el);
        }
    };

    DynamicColors.flushColors = _.debounce(function () {
        if (this.pendingColors.length === 0 || this.currentFetch && this.currentFetch.state() === "pending") {
            return;
        }

        let nodes = this.pendingColors.splice(0);
        nodes.forEach(node => this.observer.unobserve(node));

        // single fetch per session, so that Jira doesn't self DDOS :)
        this.currentFetch = this.fetchColors(nodes).always(() => {
            delete this.currentFetch;
            if (this.pendingColors.length > 0) {
                this.flushColors();
            }
        });
    }, 100);

    DynamicColors.updateColor = function (node, color) {
        node.classList.remove("ghx-grabber-loading");
        if (node.classList.contains("js-issue")) {
            node.querySelector('.ghx-grabber').style.backgroundColor = color;
        } else if (node.classList.contains("ghx-card")) {
            node.querySelector('.ghx-card-color').style.borderColor = color;
        }
    };

    DynamicColors.fetchColors = function (nodes) {
        // try getting stored colors, instead of fetching them from server
        nodes = nodes.filter(node => {
            let color = this.getColor(node.dataset.issueId);
            if (!_.isUndefined(color)) {
                this.updateColor(node, color);
            } else {
                return true;
            }
        });

        let issueIds = nodes.map(node => node.dataset.issueId);
        if (issueIds.length === 0) {
            return new $.Deferred().resolve();
        }

        // Use Ajax lib instead of regular jQuery because of usage in gadgets
        return Ajax.makeRequest({
            method: "post",
            url: `${contextPath()}/rest/greenhopper/1.0/xboard/work/${GH.RapidBoard.State.getRapidViewId()}/colors.json`,
            accept: "application/json",
            contentType: "application/json",
            dataType: "json",
            // it doesn't make sense performance-wise to query for large number of issues
            // instead we will use board query and return colors for all issues
            data: JSON.stringify({issues: issueIds.length > 30 ? null : issueIds})
        })
            .then(res => {
                if (res.isSuccess()) {
                    Object.keys(res.success.colors).forEach(issueId => {
                        this.putColor(issueId, res.success.colors[issueId]);
                    });

                    issueIds.forEach((issueId, idx) => {
                        const color = res.success.colors[issueId] || null;
                        this.putColor(issueId, color);
                        const node = nodes[idx];
                        this.updateColor(node, color);
                    });
                }
            });
    };

    DynamicColors.getColor = function (issueId) {
        return this.storedColors[issueId];
    };

    DynamicColors.resetColor = function (issueId) {
        delete this.storedColors[issueId];
    };

    DynamicColors.putColor = function (issueId, color) {
        this.storedColors[issueId] = color;
    };

    return DynamicColors;
});