/* global _ */

/**
 * A context menu that can be shown at a mouse event location.
 *
 * The options object has following expected structure:
 *
 * $options = { actionGroups: an array of $actionGroup objects}
 * $actionGroup = {
 *      label: i18n string, optional
 *      actions: an array of $action objects, required
 * }
 * $action = {
 *      id: a unique id
 *      label: i18n string
 *      name: used for analytics
 *      actionFn: a function that is executed when the action is clicked
 * }
 *
 * @param options
 * @constructor
 */
GH.ContextMenu = function (options) {
    this.actionGroups = options.actionGroups;
    this.dropdown = null;
};
GH.ContextMenu.menuSelector = "#gh-ctx-menu";
GH.ContextMenu.contentSelector = "#gh-ctx-menu-content";
GH.ContextMenu.triggerSelector = "#gh-ctx-menu-trigger";

/**
 * Executes the context action identified by the given actionId.
 * @param actionId
 * @private
 */
GH.ContextMenu.prototype._executeContextAction = function (actionId) {
    var action = this._findActionById(actionId);
    action.actionFn();
};

/**
 * Finds an action configured in the context menu by the action's id
 * @param actionId
 * @return {*}
 * @private
 */
GH.ContextMenu.prototype._findActionById = function(actionId){
    var actions = [];
    _.each(this.actionGroups, function (actionGroup) {
        actions = actions.concat(actionGroup.actions);
    });
    return _.find(actions, function (action) {
        return action.id === actionId;
    });
};

/**
 * Initializes, displays and positions the context menu at the coordinates extracted from the given event
 * @param event
 */
GH.ContextMenu.prototype.showAt = function (event) {
    if (_.isUndefined(event) || _.isNull(event)) {
        AJS.log("Cannot show the issue context menu: expected an event object to be passed");
        return;
    }

    this._initDropdown();

    var xpos = event.pageX;
    var ypos = event.pageY;

    var content = AJS.$(GH.ContextMenu.contentSelector);
    var contentHeight = content.height();
    if(! GH.Util.UI.isVisibleY(ypos + contentHeight)) {
        ypos -= contentHeight;
    }

    this._positionDropdownAt(xpos, ypos);
    this.dropdown.show();
};

GH.ContextMenu.prototype._positionDropdownAt = function(left, top){
    AJS.$(GH.ContextMenu.triggerSelector).offset({
        left:left,
        top:top
    });
};

/**
 * Initializes the dropdown component used to render the context menu
 * @private
 */
GH.ContextMenu.prototype._initDropdown = function(){
    // remove any context menu content
    AJS.$("#gh-ctx-menu").remove();

    var html = GH.tpl.contextmenu.renderContextMenu({
        ctxActionGroups:this.actionGroups
    });
    AJS.$("body").append(html);

    var self = this;
    // attach click listener to menu
    var $content = AJS.$(GH.ContextMenu.contentSelector);
    $content.click(function (event) {
        var $target = AJS.$(event.target).closest('.js-context-menu-action');
        var ctxActionId = $target.attr("data-ctx-action");
        if (_.isUndefined(ctxActionId) || ctxActionId === false) {
            return;
        }
        self._executeContextAction(ctxActionId);
        event.preventDefault();
    });

    var ddOptions = {
        content:$content,
        trigger:AJS.$(GH.ContextMenu.triggerSelector),
        alignment: AJS.LEFT
    };
    this.dropdown = AJS.Dropdown.create(ddOptions)[0];

    // Only Firefox fires a click() event for right clicks,
    // this causes annoying behaviour as the menu closes if you move the mouse while opening it.
    // So we'll ignore right clicks as close events.
    var validateClickToCloseOriginal = this.dropdown.layerController._validateClickToClose;

    var validateClickToCloseDisallowRightClick = function (e) {
        // Right click is not valid
        if (e.button === 2) {
            return false;
        }
        // In the case of no original validation method.
        if (!validateClickToCloseOriginal) {
            return true;
        }
        // Use the original validation.
        return validateClickToCloseOriginal.apply(this, arguments);
    };
    this.dropdown.layerController._validateClickToClose = validateClickToCloseDisallowRightClick;
};

/**
 * Hides the context menu and removes it's html from DOM
 */
GH.ContextMenu.prototype.hide = function () {
    if (this.dropdown) {
        this.dropdown.hide();
    }
    AJS.$(GH.ContextMenu.menuSelector).remove();
    AJS.$(GH.ContextMenu.contentSelector).remove();
    AJS.$(GH.ContextMenu.triggerSelector).remove();
    AJS.$(GH.ContextMenu.triggerSelector + "_drop").remove();
};