/* globals GH */
/**
 * Chart Picker
 * @module jira-agile/rapid/ui/component/chart-picker
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira/ajs/select/single-select
 */
define('jira-agile/rapid/ui/component/chart-picker', ['require'], function (require) {

    // REQUIRES
    var $ = require('jquery');
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var SingleSelect = require('jira/ajs/select/single-select');

    // GLOBALS... FIX ME
    var BoardState;
    var tpl = GH.tpl;

    // Resolve timing issue
    GlobalEvents.on("pre-initialization", function () {
        BoardState = GH.BoardState;
    });

    /**
     * Generic chart picker dropdown implementation
     */
    var ChartPicker = function ChartPicker(selectedItemKey, emptyItemsMsg) {
        this.selectedItemStateKey = selectedItemKey;
        this.items = [];
        this.defaultItem = undefined;
        this.emptyItemsMsg = emptyItemsMsg || AJS.I18n.getText('gh.rapid.sprint.error.nosprint');
    };

    /**
     * Set the items
     */
    ChartPicker.prototype.setItems = function (items) {
        this.items = items || [];
    };

    /**
     * Get the items
     */
    ChartPicker.prototype.getItems = function () {
        return this.items;
    };

    /**
     * Are there any items
     */
    ChartPicker.prototype.hasItems = function () {
        return !_.isEmpty(this.items);
    };

    ChartPicker.prototype.getDefaultItem = function () {
        return this.defaultItem;
    };
    ChartPicker.prototype.getDefaultItemId = function () {
        return this.defaultItem ? this.defaultItem.id : false;
    };

    /**
     * Set the default item to load when no default has been set
     * @param defaultItem an object representing one of the items in the list
     */
    ChartPicker.prototype.setDefaultItem = function (defaultItem) {
        this.defaultItem = defaultItem || undefined;
    };

    /**
     * Get an item given its id
     */
    ChartPicker.prototype.getItemById = function (itemId) {
        var item = _.find(this.items, function (item) {
            return item.id.toString() === itemId.toString();
        });
        return _.isUndefined(item) ? false : item;
    };

    /**
     * Get the currently selected item. This will automatically select the first active item if current selection
     * is invalid.
     * @return a object representing the item or false if no items are available
     */
    ChartPicker.prototype.getSelectedItem = function () {
        var item;
        var storedItemId = this.getStoredSelectedItemId();

        if (storedItemId) {
            item = this.getItemById(storedItemId);
        }
        if (!item) {
            // select the default item if we have one
            if (this.getDefaultItem()) {
                item = this.getDefaultItem();
            } else if (this.hasItems()) {
                // fall back to last item
                item = _.last(this.getItems());
            }
        }

        // update the stored item id
        if (item) {
            // store new item, but don't fire any event

            var newItemId = this.getDefaultItemId() === item.id ? -1 : item.id;
            this.storeSelectedItemId(newItemId);
        }

        return item || false;
    };

    /**
     * Set the new selected item
     */
    ChartPicker.prototype.setSelectedItem = function (selectedItemId, preventEventTrigger) {
        this.previousSelectedItem = this.getSelectedItem();
        // fetch the item for the given id
        var item = this.getItemById(selectedItemId);
        if (item) {
            var newItemId = this.getDefaultItemId() === item.id ? -1 : item.id;
            this.storeSelectedItemId(newItemId);
        } else {
            this.storeSelectedItemId(false);
        }
        if (!preventEventTrigger) {
            $(this).trigger('selectedItemChanged', { item: item });
        }
    };

    ChartPicker.prototype.removeSelectedItemAndRevert = function (container) {
        if (this.previousSelectedItem) {
            var currentSelectedItemId = this.getSelectedItem().id;
            this.items = _.reject(this.items, function (item) {
                return item.id === currentSelectedItemId;
            });
            this.setSelectedItem(this.previousSelectedItem.id);
            this.render(container);
        }
    };

    ChartPicker.prototype.getStoredSelectedItemId = function () {
        // we keep the selected issues in the session selectedItemStateKey, avoids duplicating data in several places
        return BoardState.getPerViewValue(this.selectedItemStateKey, false);
    };

    ChartPicker.prototype.storeSelectedItemId = function (selectedItemId) {
        BoardState.setPerViewValue(this.selectedItemStateKey, selectedItemId || false);
    };

    ChartPicker.prototype.bindToSelectedItemChanged = function (handler) {
        $(this).off('selectedItemChanged', handler) // avoid registering the same handler more than once
        .on('selectedItemChanged', handler);
    };

    //
    // Rendering logic
    //
    ChartPicker.prototype.render = function (container) {
        // render the dropdown choices
        var hasDropdown = this.items.length > 1;
        var selectedItem = this.getSelectedItem();

        // after the frother control is disabled, we want to redraw this
        var self = this;
        var onSelect = function onSelect(event, selected) {
            self.setSelectedItem(selected.value());
        };

        var showDropdownAndRedraw = function showDropdownAndRedraw(e) {
            self.showDropdown(e, function () {
                self.render(container);
            }, onSelect);
        };

        // todo: Do not make a select if there is only one
        if (hasDropdown) {
            container.html(tpl.component.chartpicker.renderSingleSelect({
                items: this.items,
                selectedItem: selectedItem
            }));
            $('#ghx-items-trigger').click(showDropdownAndRedraw);
        } else if (selectedItem) {
            container.html(tpl.component.chartpicker.renderSingleItem({
                selectedItemId: selectedItem.id,
                selectedItemName: selectedItem.name || ''
            }));
        } else {
            $('#ghx-chart-content').html(tpl.component.chartpicker.renderEmpty({ emptyItemsMsg: this.emptyItemsMsg }));
        }

        $('#ghx-chart-picker-form').submit(function () {
            return false;
        });
    };

    ChartPicker.prototype.showDropdown = function (e, redrawFn, selectFn) {
        $(e.target).hide();
        var $select = $('#ghx-chart-picker');
        var singleSelect = new SingleSelect({
            element: $select,
            width: 250,
            errorMessage: '',
            setMaxHeightToWindow: true
        });

        var $field = $('#ghx-chart-picker-field');

        $select.bind('selected', function (e, selected) {
            if (selected.value()) {
                selectFn(e, selected);
            }
            $field.blur();
        });

        $field.blur(redrawFn);

        $field.focus().select();

        // hack :( but I cannot find another way to force the frother control to show the suggestions
        // this seems to be the shortest timeout that works reliably
        setTimeout(function () {
            $('#ghx-chart-picker-field').trigger('click');
        }, 20);
    };

    return ChartPicker;
});