/* global _, Calendar */
if (!GH.Components) {
    GH.Components = {};
}

GH.Components.SPRINT_NAME_MAX_LENGTH = 30;
GH.Components.SPRINT_GOAL_MAX_LENGTH = 1000;

/** Event to fire when a sprint is updated */
GH.Components.EVENT_SPRINT_UPDATED = "sprintUpdated";

/**
 * Create a new SprintFilters component
 * @param getSprintByIdFn a function that returns a sprint data object for a given sprint id
 */
GH.Components.ActiveSprintInlineEdit = function(getSprintByIdFn) {
    this.getSprintById = getSprintByIdFn;
};

GH.Components.ActiveSprintInlineEdit.prototype.rapidViewConfig = {};

/**
 * Set the configuration of the current rapid view
 */
GH.Components.ActiveSprintInlineEdit.prototype.setRapidViewConfig = function(rapidViewConfig) {
    this.rapidViewConfig = rapidViewConfig;
};

GH.Components.ActiveSprintInlineEdit.prototype.setSprintData = function(sprints) {
    this.sprints = sprints || [];
};

GH.Components.ActiveSprintInlineEdit.prototype.getSprintById = function(sprintId) {
    return _.find(this.sprints, function(sprint) {
        return sprint.id === sprintId;
    });
};

GH.Components.ActiveSprintInlineEdit.prototype.updateSprintData = function(sprintId, sprintData) {
    for(var i=0; i < this.sprints.length; i++) {
        if (this.sprints[i].id === sprintId) {
            this.sprints[i] = sprintData;
        }
    }
};

/**
 * Analytics for the sprint edit dropdown
 */
GH.Components.ActiveSprintInlineEdit.prototype.analytics = function(action, label) {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    new AnalyticsTracker('gh.edit.sprint.inlineedit.change').trigger(action, label); // SAFE
};

/**
 * Inline edit data getter.
 */
GH.Components.ActiveSprintInlineEdit.prototype.createGetInlineEditDataFunction = function() {
    var self = this;
    return function(viewElement, options) {
        // force other edits to close
        GH.RapidBoard.Util.InlineEditable.submitActiveEdits();

        var editData = GH.RapidBoard.Util.InlineEditable.defaultFieldData(viewElement);
        switch (editData.fieldName) {
            case 'sprintName':
                editData.maxLength = GH.Components.SPRINT_NAME_MAX_LENGTH;
                editData.cssClass = 'medium-field';
                break;
            case 'sprintGoal':
                editData.maxLength = GH.Components.SPRINT_GOAL_MAX_LENGTH;
                editData.cssClass = 'medium-field';
                editData.extraClasses = 'overflow-ellipsis';
                break;
        }
        // fetch the sprint id from the dialog
        editData.sprintId = parseInt(viewElement.closest('.js-sprint-container').attr('data-sprint-id'), 10);
        editData.sprintData = _.clone(self.getSprintById(editData.sprintId));
        return editData;
    };
};

/**
 * Submits updated sprint data
 */
GH.Components.ActiveSprintInlineEdit.prototype.submitSprintData = function(sprintData) {
    var self = this;
    var sprintId = sprintData.id;
    var editSprintRequest = {
        name : sprintData.name,
        startDate : sprintData.startDate,
        endDate : sprintData.endDate,
        goal: sprintData.goal
    };

    // remove previous errors
    GH.Validation.clearContextualErrors('.js-sprint-container[data-sprint-id=' + sprintId + ']');

    GH.Ajax.put({
        url: '/sprint/' + sprintId,
        data: editSprintRequest,
        errorContextMap: {
            'name' : '.js-sprint-container[data-sprint-id=' + sprintId + '] .ghx-name .js-editable-field',
            // note: if there is an error for both startDate and endDate, there will be two ghx-error divs
            'startDate' : '.js-sprint-container[data-sprint-id=' + sprintId + '] .ghx-dates > :last-child',
            'endDate' : '.js-sprint-container[data-sprint-id=' + sprintId + '] .ghx-dates > :last-child',
            'dateFormatMismatch': GH.Notification.handleDateFormatMismatchError,
            'goal' : '.js-sprint-container[data-sprint-id=' + sprintId + '] .ghx-sprint-goal > .js-editable-field'
        }
    })
        .done(function(data) {
            AJS.$(GH).trigger(GH.Components.EVENT_SPRINT_UPDATED, data.success);
        });

};

GH.Components.ActiveSprintInlineEdit.prototype.validateName = function(editData) {
    var newValue = editData.editElement.val();
    if (!GH.Validation.notBlank(editData.editElement, AJS.I18n.getText('gh.sprint.error.name.required'))) {
        return false;
    }
    if (!GH.Validation.maxLength(editData.editElement, GH.Components.SPRINT_NAME_MAX_LENGTH, AJS.I18n.getText('gh.sprint.error.name.too.long', GH.Components.SPRINT_NAME_MAX_LENGTH))) {
        return false;
    }
    editData.sprintData.name = newValue;
    editData.newValue = newValue;
    return true;
};

GH.Components.ActiveSprintInlineEdit.prototype.validateStartDate = function(editData) {
    var newValue = editData.editElement.find('input').val();
    if (!GH.Validation.notBlank(editData.editElement.find('input'), AJS.I18n.getText('gh.sprint.error.start.date.required'), AJS.$(".ghx-dates"))) {
        return false;
    }
    editData.sprintData.startDate = newValue;
    editData.newValue = newValue;
    return true;
};

GH.Components.ActiveSprintInlineEdit.prototype.validateEndDate = function(editData) {
    var newValue = editData.editElement.find('input').val();
    if (!GH.Validation.notBlank(editData.editElement.find('input'), AJS.I18n.getText('gh.sprint.error.end.date.required'), AJS.$(".ghx-dates"))) {
        return false;
    }
    editData.sprintData.endDate = newValue;
    editData.newValue = newValue;
    return true;
};

GH.Components.ActiveSprintInlineEdit.prototype.validateSprintGoal = function (editData) {
    var newValue = editData.editElement.val().trim();

    if (!GH.Validation.maxLength(editData.editElement, GH.Components.SPRINT_GOAL_MAX_LENGTH, AJS.I18n.getText('gh.sprint.error.goal.too.long', GH.Components.SPRINT_GOAL_MAX_LENGTH))) {
        return false;
    }

    editData.newValue = newValue;
    editData.sprintData.goal = newValue.length ? newValue : null;
    return true;
};

GH.Components.ActiveSprintInlineEdit.prototype.postEditSprintGoal = function (editData) {
    if (!editData.fieldValue.length) {
        editData.viewElement.parent().addClass('ghx-hidden');
    }
};

GH.Components.ActiveSprintInlineEdit.prototype.initializeInlineEdits = function() {
    var self = this;

    var cal = new Calendar(); // GHS-5829 Need to call new calendar to set up short month names and day names when they are not defined

    // Saves the changed data, we always update the complete information, not just a field
    var saveSprint = function(editData) {
        // only submit changes if there are any, otherwise just update view
        var existingValues = _.clone(self.getSprintById(editData.sprintId));
        if (!_.isEqual(existingValues, editData.sprintData)) {
            self.submitSprintData(editData.sprintData);
            var state = (existingValues.state) ? existingValues.state.toLowerCase() : "";
            self.analytics("update" + state, editData.fieldName);
        }
        // give fast feedback
        GH.RapidBoard.Util.InlineEditable.updateView(editData);
    };

    var preEdit = function(editData) {
        GH.Validation.clearContextualErrors('.js-sprint-container[data-sprint-id=' + editData.sprintId + ']');
        GH.RapidBoard.Util.InlineEditable.cancelActiveEdits();
    };
    // edit sprint name
    GH.RapidBoard.Util.InlineEditable.register('.js-sprint-container .js-edit-sprintName-trigger', {
        getData: this.createGetInlineEditDataFunction(),
        validate : this.validateName,
        save : saveSprint
    });
    GH.RapidBoard.Util.InlineEditable.register('.js-sprint-container .js-edit-startDate-trigger', {
        getData: this.createGetInlineEditDataFunction(),
        preEdit: preEdit,
        renderEdit: GH.RapidBoard.Util.InlineEditable.renderDatePickerInlineEdit,
        activate: GH.RapidBoard.Util.InlineEditable.datePickerActivate,
        validate : this.validateStartDate,
        save : saveSprint
    });
    GH.RapidBoard.Util.InlineEditable.register('.js-sprint-container .js-edit-endDate-trigger', {
        getData: this.createGetInlineEditDataFunction(),
        preEdit: preEdit,
        renderEdit: GH.RapidBoard.Util.InlineEditable.renderDatePickerInlineEdit,
        activate: GH.RapidBoard.Util.InlineEditable.datePickerActivate,
        validate : this.validateEndDate,
        save : saveSprint
    });

    GH.RapidBoard.Util.InlineEditable.register('.js-sprint-container .js-edit-sprintGoal-trigger', {
        getData: this.createGetInlineEditDataFunction(),
        preEdit: preEdit,
        validate: this.validateSprintGoal,
        save: saveSprint,
        postEdit: this.postEditSprintGoal

    });
};
