/* global GH */
/**
 * Epic Burndown Report View
 * @module jira-agile/rapid/ui/chart/v2/epic-burndown-report-view
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/d3
 * @requires module:jira-agile/rapid/events
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 * @requires module:jira-agile/rapid/ui/component/chart-picker
 */
define('jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-report-view', ['require'], function(require) {
    'use strict';

    // REQUIRES
    var $ = require('jquery');
    var _ = require('underscore');
    var ChartPicker = require('jira-agile/rapid/ui/component/chart-picker');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var d3 = require('jira-agile/d3');
    var Events = require('jira-agile/rapid/events');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    // GLOBALS... FIX ME
    var Layout;
    var Ajax;
    var LinkedPagesController;
    var RapidBoardViewController;
    var RapidViewConfig;
    var ScopeBurndownBySprintChart;
    var ScopeBurndownBySprintChartLegend;
    var ScopeBurndownBySprintChartHowTo;

    GlobalEvents.on("pre-initialization", function() {
        Layout = GH.Layout;
        Ajax = GH.Ajax;
        LinkedPagesController = GH.LinkedPagesController;
        RapidBoardViewController = GH.RapidBoard.ViewController;
        RapidViewConfig = GH.RapidViewConfig;
        ScopeBurndownBySprintChart = GH.Reports.ScopeBurndownBySprintChart;
        ScopeBurndownBySprintChartLegend = GH.Reports.ScopeBurndownBySprintChartLegend;
        ScopeBurndownBySprintChartHowTo = GH.Reports.ScopeBurndownBySprintChartHowTo;
    });

    function EpicBurndownReportView() {
        new Events(this);
    }

    EpicBurndownReportView.prototype = {
        _chart: null,
        _legend: null,
        _howTo: null,
        _firstRender: true,
        _model: null,

        init: function (model) {
            this._model = model;

            ChartView.getChartContentElem(true).html(GH.tpl.reports.epicBurndownReport.renderChartBody());
            ChartView.getChartLegendD3(true).html(GH.tpl.reports.epicBurndownReport.renderLegend()).show();
            this._renderOptions();

            this._epicPicker = new ChartPicker('epicBurndown.selectedEpicId', AJS.I18n.getText('gh.report.epic.noepics'));
            this._epicPicker.bindToSelectedItemChanged(_.bind(this._handleEpicSelected, this));

            this._chart = new ScopeBurndownBySprintChart();
            this._legend = new ScopeBurndownBySprintChartLegend();
            this._howTo = new ScopeBurndownBySprintChartHowTo();

            this._renderLegend();
            this._renderHowTo();
        },

        destroy: function () {
            this._chart = null;
            this._legend = null;
            this._howTo = null;
            ChartView.getChartContentElem(true).empty();
            ChartView.getChartLegendD3(true).empty();
            $(GH).unbind(Layout.EVENT_DELAYED_RESIZE + '.epicBurndownReport');
        },

        setEpics: function (epics) {
            this._epics = epics;
            this._epicPicker.setItems(epics);
            this._epicPicker.setDefaultItem(epics[0]);
        },

        update: function (data, supportsPages) {
            if (supportsPages) {
                this._renderLinkedPagesTrigger(data.epic.key);
            }

            this._epicPicker.setSelectedItem(data.epic.key, true);

            this._epicPicker.render(ChartView.getChartSelector(true));
            this._chart.data(data);
            this._renderChart();
            this._renderSnapshot(data);
            this._renderEpicMetadata(data.epic);

            if (this._firstRender) {
                $(GH).bind(Layout.EVENT_DELAYED_RESIZE + '.epicBurndownReport', _.bind(this._handleWindowResized, this));
            } else {
                ChartView.showChartUpdatedToast();
            }
            this._firstRender = false;
        },

        updateTable: function (data) {
            this._renderTable(data);
        },

        showBlankState: function (hasNoEpic, hasNoEstimatedStories) {
            var $message = $(GH.tpl.reports.epicBurndownReport.renderBlankStateMessage({
                planModeUrl: Ajax.CONTEXT_PATH + '/secure/RapidBoard.jspa?rapidView=' + RapidViewConfig.currentData.data.id + '&view=planning' + (hasNoEpic ? '&epics=visible' : ''),
                hasNoEpic: hasNoEpic,
                hasNoEstimatedStories: hasNoEstimatedStories
            }));

            $message.find('.js-plan-mode').on('simpleClick', function (e) {
                e.preventDefault();
                if (hasNoEpic) {
                    GH.PlanController.setEpicsColumnVisible(true);
                }
                RapidBoardViewController.setMode('plan');
            });

            ChartView.showBlankStateMessage($message);
        },

        hideBlankState: function () {
            ChartView.hideBlankStateMessage();
        },

        showSpinner: function () {
            ChartView.showSpinner();
        },

        hideSpinner: function () {
            ChartView.hideSpinner();
        },

        _renderSnapshot: function (data) {
            var UNESTIMATED_ISSUES_WARNING_THRESHOLD = 30;
            var unestimatedIssuesCount = data.snapshot.estimatableIssueCount - data.snapshot.estimatedIssueCount;
            var unestimatedIssuesPercentage = data.snapshot.estimatableIssueCount > 0 ? Math.round((unestimatedIssuesCount / data.snapshot.estimatableIssueCount) * 100) : 0;
            var snapshotHtml = data.snapshot.issueCount === 0 ? '' : GH.tpl.reports.epicBurndownReport.renderSnapshot({
                estimatableIssuesCount: data.snapshot.estimatableIssueCount,
                unestimatedIssuesCount: unestimatedIssuesCount,
                unestimatedIssuesPercentage: unestimatedIssuesPercentage,
                unestimatedIssuesThresholdExceeded: unestimatedIssuesPercentage > UNESTIMATED_ISSUES_WARNING_THRESHOLD,
                workCompleted: data.estimationStatistic.formatter.formatFull(data.snapshot.workCompleted),
                workTotal: data.estimationStatistic.formatter.formatFull(data.snapshot.workCompleted + data.snapshot.workRemaining),
                estimationStatistic: data.estimationStatistic.name,
                shouldShowUnestimatedIssues : data.estimationStatistic.typeId !== 'issueCount'
            });
            $('#ghx-chart-snapshot').html(snapshotHtml);
            GH.Tooltip.tipsify({
                selector: '#ghx-chart-snapshot .ghx-unestimated-issues',
                html: true,
                className: 'ghx-tipsy-left'
            });
        },

        _renderEpicMetadata: function(epic) {
            $('.js-epic-metadata').html(GH.tpl.reports.epicBurndownReport.renderEpicMetadata(epic));
        },

        _renderLinkedPagesTrigger: function(epicId) {
            var linkPagesTrigger = GH.tpl.component.linkedpages.renderTrigger({
                title: AJS.I18n.getText('gh.epic.view.pages.tip')
            });
            var $trigger = $('#ghx-chart-header-primary .ghx-linked-pages-trigger');
            $trigger.remove();
            $trigger = $(linkPagesTrigger);
            $('#ghx-chart-selector').after($trigger);
            LinkedPagesController.initDialogTrigger(LinkedPagesController.EPIC, GH.LinkedPagesController.REPORT, epicId, $trigger);
        },

        _renderTable: function (data) {
            ChartView.getChartContentElem(true).find('.ghx-report-issue-list')
                .html(GH.tpl.reports.scopeBurndownBySprint.renderIssueList(_.extend({
                    emptySprintMessage: AJS.I18n.getText('gh.rapid.charts.epicburndown.no.work.completed')
                }, data)));
        },

        _renderLegend: function () {
            this._legend(d3.select('.ghx-chart-legend'));
        },

        _renderHowTo: function () {
            this._howTo(d3.select('.ghx-group-how'));
        },

        _renderChart: function () {
            this._chart(d3.select('#ghx-epic-burndown-report'));
        },

        _renderOptions: function() {
            var view = this;
            $('#ghx-chart-options').html(GH.tpl.reports.epicBurndownReport.renderChartViewingOptions(this._model));
            $('#js-option-zero-baseline').on('change', function() {
                view.trigger('optionChanged', { zeroBaseline: this.checked });
            });
        },

        _handleEpicSelected: function (event, data) {
            this.trigger('epicSelected', data.item);
        },

        _handleWindowResized: function () {
            this._renderChart();
        }
    };

    return EpicBurndownReportView;
});
