define('jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-chart-data-service', ['require'], function(require) {
    'use strict';

    // REQUIRES
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    // GLOBALS... FIX ME
    var Ajax;
    var RapidViewConfig;
    var ScopeBurndownBySprintTransformer;

    GlobalEvents.on("pre-initialization", function() {
        Ajax = GH.Ajax;
        RapidViewConfig = GH.RapidViewConfig;
        ScopeBurndownBySprintTransformer = GH.Reports.ScopeBurndownBySprintTransformer;
    });

    /**
     * Sends request to retrieve data
     *
     * @param {Object} requestParams
     * @returns {Promise}
     */
    function fetchData(requestParams) {
        var opts = {
            url: '/rapid/charts/epicburndownchart',
            data: requestParams
        };
        return Ajax.get(opts, 'epicBurndownChartData');
    }

    /**
     * Transform data to a form that the chart can render.
     *
     * @param {Object} data
     * @param {boolean} [zeroBaseline]
     * @returns {Object} transformed
     */
    function transformData(data, zeroBaseline) {
        return ScopeBurndownBySprintTransformer.getScopeBySprintData(data, zeroBaseline);
    }

    function EpicBurndownChartDataService() {}

    EpicBurndownChartDataService.prototype = {
        /**
         *
         * @param {number} rapidViewId
         * @param {number} epicKey
         * @param {Object} filterOptions
         * @returns {Promise}
         */
        get: function(rapidViewId, epicKey, filterOptions) {
            return this._getData(rapidViewId, epicKey, filterOptions).andThen(function(data) {
                data.rapidViewId = RapidViewConfig.currentData.id;
                data.estimationStatistic = RapidViewConfig.currentData.data.estimationStatistic;
                data.estimationStatistic.formatter = ScopeBurndownBySprintTransformer.getEstimationStatisticFormatter(data.estimationStatistic.renderer);
                data.labels = {
                    originalEstimate: AJS.I18n.getText('gh.rapid.charts.epicburndown.originalestimate'),
                    workAdded: AJS.I18n.getText('gh.rapid.work.added.epic'),
                    workRemoved: AJS.I18n.getText('gh.rapid.work.removed.epic'),
                    forecastDescription: function(sprintsRemaining) {
                        return AJS.I18n.getText('gh.rapid.charts.epicburndown.forecastdescription', sprintsRemaining);
                    },
                    helpUrl: GH.HelpPaths.getHelpPath("viewing.epic.burndown").url
                };

                data.epic = {
                    key: data.epicKey,
                    summary: data.epicSummary,
                    creationTime: data.epicCreationTime
                };

                return transformData(data, filterOptions.zeroBaseline);
            });
        },

        /**
         * Clear cached data
         */
        clearCache: function() {
            if (this._dataPromise) {
                this._dataPromise = null;
                this._dataRequestParams = null;
            }
        },

        /**
         * For testing
         *
         * @private
         */
        _transformData: transformData,

        /**
         * Gets raw data for given rapidViewId, epicKey and filterOptions. Data could be retrieved from cache if available.
         *
         * @param {number} rapidViewId
         * @param {number} epicKey
         * @param {object} [filterOptions]
         * @returns {Promise}
         * @private
         */
        _getData: function(rapidViewId, epicKey, filterOptions) {
            var requestParams = {
                rapidViewId: rapidViewId,
                epicKey: epicKey
            };
            // Only send a request if previous one cannot be used
            // TODO need a way to purge the cache when navigating away from the report
            if (!this._dataPromise || this._dataPromise.state() === 'rejected' || !_.isEqual(this._dataRequestParams, requestParams)) {
                this._dataRequestParams = requestParams;
                this._dataPromise = fetchData(requestParams);
            }
            return this._dataPromise;
        }
    };

    return EpicBurndownChartDataService;
});