
/* globals GH */
define('jira-agile/rapid/ui/chart/v2/burnup/burnup-report-view', ['require'], function (require) {
    // REQUIRES
    var jq = require('jquery');
    var _ = require('underscore');
    var d3 = require('jira-agile/d3');
    var Events = require('jira-agile/rapid/events');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var SprintBurnupChart = require('jira-agile/rapid/ui/chart/v2/burnup/burnup-chart');
    //var ChartPicker = require('jira-agile/rapid/ui/component/chart-picker');

    // GLOBALS... FIX ME
    var Layout;
    var Ajax;
    var RapidViewConfig;

    var RapidBoard = GH.RapidBoard;
    var SprintBurnupChartLegend;
    var ScopeBurndownBySprintChartHowTo;

    GlobalEvents.on('pre-initialization', function () {
        SprintBurnupChartLegend = GH.Reports.BurnupChartLegend;
        ScopeBurndownBySprintChartHowTo = GH.Reports.ScopeBurndownBySprintChartHowTo;
        Layout = GH.Layout;
        Ajax = GH.Ajax;
        RapidViewConfig = GH.RapidViewConfig;
    });

    function SprintBurnupReportView() {
        new Events(this);
    }

    SprintBurnupReportView.prototype = {
        _chart: null,
        _legend: null,
        _howTo: null,
        _firstRender: true,
        _model: null,

        init: function init(model) {
            this._model = model;

            ChartView.getChartContentElem(true).html(GH.tpl.reports.burnupReport.renderChartBody());
            ChartView.getChartLegendD3(true).html(GH.tpl.reports.burnupReport.renderLegend()).show();

            var rapidViewId = RapidViewConfig.currentData.data.id;

            this._sprintPicker = new GH.ChartPicker('sprintBurnup.selectedSprintId');
            this._sprintPicker.bindToSelectedItemChanged(_.bind(this._handleSprintSelected, this));

            this._estimatePicker = new GH.EstimatePicker('sprintBurnup.selectedEstimate');
            this._estimatePicker.load(rapidViewId, this._handleEstimateSelected.bind(this));

            this._chart = new SprintBurnupChart();
            this._legend = SprintBurnupChartLegend();
            this._howTo = ScopeBurndownBySprintChartHowTo();
        },

        destroy: function destroy() {
            this._chart = null;
            this._legend = null;
            this._howTo = null;
            this._model = null;
            ChartView.getChartContentElem(true).empty();
            ChartView.getChartLegendD3(true).empty();
            jq(GH).unbind(Layout.EVENT_DELAYED_RESIZE + '.sprintBurnupReport');
        },

        setSprints: function setSprints(sprints) {
            this._sprintPicker.setItems(sprints);
            this._sprintPicker.setDefaultItem(sprints[0]);
        },

        update: function update(data) {
            this._sprintPicker.setSelectedItem(data.sprintId, true);
            this._sprintPicker.render(ChartView.getChartSelector(true));
            this._estimatePicker.setSelectedItem(data.statisticField.id, true);
            this._estimatePicker.render(ChartView.getChartControls(true));
            this._renderLegend();
            this._renderHowTo();
            this._renderChart(data);

            this.data = data;

            if (this._firstRender) {
                jq(GH).bind(Layout.EVENT_DELAYED_RESIZE + '.sprintBurnupReport', _.bind(this._handleWindowResized, this));
            } else {
                ChartView.showChartUpdatedToast();
            }
            this._firstRender = false;
        },

        updateTable: function updateTable(data) {
            this._renderTable(data);
        },

        showBlankState: function showBlankState(hasNoEstimatedStories) {
            this._blankState = true;

            var $message = jq(GH.tpl.reports.burnupReport.renderBlankStateMessage({
                planModeUrl: Ajax.CONTEXT_PATH + '/secure/RapidBoard.jspa?rapidView=' + RapidViewConfig.currentData.data.id + '&view=planning&versions=visible',
                hasNoEstimatedStories: hasNoEstimatedStories
            }));

            $message.find('.js-plan-mode').on('simpleClick', function (e) {
                e.preventDefault();
                RapidBoard.ViewController.setMode('plan');
            });

            this._chart.hide(d3.select('#ghx-sprint-burnup-report'));
            ChartView.getChartLegendD3(true).hide();
            ChartView.getChartContentElem(true).find('.ghx-report-issue-list').hide(); // data table
            ChartView.showBlankStateMessage($message);
        },

        hideBlankState: function hideBlankState() {
            this._blankState = false;

            ChartView.hideBlankStateMessage();
            ChartView.getChartLegendD3(true).show();
            ChartView.getChartContentElem(true).find('.ghx-report-issue-list').show(); // data table
        },

        showSpinner: function showSpinner() {
            ChartView.showSpinner();
        },

        hideSpinner: function hideSpinner() {
            ChartView.hideSpinner();
        },

        _renderTable: function _renderTable(rawdata) {
            if (this._blankState) {
                return;
            }

            var chartModel = rawdata.chartData;

            var currentScope = chartModel.scope[0].value;
            var currentDone = 0;

            function mapIssueToSummary(issueArray, issueToSummary) {
                return issueArray.map(function (issue) {
                    return { key: issue, summary: issueToSummary[issue] };
                });
            }

            var rows = chartModel.scope.concat(chartModel.work).sort(function (a, b) {
                return a.timestamp > b.timestamp ? 1 : a.timestamp === b.timestamp ? 0 : -1;
            }).map(function (datum) {
                var row = {
                    timestamp: datum.timestamp,
                    dateString: chartModel.formatDate(datum.timestamp),
                    issues: mapIssueToSummary([datum.change.key], chartModel.issueToSummary),
                    event: datum.change.type,
                    eventString: chartModel.formatEventType(datum.change.type)
                };

                var delta = parseInt(datum.change.toValue) - parseInt(datum.change.fromValue) || 0;

                if (datum.change.line === 'done') {
                    currentDone += delta;
                    if (delta === 0) {
                        row.doneValue = chartModel.formatStatistic(currentDone);
                    } else {
                        row.doneChange = {
                            from: chartModel.formatStatistic(currentDone - delta),
                            to: chartModel.formatStatistic(currentDone)
                        };
                    }
                    row.scopeValue = chartModel.formatStatistic(currentScope);
                }

                if (datum.change.line === 'scope') {
                    currentScope += delta;
                    if (delta === 0) {
                        row.scopeValue = chartModel.formatStatistic(currentScope);
                    } else {
                        row.scopeChange = {
                            from: chartModel.formatStatistic(currentScope - delta),
                            to: chartModel.formatStatistic(currentScope)
                        };
                    }
                    row.doneValue = chartModel.formatStatistic(currentDone);
                }

                return row;
            });

            // collapse same timestamps
            rows = rows.reduce(function (arr, row) {
                var prevRow = arr.pop();

                if (!prevRow) {
                    arr.push(row);
                    return arr;
                }

                if (prevRow.timestamp === row.timestamp && prevRow.event === row.event && prevRow.issueKey === row.issueKey) {
                    // Collapse the 2 rows if they don't have conlicting work-done and/or scope changes.
                    if (!(prevRow.doneChange && row.doneChange) && !(prevRow.scopeChange && row.scopeChange)) {
                        // If we're collapsing the rows, push changes from the second row to the first.
                        if (!prevRow.doneChange) {
                            prevRow.doneChange = row.doneChange;
                        }
                        if (!prevRow.scopeChange) {
                            prevRow.scopeChange = row.scopeChange;
                        }
                        arr.push(prevRow);
                        return arr;
                    }
                }

                // put the rows back, and put the new one on top
                arr.push(prevRow);
                arr.push(row);

                return arr;
            }, []);

            // show the initial issue list in alphabetical order.
            rows[0].issues = mapIssueToSummary(chartModel.startKeys, chartModel.issueToSummary);

            if (!chartModel.sprintCompleted) {
                // sprint in progress... don't show the finished event.
                rows.pop();
            } else {
                // sprint done... show the final issue list in alphabetical order.
                rows[rows.length - 1].issues = mapIssueToSummary(chartModel.endKeys, chartModel.issueToSummary);
            }

            // render the info
            var dataDiv = AJS.$(GH.tpl.reports.burnupReport.renderChartTable({ rows: rows }));

            ChartView.getChartContentElem(true).find('.ghx-report-issue-list').empty().append(dataDiv);
        },

        _renderLegend: function _renderLegend() {
            if (this._blankState) {
                return;
            }

            this._legend(d3.select('.ghx-chart-legend'));
        },

        _renderHowTo: function _renderHowTo() {
            if (this._blankState) {
                return;
            }

            this._howTo(d3.select('.ghx-group-how'));
        },

        _renderChart: function _renderChart(data) {
            if (this._blankState) {
                return;
            }

            var container = d3.select('#ghx-sprint-burnup-report');

            if (container.empty()) {
                // containing element for the chart not found - don't render.
                return;
            }

            this._chart.show(data, container);
        },

        _handleSprintSelected: function _handleSprintSelected(event, data) {
            this.trigger('sprintSelected', data.item.id);
        },

        _handleEstimateSelected: function _handleEstimateSelected(selectedId) {
            this.trigger('estimateSelected', selectedId);
        },

        _handleWindowResized: function _handleWindowResized() {
            this._renderChart(this.data);
        }
    };

    return SprintBurnupReportView;
});