/* global module, test, deepEqual, ok, strictEqual, GH, _ */

AJS.test.require('com.pyxis.greenhopper.jira:gh-rapid', function() {
    module('ScopeBurndownBySprintTransformer tests', {
        setup: function() {
            this.service = GH.Reports.ScopeBurndownBySprintTransformer;
        },

        /**
         * Make n sprints, starting from Sprint 1.
         * Sprint start times are 100, 200, 300, ...
         *
         * @param n
         * @returns {Object[]}
         */
        makeSprints: function(n) {
            var SPRINT_LENGTH = 100;
            return _.map(_.range(1, 1 + n), function(sprintId) {
                return {
                    id: sprintId,
                    name: 'Sprint ' + sprintId,
                    startTime: sprintId * SPRINT_LENGTH,
                    endTime: (sprintId + 1) * SPRINT_LENGTH,
                    state: 'CLOSED'
                };
            });
        },
        /**
         * Various functions for creating the change entries returned from the server.
         */
        addToScopeBurndown: function(key) {
            return {
                key: key,
                added: true
            };
        },
        removeFromScopeBurndown: function(key) {
            return {
                key: key,
                added: false
            };
        },
        moveToDone: function(key) {
            return {
                key: key,
                column: { notDone: false }
            };
        },
        moveToNotDone: function(key) {
            return {
                key: key,
                column: { notDone: true }
            };
        },
        changeEstimate: function(key, estimate) {
            return {
                key: key,
                statC: {
                    newValue: estimate
                }
            };
        },
        /**
         * Generating a single change that:
         *     - creates an issue,
         *     - gives it an estimate,
         *     - adds it to the ScopeBurndown
         */
        addAndEstimate: function(key, estimate) {
            return [
                _.extend(
                    this.moveToNotDone(key),
                    this.changeEstimate(key, estimate),
                    this.addToScopeBurndown(key)
                )
            ];
        },

        getOriginalEstimateSprintName: function() {
            return 'Original estimate';
        }
    });

    test('Typical data', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                // Issues are created and estimated
                1: [this.moveToNotDone('TEST-1'), this.moveToNotDone('TEST-2'), this.moveToNotDone('TEST-3')],
                2: [this.changeEstimate('TEST-1', 5)],
                3: [this.changeEstimate('TEST-2', 3), this.changeEstimate('TEST-3', 0)],
                // Issues are added to the ScopeBurndown
                10: [this.addToScopeBurndown('TEST-1'), this.addToScopeBurndown('TEST-2'), this.addToScopeBurndown('TEST-3')],
                // First sprint
                100: [this.moveToDone('TEST-1')],
                // Second sprint
                200: [this.moveToDone('TEST-2')],
                299: [this.moveToNotDone('TEST-4'), this.addToScopeBurndown('TEST-4'), this.changeEstimate('TEST-4', 8)],
                // Third sprint
                300: [this.moveToDone('TEST-3')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints,  [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 8,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 8,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: 0,
                workAtStart: 8,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 5,
                workRemaining: 3,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            },
            {
                sprintId: 2,
                sprintName: 'Sprint 2',
                baseline: -8,
                workAtStart: 3,
                workAdded: 8,
                workRemoved: 0,
                workCompleted: 3,
                workRemaining: 8,
                startTime: 200,
                endTime: 300,
                isForecast: false,
                isActive: false,
                issues: ['TEST-2']
            },
            {
                sprintId: 3,
                sprintName: 'Sprint 3',
                baseline: -8,
                workAtStart: 8,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 8,
                startTime: 300,
                endTime: 400,
                isForecast: false,
                isActive: false,
                issues: ['TEST-3']
            }
        ]);
        deepEqual(result.forecast, {
            sprintsRemaining: 3,
            workRemaining: 8,
            velocity: 3,
            baseline: -8,
            isLastSprintForecast: false
        });
        deepEqual(result.snapshot, {
            workRemaining: 8,
            estimatedIssueCount: 4,
            estimatableIssueCount: 4,
            issueCount: 4,
            workCompleted: 8
        });
    });

    test('No sprints, no changes', function() {
        var data = {
            sprints: [],
            changes: {},
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: []
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 0,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 0,
                isForecast: false,
                isActive: false,
                issues: []
            }
        ]);
        deepEqual(result.snapshot, {
            workRemaining: 0,
            estimatedIssueCount: 0,
            issueCount: 0,
            workCompleted: 0,
            estimatableIssueCount: 0
        });
    });

    test('No sprints, only estimated issues in ScopeBurndown', function() {
        var data = {
            sprints: [],
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                2: this.addAndEstimate('TEST-2', 5)
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 8,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 8,
                isForecast: false,
                isActive: false,
                issues: []
            }
        ]);
        deepEqual(result.snapshot, {
            workRemaining: 8,
            estimatedIssueCount: 2,
            issueCount: 2,
            workCompleted: 0,
            estimatableIssueCount: 2
        });
    });

    test('Work done before first sprint', function() {
        var data = {
            sprints: this.makeSprints(10),
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                2: this.addAndEstimate('TEST-2', 5),
                3: [this.moveToDone('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 8,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 3,
                workRemaining: 5,
                isForecast: false,
                isActive: false,
                issues: [ 'TEST-1' ]
            }
        ]);
        deepEqual(result.snapshot, {
            workRemaining: 5,
            estimatedIssueCount: 2,
            issueCount: 2,
            workCompleted: 3,
            estimatableIssueCount: 2
        });
    });

    test('Removing all work from ScopeBurndown without doing anything', function() {
        var data = {
            sprints: this.makeSprints(2),
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                2: this.addAndEstimate('TEST-2', 3),
                3: this.addAndEstimate('TEST-3', 3),
                4: this.addAndEstimate('TEST-4', 3),
                100: [this.removeFromScopeBurndown('TEST-1')],
                101: [this.removeFromScopeBurndown('TEST-2')],
                200: [this.removeFromScopeBurndown('TEST-3')],
                201: [this.removeFromScopeBurndown('TEST-4')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 0,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 0,
                isForecast: false,
                isActive: false,
                issues: []
            }
        ]);
        deepEqual(result.snapshot, {
            workRemaining: 0,
            estimatedIssueCount: 0,
            issueCount: 0,
            workCompleted: 0,
            estimatableIssueCount: 0
        });
    });

    test('When last sprint is currently active, no remaining work sprint is returned', function() {
        var sprints = this.makeSprints(2);
        sprints[1].state = 'ACTIVE';
        var data = {
            sprints: sprints,
            changes: {
                1: this.addAndEstimate('TEST-1', 1),
                2: this.addAndEstimate('TEST-2', 2),
                100: [this.moveToDone('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 3,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 2,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            },
            {
                sprintId: 2,
                sprintName: 'Sprint 2',
                baseline: 0,
                workAtStart: 2,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 2,
                startTime: 200,
                endTime: 300,
                isForecast: false,
                isActive: true,
                issues: []
            }
        ]);
    });

    test('Changing an estimate is retroactive', function() {
        var data = {
            sprints: this.makeSprints(1),
            changes: {
                // Estimated as 1
                1: this.addAndEstimate('TEST-1', 1),
                100: [this.moveToDone('TEST-1')],
                // Changed to 3 after it is done
                200: [this.changeEstimate('TEST-1', 3)]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 3,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 3,
                workRemaining: 0,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            }
        ]);
        deepEqual(result.snapshot, {
            workRemaining: 0,
            estimatedIssueCount: 1,
            issueCount: 1,
            workCompleted: 3,
            estimatableIssueCount: 1
        });
    });

    test('Issue that is moved back to Not Done removes previous work completed', function() {
        var data = {
            sprints: this.makeSprints(1),
            changes: {
                1: this.addAndEstimate('TEST-1', 1),
                2: [this.moveToDone('TEST-1')],
                100: [this.moveToNotDone('TEST-1')],
                101: [this.moveToDone('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 1,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 0,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            }
        ]);
        deepEqual(result.snapshot, {
            workRemaining: 0,
            estimatedIssueCount: 1,
            issueCount: 1,
            workCompleted: 1,
            estimatableIssueCount: 1
        });
    });

    test('More work is done than what was remaining in previous sprint or original estimate', function() {
        var data = {
            sprints: this.makeSprints(2),
            changes: {
                100: this.addAndEstimate('TEST-1', 5),
                101: [this.moveToDone('TEST-1')],
                200: this.addAndEstimate('TEST-2', 8),
                201: this.addAndEstimate('TEST-3', 13),
                202: [this.moveToDone('TEST-2')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 0,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 0,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: -5,
                workAtStart: 0,
                workAdded: 5,
                workRemoved: 0,
                workCompleted: 5,
                workRemaining: 0,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            },
            {
                sprintId: 2,
                sprintName: 'Sprint 2',
                baseline: -5 - 8 - 13,
                workAtStart: 0,
                workAdded: 8 + 13,
                workRemoved: 0,
                workCompleted: 8,
                workRemaining: 13,
                startTime: 200,
                endTime: 300,
                isForecast: false,
                isActive: false,
                issues: ['TEST-2']
            }
        ]);
        deepEqual(result.snapshot, {
            workRemaining: 13,
            estimatedIssueCount: 3,
            issueCount: 3,
            workCompleted: 13,
            estimatableIssueCount: 3
        });
    });

    test('Sprints before work is started on a ScopeBurndown are not included', function() {
        var data = {
            sprints: this.makeSprints(9),
            changes: {
                100: this.addAndEstimate('TEST-1', 1),
                200: this.addAndEstimate('TEST-2', 2),
                300: this.addAndEstimate('TEST-3', 3),
                899: [this.moveToDone('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 6,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 6,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 8,
                sprintName: 'Sprint 8',
                baseline: 0,
                workAtStart: 6,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 5,
                startTime: 800,
                endTime: 900,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            },
            {
                sprintId: 9,
                sprintName: 'Sprint 9',
                baseline: 0,
                workAtStart: 5,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 5,
                startTime: 900,
                endTime: 1000,
                isForecast: false,
                isActive: false,
                issues: []
            }
        ]);
    });

    test('Sprints after work is completed on a ScopeBurndown are not included', function() {
        var data = {
            sprints: this.makeSprints(9),
            changes: {
                100: this.addAndEstimate('TEST-1', 1),
                899: [this.moveToDone('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 1,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 8,
                sprintName: 'Sprint 8',
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 0,
                startTime: 800,
                endTime: 900,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            }
        ]);
    });

    test('Issue added to ScopeBurndown when already done does affect scope', function() {
        var data = {
            sprints: this.makeSprints(2),
            changes: {
                100: [this.moveToNotDone('TEST-1'), this.changeEstimate('TEST-1', 1)],
                101: [this.moveToDone('TEST-1'), this.addToScopeBurndown('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 0,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 0,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                "baseline": -1,
                "endTime": 200,
                "isActive": false,
                "isForecast": false,
                "issues": [
                    "TEST-1"
                ],
                "sprintId": 1,
                "sprintName": "Sprint 1",
                "startTime": 100,
                "workAdded": 1,
                "workAtStart": 0,
                "workCompleted": 1,
                "workRemaining": 0,
                "workRemoved": 0
            }
        ]);
    });

    test('Issue added to ScopeBurndown and remove from ScopeBurndown multiple times does not affect scope multiple times' , function() {
        var data = {
            sprints: this.makeSprints(2),
            changes: {
                1: this.addAndEstimate('TEST-1', 1),
                2: this.addAndEstimate('TEST-2', 2),
                100: [this.moveToDone('TEST-1'),
                    this.removeFromScopeBurndown('TEST-2'),
                    this.addToScopeBurndown('TEST-2'),
                    this.removeFromScopeBurndown('TEST-2'),
                    this.addToScopeBurndown('TEST-2'),
                    this.removeFromScopeBurndown('TEST-2')
                ]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };

        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 3,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                "baseline": 2,
                "endTime": 200,
                "isActive": false,
                "isForecast": false,
                "issues": [
                    "TEST-1"
                ],
                "sprintId": 1,
                "sprintName": "Sprint 1",
                "startTime": 100,
                "workAdded": 0,
                "workAtStart": 3,
                "workCompleted": 1,
                "workRemaining": 0,
                "workRemoved": 2
            }
        ]);
    });

    test('Issue added to ScopeBurndown and move to done after that removed from ScopeBurndown does not affect scope', function() {
        var data = {
            sprints: this.makeSprints(2),
            changes: {
                99: this.addAndEstimate('TEST-1', 1),
                200: this.addAndEstimate('TEST-2', 2),
                201: [this.moveToDone('TEST-2'), this.removeFromScopeBurndown('TEST-2')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 1,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                "baseline": 0,
                "endTime": 300,
                "isActive": false,
                "isForecast": false,
                "issues": [],
                "sprintId": 2,
                "sprintName": "Sprint 2",
                "startTime": 200,
                "workAdded": 0,
                "workAtStart": 1,
                "workCompleted": 0,
                "workRemaining": 1,
                "workRemoved": 0
            }
        ]);
    });

    test('Issue removed from ScopeBurndown when already done does not affect scope', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                100: this.addAndEstimate('TEST-1', 1),
                101: this.addAndEstimate('TEST-2', 2),
                200: [this.moveToDone('TEST-1')],
                300: [this.removeFromScopeBurndown('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 3,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 2,
                sprintName: 'Sprint 2',
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 2,
                startTime: 200,
                endTime: 300,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            },
            {
                sprintId: 3,
                sprintName: 'Sprint 3',
                baseline: 0,
                workAtStart: 2,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 2,
                startTime: 300,
                endTime: 400,
                isForecast: false,
                isActive: false,
                issues: []
            }
        ]);
    });

    test('Issue removed from ScopeBurndown when already done, but is later not done, and add it back to ScopeBurndown does affect scope', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                100: this.addAndEstimate('TEST-1', 1),
                101: this.addAndEstimate('TEST-2', 2),
                200: [this.moveToDone('TEST-1')],
                300: [this.removeFromScopeBurndown('TEST-1')],
                400: [this.moveToNotDone('TEST-1')],
                401: [this.addToScopeBurndown('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 3,
                isForecast: false,
                isActive: false,
                issues: []
            }
        ]);
    });

    test('Issue removed from ScopeBurndown is ignored even if it is completed later', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                100: this.addAndEstimate('TEST-1', 1),
                101: this.addAndEstimate('TEST-2', 2),
                200: [this.moveToDone('TEST-1')],
                201: [this.removeFromScopeBurndown('TEST-2')],
                300: [this.moveToDone('TEST-2')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 3,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 2,
                sprintName: 'Sprint 2',
                baseline: 2,
                workAtStart: 3,
                workAdded: 0,
                workRemoved: 2,
                workCompleted: 1,
                workRemaining: 0,
                startTime: 200,
                endTime: 300,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            }
        ]);
    });

    test('Issues with estimates removed should be reflected in snapshot', function () {
        var data = {
            sprints: [],
            changes: {
                // Estimated as 1
                1: this.addAndEstimate('TEST-1', 1),
                // Un-estimate the issue
                100: [this.changeEstimate('TEST-1', undefined)]
            },
            estimatableIssueKeys: ['TEST-1']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.snapshot, {
            workRemaining: 0,
            estimatedIssueCount: 0,
            issueCount: 1,
            workCompleted: 0,
            estimatableIssueCount: 1
        });
    });

    test('Average velocity is rounded for forecast', function () {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                2: this.addAndEstimate('TEST-2', 4),
                3: this.addAndEstimate('TEST-3', 4),
                4: this.addAndEstimate('TEST-4', 3),
                5: this.addAndEstimate('TEST-5', 3),
                6: this.addAndEstimate('TEST-6', 3),
                7: this.addAndEstimate('TEST-7', 3),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4', 'TEST-5', 'TEST-6', 'TEST-7']
        };
        var result = this.service.getScopeBySprintData(data);
        strictEqual(result.forecast.velocity, 4, 'Velocity is rounded');
    });

    test('No forecast is given if velocity rounds to 0', function () {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                1: this.addAndEstimate('TEST-1', 0),
                2: this.addAndEstimate('TEST-2', 0),
                3: this.addAndEstimate('TEST-3', 1),
                4: this.addAndEstimate('TEST-4', 3),
                5: this.addAndEstimate('TEST-5', 3),
                6: this.addAndEstimate('TEST-6', 3),
                7: this.addAndEstimate('TEST-7', 3),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4', 'TEST-5', 'TEST-6', 'TEST-7']
        };
        var result = this.service.getScopeBySprintData(data);
        ok(result.forecast.forecastError, 'No forecast');
    });

    test('No forecast is given if there are fewer than 3 completed sprints', function () {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                1: this.addAndEstimate('TEST-1', 0),
                2: this.addAndEstimate('TEST-2', 0),
                3: this.addAndEstimate('TEST-3', 1),
                4: this.addAndEstimate('TEST-4', 3),
                5: this.addAndEstimate('TEST-5', 3),
                6: this.addAndEstimate('TEST-6', 3),
                7: this.addAndEstimate('TEST-7', 3),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4', 'TEST-5', 'TEST-6', 'TEST-7']
        };
        data.sprints[2].state = 'ACTIVE';
        var result = this.service.getScopeBySprintData(data);
        ok(result.forecast.forecastError, 'No forecast');
    });

    test('Active sprint is counted differently if work completed is less than average velocity', function () {
        var data = {
            sprints: this.makeSprints(4),
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                2: this.addAndEstimate('TEST-2', 3),
                3: this.addAndEstimate('TEST-3', 3),
                4: this.addAndEstimate('TEST-4', 2),
                5: this.addAndEstimate('TEST-5', 3),
                6: this.addAndEstimate('TEST-6', 3),
                7: this.addAndEstimate('TEST-7', 3),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')],
                400: [this.moveToDone('TEST-4')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4', 'TEST-5', 'TEST-6', 'TEST-7']
        };
        data.sprints[3].state = 'ACTIVE';
        var result = this.service.getScopeBySprintData(data);
        var forecast = result.forecast;
        strictEqual(_.last(result.sprints).workRemaining, 9, 'Last sprint\'s work remaining is actual');
        strictEqual(forecast.velocity, 3, 'Velocity');
        strictEqual(forecast.workRemaining, 8, 'Forecast\'s work remaining assumes work done in active sprint is equal to velocity');
        strictEqual(forecast.isLastSprintForecast, true);
    });

    test('Active sprint is used in velocity calculation if work completed is greater than average velocity', function () {
        var data = {
            sprints: this.makeSprints(4),
            changes: {
                1: this.addAndEstimate('TEST-1', 1),
                2: this.addAndEstimate('TEST-2', 3),
                3: this.addAndEstimate('TEST-3', 5),
                4: this.addAndEstimate('TEST-4', 5),
                5: this.addAndEstimate('TEST-5', 3),
                6: this.addAndEstimate('TEST-6', 3),
                7: this.addAndEstimate('TEST-7', 1),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')],
                400: [this.moveToDone('TEST-4')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4', 'TEST-5', 'TEST-6', 'TEST-7']
        };
        data.sprints[3].state = 'ACTIVE';
        var result = this.service.getScopeBySprintData(data);
        var forecast = result.forecast;
        strictEqual(forecast.velocity, 4, 'Velocity is the average of last 3 sprints incl. the active one');
        strictEqual(forecast.isLastSprintForecast, false);
        strictEqual(_.last(result.sprints).workCompleted, 5, 'Work completed in active sprint is actual');
        strictEqual(forecast.sprintsRemaining, 2, 'Only 2 forecast sprints given the new velocity');
    });

    test('Correct work completed when average velocity is greater than remaining work in active sprint', function () {
        var data = {
            sprints: this.makeSprints(4),
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                2: this.addAndEstimate('TEST-2', 3),
                3: this.addAndEstimate('TEST-3', 3),
                4: this.addAndEstimate('TEST-4', 1),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4']
        };
        data.sprints[3].state = 'ACTIVE';
        var result = this.service.getScopeBySprintData(data);
        var activeSprint = _.last(result.sprints);
        strictEqual(activeSprint.workCompleted, 0);
        strictEqual(activeSprint.workRemaining, 1);
        strictEqual(result.forecast.sprintsRemaining, 0, 'Even though no work is done in active sprint yet, we expect this to be the last');
    });

    test('Work remaining in snapshot is based on actual work done in the active sprint', function() {
        var data = {
            sprints: this.makeSprints(4),
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                2: this.addAndEstimate('TEST-2', 3),
                3: this.addAndEstimate('TEST-3', 3),
                4: this.addAndEstimate('TEST-4', 1),
                5: this.addAndEstimate('TEST-5', 3),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')],
                400: [this.moveToDone('TEST-4')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4', 'TEST-5']
        };
        data.sprints[3].state = 'ACTIVE';
        var result = this.service.getScopeBySprintData(data);
        strictEqual(result.snapshot.workRemaining, 3, 'Work remaining is correct');
    });

    test('Zero baseline', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                1: this.addAndEstimate('TEST-1', 3),
                100: [this.moveToDone('TEST-1')],
                101: this.addAndEstimate('TEST-2', 3),
                200: [this.moveToDone('TEST-2')],
                201: this.addAndEstimate('TEST-3', 3),
                300: [this.moveToDone('TEST-3')],
                301: this.addAndEstimate('TEST-4', 3)
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-4']
        };
        var result = this.service.getScopeBySprintData(data, true);
        strictEqual(result.sprints.length, 4, '1 original estimate, 3 sprints');
        strictEqual(_.all(result.sprints, function(sprint) {
            return sprint.baseline === 0;
        }), true, 'All sprints have a 0 baseline');
    });

    test('Test duplicate change events do not create extra scope', function() {
        var data = {
            sprints: this.makeSprints(1),
            changes: {
                1: this.addAndEstimate('TEST-1', 1),
                100: [this.addToScopeBurndown('TEST-1')],
                101: [this.moveToDone('TEST-1')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 1,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 0,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            }
        ]);
    });

    test('Test reopened issues are added correctly to the scope', function() {
        var data = {
            sprints: this.makeSprints(1),
            changes: {
                1: this.addAndEstimate('TEST-1', 1),
                100: [this.moveToDone('TEST-1')],
                101: [this.changeEstimate('TEST-2', 1)],
                102: [this.moveToDone('TEST-2')],
                103: [this.moveToNotDone('TEST-2')],
                104: [this.addToScopeBurndown('TEST-2')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 0,
                workRemaining: 1,
                isForecast: false,
                isActive: false,
                issues: []
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: -1,
                workAtStart: 1,
                workAdded: 1,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 1,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-1']
            }
        ]);
    });

    test('Test work done before first sprint', function() {
        var data = {
            sprints: this.makeSprints(2),
            changes: {
                1: this.addAndEstimate('TEST-1', 1),
                2: [this.moveToDone('TEST-1')],
                100: this.addAndEstimate('TEST-2', 1),
                101: [this.moveToDone('TEST-2')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints, [
            {
                sprintId: 'original',
                sprintName: this.getOriginalEstimateSprintName(),
                baseline: 0,
                workAtStart: 1,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 0,
                isForecast: false,
                isActive: false,
                issues: [ 'TEST-1' ]
            },
            {
                sprintId: 1,
                sprintName: 'Sprint 1',
                baseline: -1,
                workAtStart: 0,
                workAdded: 1,
                workRemoved: 0,
                workCompleted: 1,
                workRemaining: 0,
                startTime: 100,
                endTime: 200,
                isForecast: false,
                isActive: false,
                issues: ['TEST-2']
            }
        ]);
    });

    test('Test estimated issues calculation', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                // Issues are created and estimated
                1: [this.moveToNotDone('TEST-1'), this.moveToNotDone('TEST-2'), this.moveToNotDone('TEST-3')],
                3: [this.changeEstimate('TEST-2', 3)],
                // Issues are added to the ScopeBurndown
                10: [this.addToScopeBurndown('TEST-1'), this.addToScopeBurndown('TEST-2'), this.addToScopeBurndown('TEST-3')],
                // First sprint
                100: [this.moveToDone('TEST-1')],
                // Second sprint
                200: [this.moveToDone('TEST-2')],
                299: [this.moveToNotDone('TEST-4'), this.addToScopeBurndown('TEST-4'), this.changeEstimate('TEST-4', 8)],
                // Third sprint
                300: [this.moveToDone('TEST-3')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-2', 'TEST-4']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.snapshot, {
            workRemaining: 8,
            estimatedIssueCount: 2,
            estimatableIssueCount: 2,
            issueCount: 4,
            workCompleted: 3
        });
    });

    test('Estimatable count should only take into account issues in the version, e.g. have not been removed', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                // Issues are created and estimated
                1: [this.moveToNotDone('TEST-1'), this.moveToNotDone('TEST-2'), this.moveToNotDone('TEST-3')],
                3: [this.changeEstimate('TEST-2', 3)],
                // Issues are added to the ScopeBurndown
                10: [this.addToScopeBurndown('TEST-1'), this.addToScopeBurndown('TEST-2'), this.addToScopeBurndown('TEST-3')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-5']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.snapshot, {
            workRemaining: 3,
            estimatedIssueCount: 1,
            estimatableIssueCount: 3,
            issueCount: 3,
            workCompleted: 0
        });
    });

    test('Test unestimated issue count in epic', function() {
        var data = {
            sprints: this.makeSprints(1),
            changes: {
                // Issues are created and estimated
                1: [this.moveToNotDone('TEST-1'), this.changeEstimate('TEST-1', 8), this.moveToNotDone('TEST-2'), this.addToScopeBurndown('TEST-1'), this.addToScopeBurndown('TEST-2')],
                // First sprint
                100: [this.moveToDone('TEST-1')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2']
        };
        var result = this.service.getScopeBySprintData(data);
        strictEqual(result.forecast.remainingEstimatableIssueCount, 1, 'There should only be 1 estimable issue left in the scope');

    });

    test('Test issues that are not estimatable are not counted', function() {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                // Issues are created and estimated
                1: this.addAndEstimate('TEST-1', 10),
                2: this.addAndEstimate('TEST-2', 20),
                3: this.addAndEstimate('TEST-3', 30),
                4: this.addAndEstimate('TEST-4', 40),
                5: this.addAndEstimate('TEST-5', 50),
                6: this.addAndEstimate('TEST-6', 60),
                // First sprint
                100: [this.moveToDone('TEST-1'), this.moveToDone('TEST-2')],
                // Second sprint
                200: [this.moveToDone('TEST-4')],
                // Third sprint
                300: [this.moveToDone('TEST-5')]
            },
            labels: { originalEstimate: this.getOriginalEstimateSprintName() },
            estimatableIssueKeys: ['TEST-1', 'TEST-3', 'TEST-5']
        };
        var result = this.service.getScopeBySprintData(data);
        deepEqual(result.sprints[0], {
            sprintId: 'original',
            sprintName: this.getOriginalEstimateSprintName(),
            baseline: 0,
            workAtStart: 90,
            workAdded: 0,
            workRemoved: 0,
            workCompleted: 0,
            workRemaining: 90,
            isForecast: false,
            isActive: false,
            issues: []
        }, 'the work remaining should be the total of TEST-1, TEST-3 and TEST-5\'s estimates.');
        deepEqual(result.sprints[1], {
            sprintId: 1,
            sprintName: 'Sprint 1',
            baseline: 0,
            workAtStart: 90,
            workAdded: 0,
            workRemoved: 0,
            workCompleted: 10,
            workRemaining: 80,
            startTime: 100,
            endTime: 200,
            isForecast: false,
            isActive: false,
            issues: ['TEST-1', 'TEST-2']
        });
        deepEqual(result.sprints[2], {
            sprintId: 2,
            sprintName: 'Sprint 2',
            baseline: 0,
            workAtStart: 80,
            workAdded: 0,
            workRemoved: 0,
            workCompleted: 0,
            workRemaining: 80,
            startTime: 200,
            endTime: 300,
            isForecast: false,
            isActive: false,
            issues: ['TEST-4']
        });
        deepEqual(result.sprints[3], {
            sprintId: 3,
            sprintName: 'Sprint 3',
            baseline: 0,
            workAtStart: 80,
            workAdded: 0,
            workRemoved: 0,
            workCompleted: 50,
            workRemaining: 30,
            startTime: 300,
            endTime: 400,
            isForecast: false,
            isActive: false,
            issues: ['TEST-5']
        });
        deepEqual(result.forecast, {
            sprintsRemaining: 2,
            workRemaining: 30,
            velocity: 20,
            baseline: 0,
            isLastSprintForecast: false
        });
        strictEqual(result.snapshot.workCompleted, 60);
        strictEqual(result.snapshot.workRemaining, 30);
        strictEqual(result.snapshot.issueCount, 6);
        strictEqual(result.snapshot.estimatableIssueCount, 3);
        strictEqual(result.snapshot.estimatedIssueCount, 3);
    });

    test('Test unestimated issue count in epic when remaining issues are not estimatable', function() {
        var data = {
            sprints: this.makeSprints(1),
            changes: {
                // Issues are created and estimated
                1: [this.moveToNotDone('TEST-1'), this.changeEstimate('TEST-1', 8), this.moveToNotDone('TEST-2'), this.addToScopeBurndown('TEST-1'), this.addToScopeBurndown('TEST-2')],
                // First sprint
                100: [this.moveToDone('TEST-1')]
            },
            estimatableIssueKeys: ['TEST-1']
        };
        var result = this.service.getScopeBySprintData(data);
        strictEqual(result.forecast.remainingEstimatableIssueCount, 0, 'Only remaining issue is not estimatable');

    });

    test('Test values are accurate if only remaining issues are not estimatable', function () {
        var data = {
            sprints: this.makeSprints(3),
            changes: {
                1: this.addAndEstimate('TEST-1', 10),
                2: this.addAndEstimate('TEST-2', 20),
                3: this.addAndEstimate('TEST-3', 30),
                4: this.addAndEstimate('TEST-4', 40),
                5: this.addAndEstimate('TEST-5', 50),
                6: this.addAndEstimate('TEST-6', 60),
                7: this.addAndEstimate('TEST-7', 70),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3']
        };
        var result = this.service.getScopeBySprintData(data);
        var forecast = result.forecast;
        strictEqual(_.last(result.sprints).workRemaining, 0, 'Last sprint\'s work remaining is actual');
        deepEqual(result.snapshot, {
            workRemaining: 0,
            workCompleted: 60,
            estimatedIssueCount: 3,
            estimatableIssueCount: 3,
            issueCount: 7
        }, 'All estimatable work is completed');
        strictEqual(forecast.remainingEstimatableIssueCount, 0, 'No estimatable issue count when no work remaining');
        strictEqual(typeof forecast.workRemaining, 'undefined', 'No work remaining when no remaining estimatable issues');
        strictEqual(typeof forecast.velocity, 'undefined', 'No velocity when no remaining estimatable issues');
        strictEqual(typeof forecast.isLastSprintForecast, 'undefined', 'No last sprint forecast when no remaining estimatable issues');
    });

    test('Test values are accurate if only remaining issues are unestimated or not estimatable', function () {
        var data = {
            sprints: this.makeSprints(4),
            changes: {
                1: this.addAndEstimate('TEST-1', 10),
                2: this.addAndEstimate('TEST-2', 20),
                3: this.addAndEstimate('TEST-3', 30),
                4: this.addAndEstimate('TEST-4', 40),
                5: this.addAndEstimate('TEST-5', 50),
                6: this.addAndEstimate('TEST-6'),
                7: this.addAndEstimate('TEST-7'),
                100: [this.moveToDone('TEST-1')],
                200: [this.moveToDone('TEST-2')],
                300: [this.moveToDone('TEST-3')],
                400: [this.moveToDone('TEST-4')]
            },
            estimatableIssueKeys: ['TEST-1', 'TEST-2', 'TEST-3', 'TEST-6', 'TEST-7']
        };
        data.sprints[3].state = 'ACTIVE';
        var result = this.service.getScopeBySprintData(data);
        var forecast = result.forecast;
        strictEqual(_.last(result.sprints).workRemaining, 0, 'Last sprint\'s work remaining is actual');
        deepEqual(result.snapshot, {
            workRemaining: 0,
            workCompleted: 60,
            estimatedIssueCount: 3,
            estimatableIssueCount: 5,
            issueCount: 7
        }, 'All estimatable work except TEST-6 and TEST-7 is completed');
        strictEqual(forecast.remainingEstimatableIssueCount, 2, 'TEST-6 and TEST-7 are estimatable but unestimated');
        strictEqual(typeof forecast.workRemaining, 'undefined', 'No work remaining since TEST-6 and TEST-7 are unestimated');
        strictEqual(typeof forecast.velocity, 'undefined', 'No velocity when no work remaining');
        strictEqual(typeof forecast.isLastSprintForecast, 'undefined', 'No last sprint forecast when no work remaining');
    });
});
