/* global GH */
/**
 * Version Report View
 * @module jira-agile/rapid/ui/chart/version-report-view
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/chart/change-event
 * @requires module:jira-agile/rapid/ui/chart/chart
 * @requires module:jira-agile/rapid/ui/chart/chart-colors
 * @requires module:jira-agile/rapid/ui/chart/chart-controller
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 * @requires module:jira-agile/rapid/ui/chart/non-working-days-util
 * @requires module:jira-agile/rapid/ui/chart/version-report
 * @requires module:jira-agile/rapid/ui/chart/version-report-controller
 */
define('jira-agile/rapid/ui/chart/version-report-view', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var $ = require('jquery');
    var _ = require('underscore');
    var ChangeEvent = require('jira-agile/rapid/ui/chart/change-event');
    var Chart = require('jira-agile/rapid/ui/chart/chart');
    var ChartColors = require('jira-agile/rapid/ui/chart/chart-colors');
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var NonWorkingDaysUtils = require('jira-agile/rapid/ui/chart/non-working-days-util');
    var VersionReport = require('jira-agile/rapid/ui/chart/version-report');
    var VersionReportController;

    // GLOBALS... FIX ME
    var ChartTooltip = GH.ChartTooltip;
    var ChartUtils = GH.ChartUtils;
    var FlotChartUtils = GH.FlotChartUtils;
    var tpl = GH.tpl;

    // Resolve circular dependency
    GlobalEvents.on('pre-initialization', function () {
        VersionReportController = require('jira-agile/rapid/ui/chart/version-report-controller');
    });

    var VersionReportView = {};

    VersionReportView.unmodifiedSeries = undefined;

    VersionReportView.series = undefined;

    VersionReportView.setUserTimeZoneLabel = function (userTimeZoneLabel) {
        VersionReportView.userTimeZoneLabel = userTimeZoneLabel;
    };

    VersionReportView.renderChartLoading = function () {
        // render the skeleton for the view
        ChartView.getChartContentElem(true).html(tpl.versionreport.renderReportBody());

        // show the spinner
        ChartView.showSpinner();
    };

    VersionReportView.renderVersionPicker = function () {
        VersionReportController.versionPicker.render(ChartView.getChartSelector(true));
    };

    VersionReportView.renderChart = function () {
        VersionReportView.toggleChartVisibility(true);

        if (VersionReport.model.hasIssues() && !VersionReport.model.notStarted()) {
            var plotHolder = ChartView.getChartView(true);
            var series = VersionReport.model.getVersionSeries();

            if (VersionReportView.wallboardMode) {
                var seriesData = _.findWhere(series, { id: 'markings' });
                seriesData.color = ChartColors.nonWorkingDaysWallboard;
            }

            var options = VersionReportView.getChartOptions();

            VersionReportView.unmodifiedSeries = series;
            VersionReportView.series = series;

            if (!VersionReportController.nonWorkingDaysShown) {
                var startDate = VersionReport.model.getXAxisStart();
                series = NonWorkingDaysUtils.transformSeriesExcludingNonWorkingDays(series, startDate);

                VersionReportView.series = series;

                if (_.findWhere(series, { id: 'prediction' })) {
                    // recalculate the bottom fill rectangle for the prediction
                    var pessimisticSeries = _.findWhere(series, { id: 'pessimistic' });
                    var standardSeries = _.findWhere(series, { id: 'prediction' });
                    var firstPessimisticPoint = _.first(pessimisticSeries.data);
                    var lastPessimisticPoint = _.last(pessimisticSeries.data);
                    var lastStandardPoint = _.last(standardSeries.data);
                    var standardProjectionOverPessimistic = ChartUtils.calculateYProjection(firstPessimisticPoint, lastPessimisticPoint, lastStandardPoint);
                    _.findWhere(series, { id: 'pessimisticFillerBottom' }).data[0][1] = standardProjectionOverPessimistic;
                }

                if (_.findWhere(series, { id: 'scope' })) {
                    var lastScope = _.last(_.findWhere(series, { id: 'scope' }).data);
                    lastScope[0] = options.xaxis.max;
                }

                options.xaxis.tickFormatter = function (originalVal, xAxis) {
                    // calculate the date taking non working days into account
                    var date = NonWorkingDaysUtils.transformDateIfInNonWorkingDays(originalVal, startDate);

                    var format = '%b %d';

                    if (xAxis.tickSize[1] === 'hour') {
                        format += ' %h:%M';
                    } else if (xAxis.tickSize[1] === 'minute') {
                        format = ' %h:%M';
                    } else if (xAxis.tickSize[1] === 'second') {
                        format = ' %h:%M:%S';
                    }
                    return $.plot.formatDate(date, format);
                };
            }

            var plot = Chart.draw(VersionReportController.id, plotHolder, series, options);
            var secondYAxisLabel = false;
            if (VersionReport.model.getYAxes().length > 1) {
                secondYAxisLabel = AJS.I18n.getText('gh.rapid.charts.progress.unestimated.percent.axis');
            }

            VersionReportView.renderLabelsForNowAndRelease(plot);
            VersionReportView.setLegendOpacity();

            FlotChartUtils.setAndAlignAxisLabels(AJS.I18n.getText('gh.rapid.charts.progress.xaxis.label', VersionReportView.userTimeZoneLabel), VersionReport.model.getStatistic().name, secondYAxisLabel);

            //        VersionReportView.addNonWorkingDaysToLegend();

            // register hooks to handle dot painting
            plot.hooks.drawSeries.push(VersionReportView.gatherPoints);
            plot.hooks.draw.push(FlotChartUtils.deferredRenderPoints);

            // ensure legend and labels are redrawn after resize
            plot.hooks.draw.push(VersionReportView.renderLabelsForNowAndRelease);
            plot.hooks.draw.push(VersionReportView.setLegendOpacity);
            //        plot.hooks.draw.push(VersionReportView.addNonWorkingDaysToLegend);

            VersionReportView.bindEvents(plotHolder);
        }
    };

    VersionReportView.getChartOptions = function () {
        var model = VersionReport.model;
        var options = ChartUtils.getChartOptions(model);

        if (!VersionReportController.nonWorkingDaysShown) {
            options.xaxis.max = NonWorkingDaysUtils.deductNonWorkingDays(options.xaxis.max, options.xaxis.min);
        }

        if (VersionReportView.gadgetMode) {
            options.legend.position = 'nw';
            options.legend.container = null;
            options.legend.backgroundOpacity = 0.5;
            options.legend.margin = [6, 12];
            $('#ghx-chart-legend').hide();
        }

        if (VersionReportView.wallboardMode) {
            // wallboard
            options.grid = {
                backgroundColor: ChartColors.wallboard.backgroundColor,
                borderWidth: 1, // in pixels
                borderColor: ChartColors.wallboard.borderColor, // set if different from the grid color
                color: ChartColors.wallboard.textColor, // primary color used for outline and labels
                noGridBottomOffsetDelta: 0,
                tickColor: ChartColors.wallboard.tickColor, // color used for the ticks
                markings: VersionReportView.getMarkingsForGadget
            };
        } else {
            options.grid.markings = function (axes) {
                var markings = ChartController.getNonWorkingBlocks();
                var now = model.getNow();
                var nowLineMarkings = [];
                var releaseDateMarkings = [];
                if (!model.getVersion().released && axes.xmax > now) {
                    if (!VersionReportController.nonWorkingDaysShown) {
                        now = NonWorkingDaysUtils.deductNonWorkingDays(now, options.xaxis.min);
                    }
                    // mark 'Now'
                    // divide the y-axis into 50 dashes, each with a size:gap ratio 4:1, but without a gap at the top or bottom
                    var tickSize = axes.ymax / 198;
                    var dashSize = tickSize * 4;
                    nowLineMarkings = VersionReportView.drawDashedMarkings(now, dashSize, tickSize, 70);

                    // mark release date
                    var releaseDate = model.getVersion().releaseDate;
                    if (!VersionReportController.nonWorkingDaysShown) {
                        releaseDate = NonWorkingDaysUtils.deductNonWorkingDays(releaseDate, options.xaxis.min);
                    }
                    if (!_.isUndefined(releaseDate)) {
                        dashSize = 2 * dashSize;
                        tickSize = 2 * tickSize;
                        releaseDateMarkings = VersionReportView.drawDashedMarkings(releaseDate, dashSize, tickSize, 20, ChartColors.releaseDate);
                    }
                }
                if (VersionReportController.nonWorkingDaysShown) {
                    return markings.concat(nowLineMarkings, releaseDateMarkings);
                } else {
                    // only return the now and release date dashed lines
                    return nowLineMarkings.concat(releaseDateMarkings);
                }
            };
        }

        return options;
    };

    VersionReportView.getMarkingsForGadget = function () {
        var wallboardStripes = ChartColors.nonWorkingDaysWallboard;
        var markings = ChartController.getNonWorkingBlocks();
        return _.map(markings, function (marking) {
            marking.color = wallboardStripes;
            return marking;
        });
    };

    VersionReportView.drawDashedMarkings = function (xAxisVal, dashSize, tickSize, max, color) {
        var dashedLineMarkings = [];
        for (var i = 0; i < max; i++) {
            var dashStart = i * (dashSize + tickSize);
            dashedLineMarkings.push({
                color: color || ChartColors.dashedMarking,
                lineWidth: 1,
                xaxis: {
                    from: xAxisVal,
                    to: xAxisVal

                },
                yaxis: {
                    from: dashStart,
                    to: dashStart + dashSize
                }
            });
        }
        return dashedLineMarkings;
    };

    VersionReportView.setLegendOpacity = function () {
        // set legend opacity
        var legendTableCells = $('#ghx-chart-legend').find('.legendColorBox');
        $(legendTableCells[0]).find('div').find('div').css('opacity', 0.3);
        $(legendTableCells[1]).find('div').find('div').css('opacity', 0.8);
    };

    VersionReportView.renderIssuesLink = function () {

        var version = VersionReport.model.getVersion();
        var versionName = version.name;
        var versionNameUse = true;

        if (versionName.length > 60) {
            versionNameUse = false;
        }

        $('#ghx-report-controls').empty().append(tpl.versionreport.renderViewIssuesLink({
            projectKey: VersionReport.model.getProjectKey(),
            version: version,
            nameUse: versionNameUse
        })).show();
    };

    VersionReportView.renderLabelsForNowAndRelease = function (plot) {
        var xaxis = plot.getXAxes()[0];
        var yaxis = plot.getYAxes()[0];

        // if the chart is redrawn while data is loading (e.g. after changing dropdown)
        // do not redraw the labels because we will get a js error
        if (VersionReport.model.chartData) {
            var now = VersionReport.model.getNow();
            if (!VersionReportController.nonWorkingDaysShown) {
                now = NonWorkingDaysUtils.deductNonWorkingDays(now, VersionReport.model.getXAxisStart());
            }
            var version = VersionReport.model.getVersion();
            var releaseDate = version.releaseDate;
            if (!VersionReportController.nonWorkingDaysShown) {
                releaseDate = NonWorkingDaysUtils.deductNonWorkingDays(releaseDate, VersionReport.model.getXAxisStart());
            }
            var nowLabel;

            var topXaxisLabels = $('#ghx-axis-x-top');
            topXaxisLabels.empty();

            if (now < xaxis.max) {
                nowLabel = $(tpl.versionreport.renderXaxisTopLabel({
                    text: AJS.I18n.getText('gh.chart.today'),
                    left: plot.pointOffset({ x: now, y: yaxis.max }).left
                }));
                topXaxisLabels.append(nowLabel);
            }

            if (!_.isUndefined(releaseDate) && !version.released && releaseDate > xaxis.min && releaseDate < xaxis.max) {

                var left = plot.pointOffset({ x: releaseDate, y: yaxis.max }).left;
                var releaseDateLabel = $(tpl.versionreport.renderXaxisTopLabel({
                    text: AJS.I18n.getText('gh.chart.release.date'),
                    date: ChartUtils.renderUTCMillisAsDateWithoutTime(VersionReport.model.getVersion().releaseDate),
                    left: left
                }));
                topXaxisLabels.append(releaseDateLabel);
                if (!_.isUndefined(nowLabel)) {
                    // move the release date label if they overlap
                    if (VersionReportView.labelsAreOverlapping(nowLabel, releaseDateLabel)) {
                        releaseDateLabel.addClass('ghx-raised');
                    }
                }
            }
        }
    };

    VersionReportView.labelsAreOverlapping = function (firstLabel, secondLabel) {

        var firstLabelLeft = firstLabel.position().left;
        var firstLabelRight = firstLabelLeft + firstLabel.width();
        var secondLabelLeft = secondLabel.position().left;
        var secondLabelRight = secondLabelLeft + secondLabel.width();

        // second label starts inside the first
        if (secondLabelLeft > firstLabelLeft && secondLabelLeft < firstLabelRight ||
        // second label ends inside the first
        secondLabelRight > firstLabelLeft && secondLabelRight < firstLabelRight ||
        // second label covers the first
        secondLabelLeft < firstLabelLeft && secondLabelRight > firstLabelRight) {
            return true;
        }
        return false;
    };

    VersionReportView.toggleChartVisibility = function (checkIfStarted) {

        $('#ghx-chart-mismatch').empty();
        if (checkIfStarted && VersionReport.model.notStarted()) {
            $('#ghx-version-report-no-issues').hide();
            $('#ghx-report-content').hide();
            $('#ghx-version-report-not-started').show();
            ChartView.hideSpinner();
        } else if (VersionReport.model.hasIssues()) {
            $('#ghx-version-report-no-issues').hide();
            $('#ghx-version-report-not-started').hide();
            $('#ghx-report-content').show();
        } else {
            $('#ghx-report-content').hide();
            $('#ghx-version-report-not-started').hide();
            $('#ghx-version-report-no-issues').show();
            ChartView.hideSpinner();
        }
    };

    /**
     * Gather the data points we want to draw
     *
     * @param plot : the chart object
     * @param ctx : the canvas 2d context
     * @param series : series currently being drawn
     */
    VersionReportView.gatherPoints = function (plot, ctx, series) {
        if (!plot.mouseEntered) {
            return;
        } // Don't add data points when mouse is outside of the plot

        VersionReportController.setPointsForChart(plot, series);
    };

    VersionReportView.bindEvents = function (plotHolder) {
        // mouse enter/leave events
        plotHolder.unbind('flotMouseEnter flotMouseLeave').bind('flotMouseEnter flotMouseLeave', FlotChartUtils.handleMouseOverStateChange);

        // register plothover and plotclick event handlers
        // we first unbind here, otherwise we add on each invocation a new click handler
        plotHolder.unbind('plothover plotclick');
        plotHolder.bind('plothover', VersionReportView.handlePlotHover);
        plotHolder.bind('plotclick', FlotChartUtils.handlePlotClick);
    };

    /**
     * Handler for  plot hover events.
     */
    VersionReportView.handlePlotHover = function (event, pos, item) {
        // fetch the issue for that index
        if (item) {
            // fetch the data for the given point
            var seriesId = item.series.id;
            var dataIndex = item.dataIndex;
            var dataPoint = VersionReport.model.getDataBySeriesAtIndex(seriesId, dataIndex);

            if (!_.isEmpty(dataPoint)) {
                if (!VersionReportView.isDataPointSameAsPrevious(item)) {
                    VersionReportView.previousPlotHoverPoint = {
                        dataIndex: item.dataIndex,
                        seriesIndex: item.seriesIndex
                    };

                    // remove the previous tooltip and add a new one
                    $('#ghx-tooltip').remove();
                    VersionReportView.showChartTooltip(item, seriesId, dataPoint, item.datapoint[0]);

                    // if the tooltip has no warnings then the issue is clickable
                    if (!$('#ghx-tooltip').hasClass('ghx-has-warnings')) {
                        ChartView.getChartView().addClass('ghx-clickable');
                    }
                }
                return;
            }
        }
        $('#ghx-tooltip').remove();
        VersionReportView.previousPlotHoverPoint = null;
        ChartView.getChartView().removeClass('ghx-clickable');
    };

    VersionReportView.isDataPointSameAsPrevious = function (item) {
        if (_.isUndefined(VersionReportView.previousPlotHoverPoint) || _.isNull(VersionReportView.previousPlotHoverPoint)) {
            // can't be the same if there wasn't one
            return false;
        }

        return item.dataIndex == VersionReportView.previousPlotHoverPoint.dataIndex && item.seriesIndex == VersionReportView.previousPlotHoverPoint.seriesIndex;
    };

    /**
     * Displays a data point tooltip
     */
    VersionReportView.showChartTooltip = function (item, seriesId, dataPoint) {
        if (!dataPoint) {
            return;
        }

        var tooltip;
        var tooltipData;

        if (VersionReport.model.isReleaseDatePredictionSeriesId(seriesId)) {
            // default to standard prediction
            var description = AJS.I18n.getText('gh.rapid.charts.version.predicted.release.date');
            var realDate = _.findWhere(VersionReportView.unmodifiedSeries, { id: 'prediction' }).data[1][0];
            if (seriesId === 'optimistic') {
                description = AJS.I18n.getText('gh.rapid.charts.version.predicted.release.date') + ' (' + AJS.I18n.getText('gh.rapid.charts.optimistic') + ')';
                realDate = _.findWhere(VersionReportView.unmodifiedSeries, { id: 'optimistic' }).data[1][0];
            } else if (seriesId === 'pessimisticFillerBottom') {
                description = AJS.I18n.getText('gh.rapid.charts.version.predicted.release.date') + ' (' + AJS.I18n.getText('gh.rapid.charts.pessimistic') + ')';
                realDate = _.findWhere(VersionReportView.unmodifiedSeries, { id: 'pessimisticFillerBottom' }).data[1][0];
            }

            tooltipData = {
                predictedReleaseDate: ChartUtils.renderUTCMillisAsDateWithoutTime(realDate),
                description: description
            };
            tooltip = new ChartTooltip(item, tooltipData, ChangeEvent.releaseDate);
        } else if (seriesId === 'trendline') {
            tooltipData = {
                description: AJS.I18n.getText('gh.version.start.date'),
                dateText: ChartUtils.renderUTCMillisAsDateWithoutTime(item.datapoint[0])
            };
            tooltip = new ChartTooltip(item, tooltipData, ChangeEvent.startDate);
        } else if (seriesId === 'unestimatedPercent') {
            tooltipData = {
                unestimated: (dataPoint.values.unestimatedIssueCount / dataPoint.values.issueCount * 100).toFixed(2),
                dateText: ChartUtils.renderUTCMillisAsDate(dataPoint.time)
            };
            tooltipData.changes = _.map(dataPoint.issues, function (issue) {
                return {
                    key: issue.key,
                    event: this.getChangeEvent(issue)
                };
            }, VersionReport.model);

            tooltip = new ChartTooltip(item, tooltipData, ChangeEvent.uncertainty);
        } else {
            tooltipData = VersionReport.model.getTooltipData(seriesId, dataPoint);
            tooltip = new ChartTooltip(item, tooltipData);
        }
        tooltip.render();
    };

    VersionReportView.renderMeta = function () {
        // No meta to render
        if (!VersionReport.model.hasIssues() || VersionReport.model.notStarted()) {
            return;
        }
        var version = VersionReport.model.getVersion();
        var predictionDate = _.findWhere(VersionReportView.unmodifiedSeries, { id: 'prediction' });
        var optimisticDate = _.findWhere(VersionReportView.unmodifiedSeries, { id: 'optimistic' });
        var pessimisticDate = _.findWhere(VersionReportView.unmodifiedSeries, { id: 'pessimisticFillerBottom' });
        var versionStatus = version.released ? AJS.I18n.getText('gh.rapid.charts.version.released') : AJS.I18n.getText('gh.rapid.charts.version.unreleased');
        var completionDate = {
            prediction: predictionDate ? ChartUtils.renderUTCMillisAsDateWithoutTime(predictionDate.data[1][0]) : null,
            optimistic: optimisticDate ? ChartUtils.renderUTCMillisAsDateWithoutTime(optimisticDate.data[1][0]) : null,
            pessimisticFillerBottom: pessimisticDate ? ChartUtils.renderUTCMillisAsDateWithoutTime(pessimisticDate.data[1][0]) : null
        };

        if (VersionReportView.gadgetMode) {
            AJS.$('#ghx-gadget-version-meta').empty().append(tpl.versionreport.renderVersionReportGadgetHeader({
                version: version,
                formattedReleaseDate: version.formattedReleaseDate,
                formattedStartDate: version.formattedStartDate,
                startTimezone: version.startTimeOffset,
                releaseTimezone: version.releaseTimeOffset,
                completionDate: completionDate
            })).show();

            // Inject version status next to version name
            AJS.$('#ghx-chart-header .ghx-version-report-version-status').text(' (' + versionStatus + ')');
        } else {
            AJS.$('#ghx-version-meta').empty().append(GH.tpl.versionreport.renderVersionReportHeader({
                version: version,
                formattedReleaseDate: version.formattedReleaseDate,
                formattedStartDate: version.formattedStartDate,
                startTimezone: version.startTimeOffset,
                releaseTimezone: version.releaseTimeOffset
            })).show();

            if (!version.released) {
                AJS.$('#ghx-report-content .ghx-version-report-completion-date-estimates').empty().append(GH.tpl.versionreport.renderNonGadgetCompletionDates({
                    completionDate: completionDate
                })).show();
            }
        }
    };

    VersionReportView.renderReport = function () {
        VersionReportView.toggleChartVisibility();

        if (VersionReport.model.hasIssues()) {
            VersionReportView.renderIssuesReport();
        }
    };

    VersionReportView.renderIssuesReport = function () {
        var content = VersionReport.model.getReportContent();
        var hasIssues = VersionReport.model.hasIssues();

        $('#ghx-report-content .ghx-version-report').empty().html(tpl.versionreport.renderVersionReport({
            content: content,
            jql: VersionReport.model.getJql(),
            estimationStatistic: VersionReport.model.getStatistic(),
            hasIssues: hasIssues
        }));
        ChartView.hideSpinner();
    };

    VersionReportView.setGadgetMode = function (gadgetMode) {
        VersionReportView.gadgetMode = gadgetMode;
    };

    VersionReportView.setWallboardMode = function (wallboardMode) {
        VersionReportView.wallboardMode = wallboardMode;
    };

    return VersionReportView;
});