/* global GH */
/**
 * Version Report Controller
 * @module jira-agile/rapid/ui/chart/version-report-controller
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/chart/burndown-rate
 * @requires module:jira-agile/rapid/ui/chart/chart-controller
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 * @requires module:jira-agile/rapid/ui/chart/version-report
 * @requires module:jira-agile/rapid/ui/chart/version-report-view
 * @requires module:jira-agile/rapid/ui/component/chart-picker
 */
define('jira-agile/rapid/ui/chart/version-report-controller', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var $ = require('jquery');
    var _ = require('underscore');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var BurndownRate = require('jira-agile/rapid/ui/chart/burndown-rate');
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');
    var ChartPicker = require('jira-agile/rapid/ui/component/chart-picker');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var VersionReport = require('jira-agile/rapid/ui/chart/version-report');
    var VersionReportView = require('jira-agile/rapid/ui/chart/version-report-view');

    // GLOBALS... FIX ME
    var Ajax = GH.Ajax;
    var ChartUtils = GH.ChartUtils;
    var FlotChartUtils = GH.FlotChartUtils;
    var log = GH.log;
    var Logger = GH.Logger;
    var RapidBoardState = GH.RapidBoard.State;
    var RapidBoardUrlState = GH.RapidBoard.UrlState;
    var TimeFormat = GH.TimeFormat;
    var VersionsModel = GH.VersionsModel;

    var VersionReportController = {};
    VersionReportController.nonWorkingDaysShown = true;

    VersionReportController.id = 'versionReport';

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    VersionReportController.analytics = new AnalyticsTracker('gh.report.versionReport');

    VersionReportController.init = function () {
        /**
         * @type module:jira-agile/rapid/ui/component/chart-picker
         */
        VersionReportController.versionPicker = new ChartPicker('versionReport.selectedVersionId', AJS.I18n.getText('gh.rapid.charts.version.noversions'));
        VersionReportController.versionPicker.bindToSelectedItemChanged(VersionReportController.handleVersionChanged);
    };

    VersionReportController.isApplicable = function (rapidViewConfig) {
        return rapidViewConfig.sprintSupportEnabled;
    };

    VersionReportController.setRapidView = function (rapidViewData) {
        VersionReportController.rapidViewData = rapidViewData || {};
    };

    VersionReportController.getRapidView = function () {
        return VersionReportController.rapidViewData;
    };

    /**
     * Provide params to create the "normalized" object for this report
     */
    VersionReportController.getNormalizedFromInternal = function () {
        return {
            version: VersionReportController.getSelectedVersionId() || null
        };
    };

    /**
     * Interpret the normalized params and identify the relevant ones to be converted into URL params
     */
    VersionReportController.toUrl = function (params) {
        if (!_.isNull(params.version)) {
            return {
                version: params.version
            };
        }
        return {};
    };

    /**
     * Update state from normalized state
     */
    VersionReportController.updateInternalFromNormalized = function (params) {
        VersionReportController.setSelectedVersionFromState(params.version);
    };

    /**
     * Get normalized state for this chart type
     */
    VersionReportController.getNormalizedFromUrl = function (params) {
        return {
            version: RapidBoardUrlState.normalizeNumberParam(params.version, null)
        };
    };

    /**
     * Get the selected sprint id - used for display in the url
     */
    VersionReportController.getSelectedVersionId = function () {
        // check whether we got a selected version id
        var version = VersionReportController.versionPicker.getSelectedItem();
        if (version) {
            return version.id;
        }

        // fall back to what is stored
        var versionId = VersionReportController.versionPicker.getStoredSelectedItemId();
        if (versionId != -1) {
            return versionId;
        }
        return null;
    };

    VersionReportController.show = function () {
        log('VersionReportController.show', Logger.Contexts.ui);
        VersionReportController.displayed = true;

        // stop here if haven't got a view
        if (!VersionReportController.rapidViewData) {
            return;
        }

        VersionReportView.renderChartLoading();

        ChartController.loadRapidViewConfig(VersionReportController, VersionReportController.loadVersionsList);

        VersionReportController.analytics.trigger("show");
    };

    VersionReportController.hide = function () {
        log('VersionReportController.hide', Logger.Contexts.ui);
        VersionReportController.displayed = false;
    };

    VersionReportController.loadVersionsList = function () {

        var params = {
            rapidViewId: VersionReportController.rapidViewData.id
        };

        Ajax.get({
            url: '/xboard/plan/backlog/versions.json',
            data: params
        }, 'versionsListForReport').done(VersionReportController.processVersionData).fail(ChartView.hideSpinner);
    };

    /**
     * Load the version picker
     * Load the next set of data for the selected version
     */
    VersionReportController.processVersionData = function (data) {
        if (!VersionReportController.displayed) {
            return;
        }

        var model = VersionReportController.versionModel = new VersionsModel(data.projects);
        model.updateRawData(data.versionData);
        var versions = _.map(model.getAllVersions(), function (version) {
            return {
                id: version.id,
                name: version.project ? version.project.key + ": " + version.name : version.name
            };
        });
        versions.reverse();
        VersionReportController.versionPicker.setItems(versions);
        VersionReportController.versionPicker.setDefaultItem(_.first(versions));

        VersionReportView.setUserTimeZoneLabel(TimeFormat.userTimeZoneLabel);

        VersionReportView.renderVersionPicker();

        if (VersionReportController.versionPicker.getSelectedItem()) {
            VersionReportController.loadReportData();
        } else {
            ChartView.hideSpinner();
        }

        VersionReportController.nonWorkingDaysShown = RapidBoardState.getNonWorkingDaysShown("version");

        // update the url based on the selected epic
        RapidBoardState.replaceState();
    };

    VersionReportController.loadReportData = function () {
        VersionReport.createModel(VersionReportController.rapidViewConfig.estimationStatistic);

        var params = {
            rapidViewId: VersionReportController.getRapidView().id,
            versionId: VersionReportController.getSelectedVersionId()
        };

        var chartPromise = Ajax.get({
            url: '/rapid/charts/versionprogresschart',
            data: params
        }, 'versionChartData').done(VersionReportController.processChartData);

        var reportPromise = Ajax.get({
            url: '/rapid/charts/versionreport',
            data: params
        }, 'versionReportData').done(VersionReportController.processReportData);

        $.when(chartPromise, reportPromise).done(VersionReportController.calculatePredictions).fail(VersionReportController.failedToLoad);
    };

    VersionReportController.failedToLoad = function () {
        ChartView.hideSpinner();
        VersionReportController.versionPicker.removeSelectedItemAndRevert(ChartView.getChartSelector(true));
    };

    VersionReportController.processChartData = function (data) {
        if (!VersionReportController.displayed) {
            return;
        }

        VersionReport.model.setRawVersionChartData(data);
    };

    VersionReportController.calculatePredictions = function () {
        // Calculate naive prediction date by doubling the time we think it'll take
        // Use this to decide how far into the future we need to query for non-working days to ensure that we get a big enough range
        var naivePredictionDate = VersionReport.model.getFurthestDateForNonWorkingDaysData();

        if (naivePredictionDate < Infinity && naivePredictionDate > VersionReport.model.getVersionStartDate()) {
            var endDate = naivePredictionDate;
            // set the minimum interval for new rates data to 90 days from now, to avoid cases where the predicted date
            // is very close to now and there are non-working days close by
            var ninetyDaysFromNow = new Date().valueOf() + ChartUtils.oneDayInMillis * 90;
            if (naivePredictionDate <= ninetyDaysFromNow) {
                endDate = ninetyDaysFromNow;
            }
            Ajax.get({
                url: '/rapidviewconfig/workingdays/rates',
                data: {
                    rapidViewId: VersionReportController.rapidViewData.id,
                    // the XAxis min is slightly before the start date to ensure the first data points are shown
                    // so we need to ensure working days just before the start date are also shown
                    startDate: VersionReport.model.getVersionStartDate() - ChartUtils.oneDayInMillis * 3,
                    endDate: endDate
                }
            }, "workingdaysrates").done(function (data) {
                BurndownRate.setRateData(data);
                VersionReportController.renderChart();
            });
        } else {
            VersionReportController.renderChart();
        }
    };

    VersionReportController.renderChart = function () {
        if (!VersionReportController.displayed) {
            return;
        }

        VersionReportView.renderChart();
        VersionReportView.renderIssuesLink();
        VersionReportView.renderMeta();
    };

    VersionReportController.setSelectedVersionFromState = function (version) {
        VersionReportController.versionPicker.storeSelectedItemId(version);
    };

    VersionReportController.setPointsForChart = function (plot, series) {
        // fetch the data for the series
        var seriesId = series.id;
        var seriesData = VersionReport.model.getDataBySeries(seriesId);
        if (!seriesData) {
            return;
        }

        var realPoints = FlotChartUtils.gatherPoints(series, seriesData);

        plot.realPointsPerSeries[seriesId] = realPoints;
    };

    VersionReportController.handleVersionChanged = function (event) {
        if (!VersionReportController.displayed) {
            return;
        }
        log('VersionReportController.handleVersionChanged', Logger.Contexts.event);

        // push the changed epic into the url
        RapidBoardState.pushState();
        VersionReportController.loadReportData();
    };

    VersionReportController.processReportData = function (data) {
        if (!VersionReportController.displayed) {
            return;
        }

        VersionReport.model.setRawData(data);
        VersionReportView.renderReport();
    };

    return VersionReportController;
});