/* global GH */
/**
 * Version Report Chart Controller
 * @module jira-agile/rapid/ui/chart/version-report-chart-controller
 * @requires module:jquery
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/chart/burndown-rate
 * @requires module:jira-agile/rapid/ui/chart/chart-controller
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 * @requires module:jira-agile/rapid/ui/chart/version-report
 * @requires module:jira-agile/rapid/ui/chart/version-report-view
 */
define('jira-agile/rapid/ui/chart/version-report-chart-controller', ['require'], function (require) {
    'use strict';

    // REQUIRES
    var $ = require('jquery');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var BurndownRate = require('jira-agile/rapid/ui/chart/burndown-rate');
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var VersionReport = require('jira-agile/rapid/ui/chart/version-report');
    var VersionReportView = require('jira-agile/rapid/ui/chart/version-report-view');

    // GLOBALS... FIX ME
    var Ajax = GH.Ajax;
    var ChartUtils = GH.ChartUtils;
    var FlotChartUtils = GH.FlotChartUtils;
    var log = GH.log;
    var Logger = GH.Logger;
    var TimeFormat = GH.TimeFormat;

    var VersionReportChartController = {};
    VersionReportChartController.nonWorkingDaysShown = true;

    VersionReportChartController.id = 'versionReportChart';

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    VersionReportChartController.analytics = new AnalyticsTracker('gh.report.versionReportChart');

    VersionReportChartController.isApplicable = function (rapidViewConfig) {
        return rapidViewConfig.sprintSupportEnabled;
    };

    VersionReportChartController.setRapidView = function (rapidViewData) {
        VersionReportChartController.rapidViewData = rapidViewData || {};
    };

    VersionReportChartController.getRapidView = function () {
        return VersionReportChartController.rapidViewData;
    };

    VersionReportChartController.setVersionId = function (versionId) {
        VersionReportChartController.versionId = versionId;
    };

    VersionReportChartController.getVersionId = function () {
        return VersionReportChartController.versionId;
    };

    VersionReportChartController.show = function () {
        log('VersionReportChartController.show', Logger.Contexts.ui);
        VersionReportChartController.displayed = true;

        // stop here if haven't got a view
        if (!VersionReportChartController.rapidViewData) {
            return;
        }

        VersionReportView.renderChartLoading();

        ChartController.loadRapidViewConfig(VersionReportChartController, VersionReportChartController.loadReportData);

        VersionReportChartController.analytics.trigger("show");
    };

    VersionReportChartController.loadReportData = function (rapidViewConfigData) {
        VersionReport.createModel(rapidViewConfigData.estimationStatistic);
        VersionReportView.setUserTimeZoneLabel(TimeFormat.userTimeZoneLabel);

        var params = {
            rapidViewId: VersionReportChartController.getRapidView().id,
            versionId: VersionReportChartController.getVersionId()
        };

        var chartPromise = Ajax.get({
            url: '/rapid/charts/versionprogresschart',
            data: params
        }, 'versionChartData')
            .done(VersionReportChartController.processChartData);
        $.when(chartPromise)
            .done(VersionReportChartController.calculatePredictions)
            .fail(VersionReportChartController.failedToLoad);
    };

    VersionReportChartController.failedToLoad = function () {
        ChartView.hideSpinner();
    };

    VersionReportChartController.processChartData = function (data) {
        if (!VersionReportChartController.displayed) {
            return;
        }

        VersionReport.model.setRawVersionChartData(data);
    };

    VersionReportChartController.calculatePredictions = function () {
        // Calculate naive prediction date by doubling the time we think it'll take
        // Use this to decide how far into the future we need to query for non-working days to ensure that we get a big enough range
        var naivePredictionDate = VersionReport.model.getFurthestDateForNonWorkingDaysData();

        if (naivePredictionDate < Infinity && naivePredictionDate > VersionReport.model.getVersionStartDate()) {
            var endDate = naivePredictionDate;
            // set the minimum interval for new rates data to 90 days from now, to avoid cases where the predicted date
            // is very close to now and there are non-working days close by
            var ninetyDaysFromNow = new Date().valueOf() + (ChartUtils.oneDayInMillis * 90);
            if (naivePredictionDate <= ninetyDaysFromNow) {
                endDate = ninetyDaysFromNow;
            }
            Ajax.get({
                url: '/rapidviewconfig/workingdays/rates',
                data: {
                    rapidViewId: VersionReportChartController.getRapidView().id,
                    // the XAxis min is slightly before the start date to ensure the first data points are shown
                    // so we need to ensure working days just before the start date are also shown
                    startDate: VersionReport.model.getVersionStartDate() - (ChartUtils.oneDayInMillis * 3),
                    endDate: endDate
                }
            }, "workingdaysrates")
                .done(function (data) {
                    BurndownRate.setRateData(data);
                    VersionReportChartController.renderChart();
                });
        } else {
            VersionReportChartController.renderChart();
        }
    };

    VersionReportChartController.renderChart = function () {
        if (!VersionReportChartController.displayed) {
            return;
        }

        VersionReportView.renderChart();
        VersionReportView.renderMeta();

        ChartView.hideSpinner();
        VersionReportChartController.onChartRendered();
    };

    VersionReportChartController.setPointsForChart = function (plot, series) {
        // fetch the data for the series
        var seriesId = series.id;
        var seriesData = VersionReport.model.getDataBySeries(seriesId);
        if (!seriesData) {
            return;
        }

        var realPoints = FlotChartUtils.gatherPoints(series, seriesData);

        plot.realPointsPerSeries[seriesId] = realPoints;
    };

    //callback function when we finished load version report chart
    //client will override this one to do some useful stuffs like resize the gadget
    VersionReportChartController.onChartRendered = function () {
    };

    return VersionReportChartController;
});
AJS.namespace('GH.VersionReportChartController', null, require('jira-agile/rapid/ui/chart/version-report-chart-controller'));
