/* global AJS, GH, _ */
/**
 * Chart view component.
 *
 * Loads chart data, then passes contorl to chart implementation for actual rendering
 */
(function () {

    var $ = require('jquery');

    var RapidBoardDataLoader = require('jira-agile/rapid/rapid-board-data-loader');

    GH.SprintRetrospectiveController = {};

    GH.SprintRetrospectiveController.id = 'sprintRetrospective';

    GH.SprintRetrospectiveController.displayed = false;
    GH.SprintRetrospectiveController.supportsPages = false;
    GH.SprintRetrospectiveController.rapidViewData = {};
    GH.SprintRetrospectiveController.rapidViewConfig = {};
    GH.SprintRetrospectiveController.referredBySprintCompleteDialog = false;

    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.SprintRetrospectiveController.analytics = new AnalyticsTracker("gh.report.sprintRetrospective");
    GH.SprintRetrospectiveController.linkPagesAnalytics = {
        action: new AnalyticsTracker("gh.linkedPages.discovery.action"),
        show: new AnalyticsTracker("gh.linkedPages.discovery.show")
    };

    GH.SprintRetrospectiveController.setRapidView = function (rapidViewData) {
        GH.SprintRetrospectiveController.rapidViewData = rapidViewData || {};
    };

    GH.SprintRetrospectiveController.isApplicable = function (rapidViewConfig) {
        return rapidViewConfig.sprintSupportEnabled;
    };

    /**
     * Provide params to create the "normalized" object for this report
     */
    GH.SprintRetrospectiveController.getNormalizedFromInternal = function () {
        return {
            sprint: GH.SprintRetrospectiveController.getSelectedSprintId() || null
        };
    };

    /**
     * Interpret the normalized params and identify the relevant ones to be converted into URL params
     */
    GH.SprintRetrospectiveController.toUrl = function (params) {
        if (_.isNull(params.sprint)) {
            return {};
        }
        return {
            sprint: params.sprint
        };
    };

    /**
     * Update state from normalized state
     * @param params
     */
    GH.SprintRetrospectiveController.updateInternalFromNormalized = function (params) {
        GH.SprintRetrospectiveController.setSelectedSprintFromUrlState(params.sprint);
    };

    /**
     * Get normalized state for this chart type
     */
    GH.SprintRetrospectiveController.getNormalizedFromUrl = function (params) {
        return {
            sprint: GH.RapidBoard.UrlState.normalizeNumberParam(params.sprint, null)
        };
    };

    GH.SprintRetrospectiveController.init = function () {
        var ChartPicker = require('jira-agile/rapid/ui/component/chart-picker');
        // create a new sprint picker
        /**
         * @type module:jira-agile/rapid/ui/component/chart-picker
         */
        GH.SprintRetrospectiveController.sprintPicker = new ChartPicker('sprintRetrospective.selectedSprintId');
    };

    GH.SprintRetrospectiveController.show = function () {
        GH.log('GH.SprintRetrospectiveController.show', GH.Logger.Contexts.ui);
        GH.SprintRetrospectiveController.displayed = true;

        // stop here if haven't got a view
        if (!GH.SprintRetrospectiveController.rapidViewData) {
            return;
        }

        // render the view
        GH.SprintRetrospectiveController.renderChartView();

        GH.SprintRetrospectiveController.analytics.trigger("show");
    };

    GH.SprintRetrospectiveController.hide = function () {
        GH.log('GH.SprintRetrospectiveController.hide', GH.Logger.Contexts.ui);
        GH.SprintRetrospectiveController.displayed = false;

        GH.SprintRetrospectiveView.hide();

        // unbind listeners
        AJS.$(GH).unbind('.' + GH.SprintRetrospectiveController.id);
    };

    /**
     * Loads the chart data for a given chart type
     */
    GH.SprintRetrospectiveController.renderChartView = function () {
        // controls
        GH.SprintRetrospectiveController.initializeControls();

        // render the view
        GH.ChartView.getChartContentElem(true).html(GH.tpl.sprintretrospective.renderReportBody());

        // show the spinner
        GH.ChartView.showSpinner();

        // then load the chart
        GH.SprintRetrospectiveController.loadRapidViewConfig();
    };

    /**
     * Renders the dropdown menu for the Chart Menu items & attach events
     */
    GH.SprintRetrospectiveController.initializeControls = function () {
        // controls structure
        AJS.$('#ghx-chart-controls').hide(); //.html(GH.tpl.chartview.renderChartControls());

        // and listen to selection changes
        GH.SprintRetrospectiveController.sprintPicker.bindToSelectedItemChanged(GH.SprintRetrospectiveController.handleSprintChanged);
    };

    GH.SprintRetrospectiveController.loadRapidViewConfig = function () {
        var callback = function callback(data) {
            // ignore if we are not displayed anymore
            if (!GH.SprintRetrospectiveController.displayed) {
                return;
            }
            GH.SprintRetrospectiveController.rapidViewConfig = data;
            GH.SprintRetrospectiveController.loadSprintOptions();
        };

        GH.RapidViewConfig.fetchConfiguration(GH.SprintRetrospectiveController.rapidViewData.id).done(callback);
    };

    /**
     * Loads the chart.
     * This function updates the tools, initializes the spinner, then sends off a request to load the data
     */
    GH.SprintRetrospectiveController.loadSprintOptions = function () {
        var rapidViewId = GH.RapidBoard.State.getRapidViewId();

        // then issue the request
        GH.Ajax.get({
            url: '/sprintquery/' + rapidViewId,
            data: {
                includeHistoricSprints: false,
                includeFutureSprints: false
            }
        }, 'rapidChartData').done(GH.SprintRetrospectiveController.processSprintsData);
    };

    /**
     * Processes the returned chart data.
     */
    GH.SprintRetrospectiveController.processSprintsData = function processSprintsData(data) {
        // ignore if we are not displayed anymore
        if (!GH.SprintRetrospectiveController.displayed) {
            return;
        }

        // organise the sprints
        data.sprints.reverse();

        // set the sprints onto the picker
        GH.SprintRetrospectiveController.sprintPicker.setItems(data.sprints);

        // determine the default sprint
        GH.SprintRetrospectiveController.sprintPicker.setDefaultItem(GH.SprintRetrospectiveController.getDefaultSprintForPicker(data.sprints));

        // fetch the selected sprint
        var selectedSprint = GH.SprintRetrospectiveController.sprintPicker.getSelectedItem();

        // render the dropdown
        GH.SprintRetrospectiveController.sprintPicker.render(AJS.$('#ghx-chart-selector'));
        GH.ChartView.showChartGroup(); // ensure we show the dropdown already

        // update the url based on the selected sprint
        GH.RapidBoard.State.pushState();

        // and start drawing the report for the selectedSprint
        if (selectedSprint) {
            GH.SprintRetrospectiveController.loadReportData();
        } else {
            // hide spinner, nothing to show
            GH.ChartView.hideSpinner();
            // redraw header
            GH.ReportController.updateChartHeader(GH.ReportController.getCurrentChart());
            // hide the intro
            GH.ReportController.closeIntro();
            // update intro state
            GH.RapidBoard.State.setChartIntroState(false);
            // clear actions menu
            GH.ReportController.renderActions($('#ghx-chart-actions'), []);
        }
    };

    GH.SprintRetrospectiveController.loadReportData = function () {
        var selectedSprint = GH.SprintRetrospectiveController.sprintPicker.getSelectedItem();

        // load the report
        GH.Ajax.get({
            url: 'rapid/charts/sprintreport',
            data: {
                rapidViewId: GH.SprintRetrospectiveController.rapidViewConfig.id,
                sprintId: selectedSprint.id
            }
        }, "rapidChartData").done(function (sprintData) {
            if (!GH.SprintRetrospectiveController.displayed) {
                return;
            }
            GH.SprintRetrospectiveController.processReportData(selectedSprint, sprintData);
        }).fail(GH.ChartView.hideSpinner);

        // load the burndown datavar
        GH.Ajax.get({
            url: '/rapid/charts/scopechangeburndownchart.json',
            data: {
                'rapidViewId': GH.SprintRetrospectiveController.rapidViewConfig.id,
                'sprintId': selectedSprint.id
            }
        }, 'rapidChartData.burndown').done(function (data) {
            if (!GH.SprintRetrospectiveController.displayed) {
                return;
            }
            GH.SprintRetrospectiveView.showMiniBurndown(data);
        }).fail(GH.ChartView.hideSpinner);
    };

    GH.SprintRetrospectiveController.processReportData = function (sprint, sprintData) {
        GH.SprintRetrospectiveController.supportsPages = sprintData.supportsPages;

        GH.SprintRetrospectiveView.show();

        // get the renderer for the estimationstatistic
        var renderer;
        if (GH.SprintRetrospectiveController.rapidViewConfig.estimationStatistic.renderer == 'duration') {
            renderer = GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration;
        } else {
            renderer = GH.NumberFormat.format;
        }

        [sprintData.contents.completedIssues, sprintData.contents.issuesNotCompletedInCurrentSprint, sprintData.contents.issuesCompletedInAnotherSprint, sprintData.contents.puntedIssues].forEach(function (issues) {
            return RapidBoardDataLoader.resolveEntities(issues, sprintData.contents.entityData);
        });

        // handler for convert null value into dash (-)
        var renderValueOrDash = function renderValueOrDash(val) {
            if (val == null) {
                return '-';
            }

            return renderer(val);
        };

        // use the renderer to populate text for statistics values
        sprintData.contents.completedIssuesEstimateSum.text = renderValueOrDash(sprintData.contents.completedIssuesEstimateSum.value);
        sprintData.contents.completedIssuesInitialEstimateSum.text = renderValueOrDash(sprintData.contents.completedIssuesInitialEstimateSum.value);
        sprintData.contents.issuesNotCompletedEstimateSum.text = renderValueOrDash(sprintData.contents.issuesNotCompletedEstimateSum.value);
        sprintData.contents.issuesNotCompletedInitialEstimateSum.text = renderValueOrDash(sprintData.contents.issuesNotCompletedInitialEstimateSum.value);
        sprintData.contents.issuesCompletedInAnotherSprintEstimateSum.text = renderValueOrDash(sprintData.contents.issuesCompletedInAnotherSprintEstimateSum.value);
        sprintData.contents.issuesCompletedInAnotherSprintInitialEstimateSum.text = renderValueOrDash(sprintData.contents.issuesCompletedInAnotherSprintInitialEstimateSum.value);
        sprintData.contents.puntedIssuesEstimateSum.text = renderValueOrDash(sprintData.contents.puntedIssuesEstimateSum.value);
        sprintData.contents.puntedIssuesInitialEstimateSum.text = renderValueOrDash(sprintData.contents.puntedIssuesInitialEstimateSum.value);
        sprintData.contents.allIssuesEstimateSum.text = renderValueOrDash(sprintData.contents.allIssuesEstimateSum.value);

        // update
        var statFieldUpdater = function statFieldUpdater(issue) {
            // if there is no estimate statistic, we should put in a dash
            GH.SprintRetrospectiveController._setStatsValueToRenderValueOrDash(issue, 'estimateStatistic', renderer);

            // if there is no current estimate statistic, we should put in a dash
            GH.SprintRetrospectiveController._setStatsValueToRenderValueOrDash(issue, 'currentEstimateStatistic', renderer);
        };
        _.each(sprintData.contents.completedIssues, statFieldUpdater);
        _.each(sprintData.contents.issuesNotCompletedInCurrentSprint, statFieldUpdater);
        _.each(sprintData.contents.issuesCompletedInAnotherSprint, statFieldUpdater);
        _.each(sprintData.contents.puntedIssues, statFieldUpdater);

        var sprintJql = GH.ChartUtils.createIssueJql(sprintData.contents.completedIssues.concat(sprintData.contents.issuesNotCompletedInCurrentSprint).concat(sprintData.contents.puntedIssues));
        var completedIssuesJql = GH.ChartUtils.createIssueJql(sprintData.contents.completedIssues);
        var issuesNotCompletedInCurrentSprintJql = GH.ChartUtils.createIssueJql(sprintData.contents.issuesNotCompletedInCurrentSprint);
        var issuescompletedInAnotherSprintJql = GH.ChartUtils.createIssueJql(sprintData.contents.issuesCompletedInAnotherSprint);
        var puntedIssuesJql = GH.ChartUtils.createIssueJql(sprintData.contents.puntedIssues);

        var showPages = true; // this is temporary while we wait for a flag that indicates this!

        GH.SprintRetrospectiveView.showSprintReport(sprint, sprintData, GH.SprintRetrospectiveController.rapidViewConfig.estimationStatistic, encodeURIComponent(sprintJql), encodeURIComponent(completedIssuesJql), encodeURIComponent(issuesNotCompletedInCurrentSprintJql), encodeURIComponent(issuescompletedInAnotherSprintJql), encodeURIComponent(puntedIssuesJql), showPages);

        GH.SprintRetrospectiveController.postProcessReportData(sprint);
    };

    /**
     * Called whenever the sprint picker changed
     */
    GH.SprintRetrospectiveController.handleSprintChanged = function handleSprintChanged(event) {

        GH.SprintRetrospectiveView.clearDialog();

        if (!GH.SprintRetrospectiveController.displayed) {
            return;
        }

        GH.log('GH.SprintRetrospectiveController.handleSprintChanged', GH.Logger.Contexts.event);

        // push the changed sprint into the url
        GH.RapidBoard.State.pushState();

        // and load the report data again
        GH.SprintRetrospectiveController.loadReportData();
    };

    GH.SprintRetrospectiveController.getDefaultSprintForPicker = function (sprints) {
        // for a retro report, we want the last completed sprint as the default
        var completedSprints = _.filter(sprints, function (sprint) {
            return 'CLOSED' === sprint.state;
        });

        if (completedSprints.length > 0) {
            return _.first(completedSprints);
        }
        return _.first(sprints);
    };

    /**
     * Get the selected sprint id - used for display in the url
     */
    GH.SprintRetrospectiveController.getSelectedSprintId = function () {
        // check whether we got a selected sprint id
        var sprint = GH.SprintRetrospectiveController.sprintPicker.getSelectedItem();
        if (sprint) {
            return sprint.id;
        }

        // fall back to what is stored
        var sprintId = GH.SprintRetrospectiveController.sprintPicker.getStoredSelectedItemId();
        if (sprintId != -1) {
            return sprintId;
        }
        return null;
    };

    /**
     * Set the selected sprint id as provided in the url
     */
    GH.SprintRetrospectiveController.setSelectedSprintFromUrlState = function (sprint) {
        GH.SprintRetrospectiveController.sprintPicker.storeSelectedItemId(sprint);
    };

    /**
     * Hook executed after the report has finished processing
     */
    GH.SprintRetrospectiveController.postProcessReportData = function (sprint) {
        // Create Retro Feature Discovery Inline Dialog
        if (GH.SprintRetrospectiveController.supportsPages && GH.SprintRetrospectiveController.shouldShowCreateRetroDialog(sprint.id)) {
            GH.SprintRetrospectiveView.showCreateRetrospectiveDialog(sprint);
        }
    };

    /**
     * Boolean indicating whether or not the user needs to be displayed the discovery dialog
     * for creating a retro
     */
    GH.SprintRetrospectiveController.shouldShowCreateRetroDialog = function (sprintId) {
        return !!_.find(GH.SprintRetrospectiveController.getSprintsMarkedForRetroDialog(), function (sprint) {
            return sprint.sprintId === sprintId && !sprint.dismissed;
        });
    };

    /**
     * Get the number of times the user has been displayed the create retro discovery dialog from local storage
     */
    GH.SprintRetrospectiveController.getSprintsMarkedForRetroDialog = function () {
        return GH.storage.get('gh.markedSprintsForRetroDialog') || [];
    };

    GH.SprintRetrospectiveController.markSprintAsDismissed = function (sprintId) {
        var sprints = GH.SprintRetrospectiveController.getSprintsMarkedForRetroDialog();
        _.each(sprints, function (sprint) {
            if (sprint.sprintId === sprintId) {
                sprint.dismissed = true;
            }
        });
        GH.storage.put('gh.markedSprintsForRetroDialog', sprints);
    };

    /**
     * Increase the number of times the user has been displayed the create retro discovery dialog
     */
    GH.SprintRetrospectiveController.tryMarkSprintForRetroDialog = function (sprintId) {
        var markedSprints = GH.SprintRetrospectiveController.getSprintsMarkedForRetroDialog();
        function isSpecifiedSprint(elem) {
            return elem.sprintId === sprintId;
        }
        if (!_.find(markedSprints, isSpecifiedSprint) && markedSprints.length < 5) {
            markedSprints.push({
                dismissed: false,
                sprintId: sprintId
            });
        }
        GH.storage.put('gh.markedSprintsForRetroDialog', markedSprints);
    };

    /**
     * Fetch the primary appLink and jiraApplicationLink data
     */
    GH.SprintRetrospectiveController.getAppLinkData = function () {
        var sprint = GH.SprintRetrospectiveController.sprintPicker.getSelectedItem();
        function resolveAppLinkUrl(applicationLinks) {
            var appLink = _.findWhere(applicationLinks, { primary: true });
            return appLink ? appLink.url + GH.LinkedPagesController.CONFLUENCE_CREATE_PAGE_ACTION : null;
        }
        return GH.LinkedPagesController.getLinkedPagesForEntity(GH.LinkedPagesController.SPRINT, sprint.id).andThen(function (data) {
            var appLinkData = {
                jiraApplicationId: data.jiraApplicationId,
                appLinkUrl: resolveAppLinkUrl(data.applicationLinks),
                entityId: sprint.id,
                jiraUrl: data.jiraBaseUrl
            };
            return appLinkData;
        });
    };

    /**
     * Get the render value from renderer handle if the statistic value is defined.
     * Otherwise, set it as "-"
     */
    GH.SprintRetrospectiveController._setStatsValueToRenderValueOrDash = function (issue, propName, rendererHandler) {
        if (_.isUndefined(issue[propName]) || _.isUndefined(issue[propName].statFieldValue.value)) {
            issue[propName] = {
                statFieldValue: {
                    text: '-'
                }
            };
            // otherwise we just render the value in to the text field
        } else {
            issue[propName].statFieldValue.text = rendererHandler(issue[propName].statFieldValue.value);
        }
    };
})();