/* global GH */
/**
 * Release Date Predictor
 * @module jira-agile/rapid/ui/chart/release-date-predictor
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/chart/release-date-predictor', ['require'], function (require) {
    'use strict';

    // REQUIRES
    var _ = require('underscore');
    var BurndownRate = require('jira-agile/rapid/ui/chart/burndown-rate');
    var ChangeEvent = require('jira-agile/rapid/ui/chart/change-event');

    // GLOBALS... FIX ME
    var ChartUtils = GH.ChartUtils;

    var ReleaseDatePredictor = {};

    /**
     * Calculates predicted release dates based on the given options
     *
     *
     * @return {{standard: Array, optimistic: Array, pessimistic: Array, scope: Array}}
     */
    ReleaseDatePredictor.calculateSeries = function (options) {
        var emptyPredictions = {
            standard: [],
            optimistic: [],
            pessimistic: [],
            scope: []
        };

        // if we haven't yet completed at least 10% of the scope, don't try to predict
        // predictions will be too far into the future and meaningless
        if (options.totalCompleted / options.totalScope < 0.1 || options.totalCompleted === options.totalScope) {
            return emptyPredictions;
        }

        var standard = ReleaseDatePredictor.createDataSeries(_.defaults({estimatePerDay: options.estimatePerDay.standard}, options));
        var predictedEndDate = Infinity;
        if (standard.length > 0) {
            predictedEndDate = _.last(standard)[0];
        }

        if (predictedEndDate == Infinity) {
            return emptyPredictions;
        }

        var optimistic = ReleaseDatePredictor.createDataSeries(_.defaults({estimatePerDay: options.estimatePerDay.optimistic}, options));
        var pessimistic = ReleaseDatePredictor.createDataSeries(_.defaults({estimatePerDay: options.estimatePerDay.pessimistic}, options));

        var xAxisEndDate = ReleaseDatePredictor.calculateXAxisEndDate(pessimistic, predictedEndDate);

        var scope = [];
        scope.push([options.startDate, options.totalScope]);
        scope.push([xAxisEndDate, options.totalScope]);

        return {
            standard: standard,
            optimistic: optimistic,
            pessimistic: pessimistic,
            scope: scope,
            xAxisEndDate: xAxisEndDate
        };
    };

    ReleaseDatePredictor.calculateXAxisEndDate = function (dateRange, defaultDate) {
        var xAxisEndDate = Infinity;
        if (dateRange.length > 0) {
            xAxisEndDate = _.last(dateRange)[0] + ChartUtils.oneDayInMillis;
        }

        if (xAxisEndDate === Infinity) {
            xAxisEndDate = defaultDate + ChartUtils.oneDayInMillis;
        }

        return xAxisEndDate;
    };

    ReleaseDatePredictor.createDataSeries = function (options) {

        var series = [];
        var estimatePerDay = options.estimatePerDay;

        // starting value
        series.push([options.startDate, options.totalCompleted]);

        if (estimatePerDay <= 0) {
            // we'll never finish, so just create an end point of infinity and return
            series.push([Infinity, options.totalCompleted]);
            return series;
        }

        var workingDays = BurndownRate.getWorkingDays();
        if (workingDays.length === 0) {
            // we have no working days at all (exceptional case) so don't try to predict anything
            return [];
        }

        var remainingScope = options.totalScope - options.totalCompleted;
        var completedScope = options.totalCompleted;

        for (var i = 0; i < workingDays.length; i++) {
            var block = workingDays[i];
            // unless the block end falls after the start date, ignore it
            if (block.end > options.startDate) {
                var dateToCalculateFrom = block.start;

                if (block.start > options.startDate) {
                    // if the block starts after the chart start date add a point for the block start with current estimate
                    series.push([block.start, completedScope]);
                } else {
                    // if the chart starts midway through the block, start calculating from the chart start
                    dateToCalculateFrom = options.startDate;
                }

                var days = (block.end - dateToCalculateFrom) / ChartUtils.oneDayInMillis;
                var remainingDaysToComplete = remainingScope / estimatePerDay;

                if (remainingDaysToComplete <= days) {
                    // we will complete the work within this block
                    // so draw a line from the start to the end date
                    var date = new Date(dateToCalculateFrom + (remainingDaysToComplete * ChartUtils.oneDayInMillis));
                    series.push([date.valueOf(), options.totalScope]);
                    // we finished, stop the loop
                    break;
                } else {
                    var scopeCompleteWithinBlock = days * estimatePerDay;
                    completedScope += scopeCompleteWithinBlock;
                    remainingScope -= scopeCompleteWithinBlock;
                    series.push([block.end, completedScope]);
                }
            }
        }

        // Our line goes directly to the end, ignoring the non-working days
        return [_.first(series), _.last(series)];
    };

    /**
     * Returns an array containing the completed estimate for each day.
     *
     * @param completedEstimateDataPoints
     * @param startTime
     * @param endTime
     * @return {Array}
     */
    ReleaseDatePredictor.getCompletedEstimatePerDay = function (completedEstimateDataPoints, startTime, endTime, seriesDataPoints) {
        var completedData = [];
        var currentEstimateCompleted = 0;
        var currentIndex = 0;
        var issuesCounted = [];
        var workingDays = 0;

        function addToIssuesCounted(issue) {
            // keep track of burned down issues
            var changeEvent = ChangeEvent.getChangeEvent(issue);
            if (changeEvent == ChangeEvent.burndown || changeEvent == ChangeEvent.completeScopeAdded) {
                issuesCounted.push(issue.key);
            }
        }

        function removeEstimateAlreadyCounted(currentComplete, issue) {
            // if we already counted this issue in the burndown
            if (_.contains(issuesCounted, issue.key)) {
                // if it's now being re-added to scope, don't count it again
                var changeEvent = ChangeEvent.getChangeEvent(issue);
                if (changeEvent === ChangeEvent.completeScopeAdded || changeEvent === ChangeEvent.scopeAdded) {
                    return currentComplete - issue.completedEstimate;
                }
            }

            return currentComplete;
        }

        // for each day since the start of the chart, check for estimate completed
        for (var i = startTime; i <= endTime; i += ChartUtils.oneDayInMillis) {
            // ignore non-working days
            if (!BurndownRate.isDateInNonWorkingBlock(i)) {
                workingDays++;
            }

            var completedToday = 0;
            var issuesChanged;
            // check the next data point in the series
            for (var j = currentIndex; j < completedEstimateDataPoints.length; j++) {
                var dateOfChange = completedEstimateDataPoints[j][0];

                // if this occurred in the last day, add it
                if (dateOfChange <= i) {
                    // if we didn't already pass the start time, don't add the points
                    if (dateOfChange > startTime) {
                        var estimate = completedEstimateDataPoints[j][1];

                        // track the burned down issues we count and reduce estimate for those we've already counted e.g. re-added scope
                        issuesChanged = (seriesDataPoints[j]) ? seriesDataPoints[j].issues : [];
                        if (issuesChanged) {
                            estimate = _.reduce(issuesChanged, removeEstimateAlreadyCounted, estimate);
                            _.each(issuesChanged, addToIssuesCounted);
                        }

                        if (estimate > currentEstimateCompleted) {
                            // Add the difference between the last change and the current value
                            completedToday += (estimate - currentEstimateCompleted);
                        }
                        // Reset the current value to compare for next time
                        currentEstimateCompleted = completedEstimateDataPoints[j][1];
                    } else {
                        // still want to add to the issues we've counted so we don't count it again
                        issuesChanged = (seriesDataPoints[j]) ? seriesDataPoints[j].issues : [];
                        if (issuesChanged) {
                            _.each(issuesChanged, addToIssuesCounted);
                        }
                        // also need to reset the current estimate completed
                        currentEstimateCompleted = completedEstimateDataPoints[j][1];
                    }
                } else {
                    // Passed today, reset index for next loop
                    currentEstimateCompleted = completedEstimateDataPoints[j][1];
                    currentIndex = j;
                    break;
                }
            }

            completedData.push(
                {
                    time: i,
                    estimate: completedToday
                }
            );

        }
        return {
            completedData: completedData,
            workingDays: workingDays
        };
    };

    /**
     * Calculates the average estimate completed per day based on the given array containing the completed estimate per day.
     *
     * @param estimatePerDay An array consisting of data points that contain the completed estimate per day.
     * @return {number} The average
     */
    ReleaseDatePredictor.calculateAverageEstimateCompletedPerDay = function (estimatePerDay) {
        var totalDays = estimatePerDay.workingDays;
        if (totalDays === 0) {
            return 0;
        }
        var totalEstimate = _.reduce(estimatePerDay.completedData, function (total, dataPoint) {
            return total + dataPoint.estimate;
        }, 0);
        return totalEstimate / totalDays;
    };

    /**
     * Calculates the deviation amount for the completed estimate per day.
     */
    ReleaseDatePredictor.calculateDeviation = function (averageEstimateCompletedPerDay) {
        return averageEstimateCompletedPerDay * 0.1;
    };

    ReleaseDatePredictor.getPredictedCompletedEstimatePerDay = function (completedEstimateDataPoints, startTime, endTime, seriesDataPoints) {

        var estimatePerDay = ReleaseDatePredictor.getCompletedEstimatePerDay(completedEstimateDataPoints, startTime, endTime, seriesDataPoints);
        var averageEstimateCompleted = ReleaseDatePredictor.calculateAverageEstimateCompletedPerDay(estimatePerDay);
        var deviation = ReleaseDatePredictor.calculateDeviation(averageEstimateCompleted);
        var maxPerDay = averageEstimateCompleted + deviation;
        var minPerDay = Math.max(averageEstimateCompleted - deviation, 0);

        return {
            standard: averageEstimateCompleted,
            optimistic: maxPerDay,
            pessimistic: minPerDay
        };
    };

    ReleaseDatePredictor.calculatePredictionDate = function (startDate, estimatePerDay, remainingScope) {
        if (estimatePerDay <= 0) {
            return Infinity;
        }
        var daysToComplete = remainingScope / estimatePerDay;
        return new Date(startDate + (ChartUtils.oneDayInMillis * daysToComplete)).valueOf();
    };

    return ReleaseDatePredictor;
});
