/* global AJS, GH, _ */
GH.EpicReportController = {};

GH.EpicReportController.id = 'epicReport';

(function() {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.EpicReportController.analytics = new AnalyticsTracker('gh.report.epicReport');
})();

GH.EpicReportController.init = function () {
    var ChartPicker = require('jira-agile/rapid/ui/component/chart-picker');
    // create a new sprint picker
    /**
     * @type module:jira-agile/rapid/ui/component/chart-picker
     */
    GH.EpicReportController.epicPicker = new ChartPicker('epicReport.selectedEpicId', AJS.I18n.getText('gh.report.epic.noepics'));
    GH.EpicReportController.epicPicker.bindToSelectedItemChanged(GH.EpicReportController.handleEpicChanged);
};

GH.EpicReportController.setRapidView = function(rapidViewData) {
    GH.EpicReportController.rapidViewData = rapidViewData || {};
};

GH.EpicReportController.isApplicable = function(rapidViewConfig) {
    return rapidViewConfig.sprintSupportEnabled;
};

/**
 * Provide params to create the "normalized" object for this report
 */
GH.EpicReportController.getNormalizedFromInternal = function() {
    return {
        epic: GH.EpicReportController.getSelectedEpicKey() || null
    };
};

/**
 * Interpret the normalized params and identify the relevant ones to be converted into URL params
 */
GH.EpicReportController.toUrl = function(params) {
    if (_.isNull(params.epic)) {
        return {};
    }
    return {
        epic: params.epic
    };
};

/**
 * Update state from normalized state
 * @param params
 */
GH.EpicReportController.updateInternalFromNormalized = function(params) {
    GH.EpicReportController.setSelectedEpicFromUrlState(params.epic);
};

/**
 * Get normalized state for this chart type
 */
GH.EpicReportController.getNormalizedFromUrl = function(params) {
    return {
        epic: GH.RapidBoard.UrlState.normalizeStringParam(params.epic, null)
    };
};

GH.EpicReportController.hide = function() {
    GH.log('GH.EpicReportController.hide', GH.Logger.Contexts.ui);
    GH.EpicReportController.displayed = false;
};

GH.EpicReportController.show = function() {
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');

    GH.log('GH.EpicReportController.show', GH.Logger.Contexts.ui);
    GH.EpicReportController.displayed = true;

    // stop here if haven't got a view
    if (!GH.EpicReportController.rapidViewData) { return; }

    GH.EpicReportView.renderChartLoading();

    ChartController.loadRapidViewConfig(GH.EpicReportController, GH.EpicReportController.loadEpicsList);

    GH.EpicReportController.analytics.trigger("show");
};

/**
 * Load the list of epics
 */
GH.EpicReportController.loadEpicsList = function() {

    var params = {
        rapidViewId: GH.EpicReportController.rapidViewData.id
    };

    GH.Ajax.get({
        url: '/xboard/plan/backlog/epics.json',
        data: params
    }, 'epicListForEpicReport')
        .done(GH.EpicReportController.processEpicData)
        .fail(GH.ChartView.hideSpinner);

};

/**
 * Load the epic picker
 * Load the next set of data for the selected epic
 */
GH.EpicReportController.processEpicData = function(data) {
    if (!GH.EpicReportController.displayed) { return; }

    GH.EpicReportController.epicsData = data;

    var epics = _.map(data.epics, function(epic) {
        return {
            id: epic.key,
            name: GH.EpicConfig.getEpicNameString(epic.epicLabel, epic.key)
        };
    });

    GH.EpicReportController.epicPicker.setItems(epics);
    GH.EpicReportController.epicPicker.setDefaultItem(_.first(epics));

    GH.EpicReportView.renderEpicPicker();

    GH.EpicReportView.setUserTimeZoneLabel(GH.TimeFormat.userTimeZoneLabel);

    if (GH.EpicReportController.epicPicker.getSelectedItem()) {
        GH.EpicReportController.loadReportData();
    } else {
        GH.ChartView.hideSpinner();
    }

    // update the url based on the selected epic
    GH.RapidBoard.State.replaceState();
};

GH.EpicReportController.loadReportData = function() {
    GH.EpicReport.createModel();
    var epicKey = GH.EpicReportController.epicPicker.getSelectedItem().id;
    // get list of epics
    var params  = {
        rapidViewId: GH.EpicReportController.rapidViewData.id,
        epicKey: epicKey
    };

    var reportDataPromise = GH.Ajax.get({
        url:'/rapid/charts/epicreport',
        data: params
    }, 'epicReportData' + epicKey)
        .done(GH.EpicReportController.processReportData);

    var reportChartPromise = GH.Ajax.get({
        url:'/rapid/charts/epicprogresschart',
        data: params
    }, 'epicprogresschart' + epicKey)
        .done(GH.EpicReportController.processChartData);

    // kick off the data comparison check when both requests finished loading
    AJS.$.when(reportDataPromise, reportChartPromise)
        .done(GH.EpicReportController.performDataConsistencyCheck)
        .fail(GH.EpicReportController.failedToLoad);
};

GH.EpicReportController.failedToLoad = function() {
    GH.ChartView.hideSpinner();
    GH.EpicReportController.epicPicker.removeSelectedItemAndRevert(GH.ChartView.getChartSelector(true));
};

GH.EpicReportController.processReportData = function(data) {
    if (!GH.EpicReportController.displayed) { return; }

    var fullEpic = _.findWhere(GH.EpicReportController.epicsData.epics, { id: data.epic.id });
    if (fullEpic) {
        _.extend(data, _.pick(GH.EpicReportController.epicsData, 'canEditEpics', 'supportsPages'));
    }

    GH.EpicReport.model.setRawData(data);
    GH.EpicReportView.renderReport();
};

GH.EpicReportController.processChartData = function(data) {
    if (!GH.EpicReportController.displayed) { return; }

    GH.EpicReport.model.setRawEpicChartData(data);
    GH.EpicReportView.renderChart();
};

/**
 * In case a user changed issue type, it is possible that issues still contain statistic values
 * even though under the new type they should not have any value. As the report fetches the correct values,
 * but the chart solely looks at the change history we end up with a mismatch between the two.
 */
GH.EpicReportController.performDataConsistencyCheck = function() {
    if (!GH.EpicReportController.displayed) { return; }

    var problematicIssues = GH.EpicReport.model.getIssuesWithStatisticFieldMismatch();
    if (!_.isEmpty(problematicIssues)) {
        GH.EpicReportView.renderStatisticMismatchWarning(problematicIssues);
    }
};

GH.EpicReportController.handleEpicChanged = function() {
    if (!GH.EpicReportController.displayed) { return; }
    GH.log('GH.EpicReportController.handleEpicChanged', GH.Logger.Contexts.event);

    // push the changed epic into the url
    GH.RapidBoard.State.pushState();

    GH.EpicReportController.loadReportData();
};

GH.EpicReportController.getSelectedEpicKey = function() {
    var epic = GH.EpicReportController.epicPicker.getSelectedItem();
    if (epic) {
        return epic.id;
    }
};

GH.EpicReportController.setSelectedEpicFromUrlState = function(epicKey) {
    GH.EpicReportController.epicPicker.storeSelectedItemId(epicKey);
};

GH.EpicReportController.setPointsForChart = function(plot, series) {
    // fetch the data for the series
    var seriesId = series.id;
    var seriesData = GH.EpicReport.model.getDataBySeries(seriesId);
    if (!seriesData) { return; }

    var realPoints = GH.FlotChartUtils.gatherPoints(series, seriesData);

    plot.realPointsPerSeries[seriesId] = realPoints;
};
