(function(){
const InlineDialog = require('aui/inline-dialog');

GH.ChartTimeFrames = {};

/**
 * # of milliseconds in a day
 */
GH.ChartTimeFrames.day = 1000 * 60 * 60 * 24;

GH.ChartTimeFrames.dateFormat = "%d.%m.%Y";
GH.ChartTimeFrames.timeFormat = "%d/%m/%Y %H:%M";

// GHS-6145: format used by moment.js (which queries JIRA's look-and-feel date format settings)
GH.ChartTimeFrames.momentDateFormat = "LL"; // LL -> dmy
GH.ChartTimeFrames.momentTimeFormat = "LLL";  // LLL -> dmy: time

/**
 * Stores the current values
 */
GH.ChartTimeFrames.from = null;
GH.ChartTimeFrames.to = null;
GH.ChartTimeFrames.days = 7; // 7 days by default
GH.ChartTimeFrames.now = new Date().getTime();
GH.ChartTimeFrames.isRelative = true;
GH.ChartTimeFrames.minXValue = null;
GH.ChartTimeFrames.chartId = null;


/**
 * Initializes the chart time frames component
 */
GH.ChartTimeFrames.init = function() {
    // hacks to make inline dialog behave
    GH.ChartUtils.registerInlineDialogHacks("chartTimeFramesInlineDialog");
};

/**
 * Due to the appalling implementation of the Date.parseDate function in Calendar.js,
 * we need a special function to check if a given string represents a date.
 *
 * @param dateToCheck String representation of a date, consistent with dateFormat field.
 */
GH.ChartTimeFrames.isValidDate = function(dateToCheck) {
    var isCorrect = function(d) {
        // the parseDate method returns current date if the date given is incorrect
        // if a date was correctly parsed, only year, month and date would have been set
        return d.getHours() === 0 && d.getMinutes() === 0 && d.getSeconds() === 0 && d.getMilliseconds() === 0;
    };

    // however, there is a slight chance that the current date actually has all the other fields equal to zero
    // that's why we parse it again, hoping that at least the milliseconds part would be different the second time
    return isCorrect(Date.parseDate(dateToCheck, GH.ChartTimeFrames.dateFormat)) &&
           isCorrect(Date.parseDate(dateToCheck, GH.ChartTimeFrames.dateFormat));
};

GH.ChartTimeFrames.getDateAsText = function(date) {
    return new Date(date).print(GH.ChartTimeFrames.dateFormat);
};

GH.ChartTimeFrames.roundMillisToNextDay = function(millis) {
    var currentDayMillis = GH.ChartTimeFrames.roundMillisToBeginOfDay(millis);
    var nextDayMillis;
    if (currentDayMillis < millis) {
        nextDayMillis = currentDayMillis + GH.ChartTimeFrames.day;
    } else {
        nextDayMillis = currentDayMillis;
    }
    return nextDayMillis;
};

GH.ChartTimeFrames.roundMillisToBeginOfDay = function(millis) {
    var days = Math.floor(millis / GH.ChartTimeFrames.day);
    return days * GH.ChartTimeFrames.day;
};

/**
 * Convert milliseconds to days
 */
GH.ChartTimeFrames.millisToDays = function(duration) {
    if (duration) {
        return duration / GH.ChartTimeFrames.day;
    } else {
        return duration;
    }
};

/**
 * Convert days to milliseconds
 */
GH.ChartTimeFrames.daysToMillis = function(days) {
    if (days) {
        return days * GH.ChartTimeFrames.day;
    } else {
        return days;
    }
};

/**
 * Predefined time frames. All have today defined as end date
 */
GH.ChartTimeFrames.timeFrames = [
    {
        relative: true,
        days: 7,
        name: AJS.I18n.getText('gh.rapid.timeframes.1w')
    },
    {
        relative: true,
        days: 14,
        name: AJS.I18n.getText('gh.rapid.timeframes.2w')
    },
    {
        relative: true,
        days: 30,
        name: AJS.I18n.getText('gh.rapid.timeframes.1m')
    },
    {
        relative: true,
        days: 90,
        name: AJS.I18n.getText('gh.rapid.timeframes.3m')
    },
    {
        relative: true,
        days: 180,
        name: AJS.I18n.getText('gh.rapid.timeframes.6m')
    },
    {
        relative: true,
        days: 0,
        name: AJS.I18n.getText('gh.rapid.timeframes.inf')
    },
    {
        relative: false,
        days: -1,
        name: AJS.I18n.getText('gh.rapid.timeframes.custom')
    }
];

GH.ChartTimeFrames.getSelectedTimeFrame = function() {
    for(var i=0; i < GH.ChartTimeFrames.timeFrames.length; i++) {
        if (!GH.ChartTimeFrames.isRelative && !GH.ChartTimeFrames.timeFrames[i].relative) {
            return GH.ChartTimeFrames.timeFrames[i];
        }
        else if (GH.ChartTimeFrames.timeFrames[i].days == GH.ChartTimeFrames.days) {
            return GH.ChartTimeFrames.timeFrames[i];
        }
    }
    return false;
};

/**
 * Get the start date of the chart
 */
GH.ChartTimeFrames.getChartStartDate = function() {
    var day = GH.ChartTimeFrames.day;
    if(GH.ChartTimeFrames.isRelative) {
        // infinite
        if (GH.ChartTimeFrames.days == 0) {
            var min = GH.ChartTimeFrames.minXValue;
            var nowLessOne = GH.ChartTimeFrames.now - 1*day; // backtrack by 1 day
            if (!min || min > nowLessOne) {
                min = nowLessOne;
            }
            return min;
            // ensure we show at least 1 day
            //return GH.ChartTimeFrames.minXValue ? GH.ChartTimeFrames.minXValue : GH.ChartTimeFrames.now - 7*day; // return minXValue or 7 days
        }
        // # of days
        else {
            // return now - number of days
            return GH.ChartTimeFrames.now - GH.ChartTimeFrames.days*day;
        }
    } else {
        // return start date
        return GH.ChartTimeFrames.from;
    }
};

/**
 * Get the end date of the chart
 */
GH.ChartTimeFrames.getChartEndDate = function() {
    var day = GH.ChartTimeFrames.day;
    if(GH.ChartTimeFrames.isRelative) {
        return GH.ChartTimeFrames.now;
    } else {
        return GH.ChartTimeFrames.to;
    }
};

/**
 * Called before the ajax request to load a chart
 */
GH.ChartTimeFrames.update = function() {
    // load the configuration from the state
    var days = GH.RapidBoard.State.getBoardSetting(null, 'chart.timeframe.days', 7); // fall back to 7 days if not defined
    var from = new Date().getTime();
    from = GH.RapidBoard.State.getBoardSetting(null, 'chart.timeframe.from', from);
    var to = new Date().getTime() - 7*GH.ChartTimeFrames.day;
    to = GH.RapidBoard.State.getBoardSetting(null, 'chart.timeframe.to', to);

    GH.ChartTimeFrames.setNewChartLimits(days, from, to, true);
};

/**
 * Called once the chart data has been loaded
 */
GH.ChartTimeFrames.setBoundaries = function(minXValue, currentTime) {
    // set the internal values
    GH.ChartTimeFrames.now = currentTime;
    GH.ChartTimeFrames.minXValue = minXValue || currentTime-7*GH.ChartTimeFrames.day; // use currentTime if we got no minXValue

    // TODO: load persisted values

    // update the display
    GH.ChartTimeFrames.updateDatesInfo();
};

/**
 * Updates the from and to dates depending on the current selection
 */
GH.ChartTimeFrames.updateDialogFromAndToDates = function(contents) {
    // figure out which days is currently selected
    var activeButton = contents.find('.js-timeframe-button.active');
    var activeDays = activeButton.attr('data-timeframe-id');

    // nothing to do if the custom option is selected
    if (activeDays == -1) {
        return;
    }

    // calculate from and to dates
    var to = GH.ChartTimeFrames.now;
    var from;
    if (activeDays > 0) {
        from = to - activeDays*GH.ChartTimeFrames.day;
    } else {
        // infinitive, use min value
        from = GH.ChartTimeFrames.minXValue;
    }

    // render the strings
    var fromDateString = new Date(from).print(GH.ChartTimeFrames.dateFormat);
    var toDateString = new Date(to).print(GH.ChartTimeFrames.dateFormat);

    // update the fields
    contents.find('#ghx-field-from').val(fromDateString);
    contents.find('#ghx-field-to').val(toDateString);
};

/**
 * Updates the displayed dates information
 */
GH.ChartTimeFrames.updateDatesInfo = function() {
    // update from and to according to the days if we are relative. otherwise from and to are already set
    var selectedTimeFrame = GH.ChartTimeFrames.getSelectedTimeFrame();
    if (selectedTimeFrame.relative) {
        GH.ChartTimeFrames.to = GH.ChartTimeFrames.now;
        if (selectedTimeFrame.days > 0) {
            GH.ChartTimeFrames.from = GH.ChartTimeFrames.to - selectedTimeFrame.days * GH.ChartTimeFrames.day;
        } else {
            // infinitive, use min value
            GH.ChartTimeFrames.from = GH.ChartTimeFrames.minXValue;
        }
    }

    // format the dates
    var fromDateString = new Date(GH.ChartTimeFrames.from).print(GH.ChartTimeFrames.dateFormat);
    var toDateString = new Date(GH.ChartTimeFrames.to).print(GH.ChartTimeFrames.dateFormat);

    var display = AJS.I18n.getText('gh.rapid.timeframes.label', fromDateString, toDateString, selectedTimeFrame.name);
    AJS.$('.js-timeframe-info').text(display);
};

/**
 * Set the new chart limits and updates the chart
 */
GH.ChartTimeFrames.setNewChartLimits = function(activeDays, from, to, dontFireEvent) {
    if (activeDays < 0) { // absolute
        GH.ChartTimeFrames.from = from;
        GH.ChartTimeFrames.to = to;
        GH.ChartTimeFrames.isRelative = false;
        GH.ChartTimeFrames.days = -1;
    } else { // relative
        GH.ChartTimeFrames.isRelative = true;
        GH.ChartTimeFrames.days = activeDays;
    }

    // update the state variables
    GH.RapidBoard.State.setBoardSetting('chart.timeframe.days', GH.ChartTimeFrames.days);
    GH.RapidBoard.State.setBoardSetting('chart.timeframe.from', from);
    GH.RapidBoard.State.setBoardSetting('chart.timeframe.to', to);

    // update the date info
    GH.ChartTimeFrames.updateDatesInfo();

    // refresh the chart, no need to reload data
    if (!dontFireEvent) {
        AJS.$(GH).trigger('chartTimeFramesChanged');
    }
};


/**
 * Renders the dropdown menu for the Chart Menu items & attach events
 */
GH.ChartTimeFrames.initializeDialog = function(chartId) {
    // kill any previous inline dialogs
    AJS.$('#inline-dialog-chartTimeFramesInlineDialog').remove();

    // set up the chart ID so we know how to appropriately display time frames
    GH.ChartTimeFrames.chartId = chartId;

    // initialize inline dialogs
    var options = {
        width: 710,
        cacheContent : false, // don't cache the dialog content
        hideDelay : 60000 // set longer timeout (default is 10 seconds)
    };
    InlineDialog("#js-chart-timeframe", "chartTimeFramesInlineDialog", GH.ChartTimeFrames.renderChartTimeFramesDialog, options);
};

/**
 * Renders the reporting timeframes dialog
 */
GH.ChartTimeFrames.renderChartTimeFramesDialog = function(contents, trigger, showPopup) {
    // render the from and to dates. These will be empty if a relative time is selected
    var fromDateString = GH.ChartTimeFrames.from ? new Date(GH.ChartTimeFrames.from).print(GH.ChartTimeFrames.dateFormat): '';
    var toDateString = GH.ChartTimeFrames.to ? new Date(GH.ChartTimeFrames.to).print(GH.ChartTimeFrames.dateFormat) : '';
    var description = false;
    if (GH.ChartTimeFrames.chartId == GH.CFDController.id) {
        description = AJS.I18n.getText('gh.rapid.charts.cfd.range.selection.description');
    } else if (GH.ChartTimeFrames.chartId == GH.Reports.ControlChartController.id) {
        description = AJS.I18n.getText('gh.rapid.charts.control.range.selection.description');
    }

    // render the content. this is rerendered every time the dialog opens so we don't have to worry about updating the html
    contents.html(GH.tpl.charttimeframes.renderChartTimeFrames({
        'timeFrames': GH.ChartTimeFrames.timeFrames,
        'relative': GH.ChartTimeFrames.isRelative,
        'days': GH.ChartTimeFrames.days,
        'from': fromDateString,
        'to': toDateString,
        'desc': description
    }));

    // mark this for GH styling
    contents.closest('.aui-inline-dialog').addClass('ghx-inline-dialog');

    // update the date fields, necessary for relative options
    GH.ChartTimeFrames.updateDialogFromAndToDates(contents);

    // handler called to select one of the buttons
    var selectButton = function(button) {
        // skip if button is disabled
        if(button.hasClass('disabled')) {
            return;
        }

        // select/unselect others
        contents.find('.js-timeframe-button').removeClass('active');
        button.addClass('active');

        // update the from and to dates
        GH.ChartTimeFrames.updateDialogFromAndToDates(contents);
    };

    // handle clicks on the different buttons
    contents.find('.js-timeframe-button').click(function(event) {
        event.preventDefault(); // stop the button submitting early
        selectButton(AJS.$(this));
    });

    Calendar.setup({
        firstDay : 0,
        inputField : 'ghx-field-from',
        button : 'ghx-field-from-button',
        align : 'Tl',
        singleClick : true,
        cache: false,
        positionStyle : 'fixed',
        ifFormat : GH.ChartTimeFrames.dateFormat,
        showsTime: false,
        onUpdate: function() {
            selectButton(contents.find('.js-timeframe-button[data-timeframe-id="-1"]'));
        }
    });
    Calendar.setup({
        firstDay : 0,
        inputField : 'ghx-field-to',
        button : 'ghx-field-to-button',
        align : 'Tl',
        singleClick : true,
        cache: false,
        positionStyle : 'fixed',
        ifFormat : GH.ChartTimeFrames.dateFormat,
        showsTime: false,
        onUpdate: function() {
            selectButton(contents.find('.js-timeframe-button[data-timeframe-id="-1"]'));
        }
    });

    // handle click on update and cancel
    contents.find('.js-save-button').click(function(e) {
        e.preventDefault();
        GH.ChartTimeFrames.handleDialogSave(contents);
    });
    contents.find('.js-cancel-button').click(function(e) {
        InlineDialog.current.hide();
    });

    // now display the dialog
    showPopup();
};

/**
 * Called to save the dialog data
 */
GH.ChartTimeFrames.handleDialogSave = function handleDialogSave(contents) {
    // update the columns
    var activeButton = contents.find('.js-timeframe-button.active');
    var activeDays = parseInt(activeButton.attr('data-timeframe-id'), 10);
    var from = null;
    var to = null;

    // remove all errors
    contents.find('.ghx-error').remove();

    // fetch from and to in case of custom dates
    if (activeDays < 0) {
        // set from and to
        // load the from and to dates
        var fromDate = contents.find('#ghx-field-from').val();
        if (!fromDate || !GH.ChartTimeFrames.isValidDate(fromDate)) {
            contents.find('#ghx-field-from').closest('.field-group').append('<div class="ghx-error">'+AJS.I18n.getText('gh.rapid.timeframes.error.from')+'</div>');
            return;
        }
        fromDate = Date.parseDate(fromDate, GH.ChartTimeFrames.dateFormat);
        var toDate = contents.find('#ghx-field-to').val();
        if (!toDate || !GH.ChartTimeFrames.isValidDate(toDate)) {
            contents.find('#ghx-field-to').closest('.field-group').append('<div class="ghx-error">'+AJS.I18n.getText('gh.rapid.timeframes.error.to')+'</div>');
            return;
        }
        toDate = Date.parseDate(toDate, GH.ChartTimeFrames.dateFormat);
        if (fromDate.getTime() > toDate.getTime()) {
            contents.find('#ghx-field-to').closest('.field-group').append('<div class="ghx-error">'+AJS.I18n.getText('gh.rapid.timeframes.error.reverse')+'</div>');
            return;
        }
        from = fromDate.getTime();
        to = toDate.getTime();
    }

    // set the new limits
    GH.ChartTimeFrames.setNewChartLimits(activeDays, from, to);
    GH.RapidBoard.State.pushState();

    // hide dialog
    if (InlineDialog.current) {
        InlineDialog.current.hide();
    }
};
})();