/* global GH */
/**
 * Burndown Chart Model
 * @module jira-agile/rapid/ui/chart/burndown-chart-model
 * @requires module:underscore
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/ui/chart/burndown-chart-view
 * @requires module:jira-agile/rapid/ui/chart/burndown-rate
 * @requires module:jira-agile/rapid/ui/chart/burndown-table-producer
 * @requires module:jira-agile/rapid/ui/chart/burndown-timeline-producer
 * @requires module:jira-agile/rapid/ui/chart/chart-colors
 */
define('jira-agile/rapid/ui/chart/burndown-chart-model', ['require'], function (require) {
    'use strict';

    /// REQUIRES
    var _ = require('underscore');
    var BurndownChartView = require('jira-agile/rapid/ui/chart/burndown-chart-view');
    var BurndownRate = require('jira-agile/rapid/ui/chart/burndown-rate');
    var BurndownTableProducer;
    var BurndownTimelineProducer;
    var ChartColors = require('jira-agile/rapid/ui/chart/chart-colors');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    // GLOBALS... FIX ME
    var ChartUtils = GH.ChartUtils;
    var FlotChartUtils = GH.FlotChartUtils;

    // Resolve circular dependency
    GlobalEvents.on('pre-initialization', function () {
        BurndownTableProducer = require('jira-agile/rapid/ui/chart/burndown-table-producer');
        BurndownTimelineProducer = require('jira-agile/rapid/ui/chart/burndown-timeline-producer');
    });

    /**
     * Holds the data displayed in the burndown chart as well as the details table below it
     */
    var BurndownChartModel = {};

    /**
     * Holds the response data object
     */
    BurndownChartModel.rawData = {};

    /**
     * Configured table handler, defines displayed statistics as well as renders change events
     */
    BurndownChartModel.tableHandler = {};

    /**
     * Holds the timeline data
     */
    BurndownChartModel.timelineData = {};

    /**
     * Holds the series for burndown chart
     *
     * The number of series depends on the actual chart (2 in normal burndown (guideline, burndown), 3 in hour burndown (estimate, time spent, guideline).
     */
    BurndownChartModel.series = [];

    /**
     * Holds the data associated with points of the different series.
     *
     * Series can but don't have to hold data, access data by the series id, then the data point index
     */
    BurndownChartModel.seriesData = {};

    /**
     * Holds the representation of the y-axis
     */
    BurndownChartModel.yAxisData = {};

    /**
     * Holds the data for the data table
     */
    BurndownChartModel.dataTable = {};

    /**
     * Sets the control chart data received from the server
     */
    BurndownChartModel.setRawData = function (data) {
        // set the supplied data object
        BurndownChartModel.rawData = data;

        // set the rate data
        // TODO: move somewhere else
        BurndownRate.setRateData(data.workRateData);

        // prepares the data
        BurndownChartModel.prepareData();
    };

    /**
     * Returns the configured statistic for the burndown data
     */
    BurndownChartModel.getStatistic = function () {
        return BurndownChartModel.rawData.statisticField;
    };

    BurndownChartModel.isTimeTracking = function () {
        return BurndownChartModel.getStatistic().fieldId === 'timeestimate';
    };

    /**
     * Return the issue key of the specified issue's parent; or <tt>false</tt> if issue is a parent.
     */
    BurndownChartModel.getParentIssueKey = function (issueKey) {
        if (!_.isUndefined(BurndownChartModel.rawData.issueToParentKeys)) {
            return BurndownChartModel.rawData.issueToParentKeys[issueKey] || false;
        }
        return false;
    };

    /**
     * Calculate the series for the raw data
     */
    BurndownChartModel.prepareData = function () {
        // choose the correct statisticConsumer and table handler
        var isTimeTracking = BurndownChartModel.isTimeTracking();
        if (isTimeTracking) {
            BurndownChartModel.tableHandler = BurndownTableProducer.TimeTrackingTableHandler;
        } else {
            BurndownChartModel.tableHandler = BurndownTableProducer.EstimateStatisticTableHandler;
        }
        var statisticConsumer = ChartUtils.getStatisticConsumer(isTimeTracking);

        // calculate the timeline
        BurndownChartModel.timelineData = BurndownTimelineProducer.calculateTimelineData(BurndownChartModel.rawData, statisticConsumer);

        // calculate the series
        BurndownChartModel.calculateSeries();

        // if the sprint is closed then this is the last user to close it
        var lastUserWhoClosedHtml = BurndownChartModel.rawData.lastUserWhoClosedHtml;

        // calculate the data tables
        var dataTable = BurndownTableProducer.getDataTable(BurndownChartModel.timelineData, BurndownChartModel.tableHandler, statisticConsumer, lastUserWhoClosedHtml);

        // ensure we render all values in text form
        BurndownChartModel.renderDataTableValues(dataTable);

        BurndownChartModel.dataTable = dataTable;
    };


    BurndownChartModel.getChartStartTime = function () {
        var data = BurndownChartModel.timelineData;
        return data.startTime;
    };

    BurndownChartModel.calculateGuidelineSeries = function (data) {
        // total units to burn down
        var remainingPoints = data.startValue;

        // fetch all rates, reduced to the time frame of the guide line
        var blocks = BurndownRate.getRateDefinitions();
        var blocksInRange = BurndownRate.limitToTimeRange(blocks, data.startTime, data.endTime);

        // calculate the time per units, that is how long it will take to burn down one unit
        var timePerUnit = BurndownChartModel.calculateTimePerUnit(blocksInRange, remainingPoints);

        // beginning with start value, build array of points
        var seriesData = [[data.startTime, data.startValue]].concat(_.map(blocksInRange, function (block) {

            // if this is a working block we have to reduce remaining points accordingly
            if (block.rate === BurndownRate.WORKING_DAY) {
                // calculate the units and deduct from remaining points
                var time = block.start - block.end;
                var units = time / timePerUnit;
                remainingPoints -= units;
            }

            // don't go past the end of the chart
            return [_.min([block.end, data.endTime]), _.max([remainingPoints, 0])];
        }));

        seriesData = _.reject(seriesData, function (data) {
            return data.length === 0;
        });

        return {
            id: 'guideline',
            data: seriesData,
            color: '#999',
            label: AJS.I18n.getText('gh.rapid.chart.burndown.guideline.label')
        };
    };

    /**
     * Calculates the time it takes to burn down one unit of work according to the sprint start and end times only taking
     * working days into account
     */
    BurndownChartModel.calculateTimePerUnit = function (blocksInRange, units) {
        // only retain working days
        var workingBlocks = BurndownRate.limitToWorkingDays(blocksInRange);

        // calculate the total duration of all blocks
        var totalTime = _.reduce(workingBlocks, function (memo, block) {
            return memo + (block.start - block.end);
        }, 0);

        // calculate how long each unit will take
        return totalTime / units;
    };

    /**
     * Calculates the series given the timeline
     *
     * [{data: [[7, 60], [12, 0]]}, {data:[[7, 60], [10, 50], [12, 0]]}];
     */
    BurndownChartModel.calculateSeries = function () {
        var data = BurndownChartModel.timelineData;
        var timeline = data.timeline;
        if (_.isEmpty(timeline)) {
            return;
        }

        // fetch the timeline
        var startValue = timeline[0];

        // Need a guide line - start value of statistic -> 0
        var series = [];
        series.push(BurndownChartModel.calculateGuidelineSeries({
            startTime: data.startTime,
            startValue: startValue.values.estimate,
            endTime: data.endTime
        }));

        // we got two lines, a estimate burndown and time spent burnup
        var seriesIds = ['estimate'];

        // add time spent if we are in time tracking mode
        if (BurndownChartModel.isTimeTracking()) {
            seriesIds.push('timeSpent');
        }

        // we store the series points and the data associated with these points
        var seriesPoints = {};
        var seriesData = {};
        _.each(seriesIds, function (serieId) {
            seriesPoints[serieId] = [];
            seriesData[serieId] = [];
        });

        // process the timeline
        _.each(timeline, function (timelineEntry) {

            if (!timelineEntry.openCloseEntries) {

                // handle each serie separately
                _.each(seriesIds, function (serieId) {
                    // add a point for the current time with previous value, but only if delta is non-null
                    if (timelineEntry.deltas && _.isNumber(timelineEntry.deltas[serieId]) && timelineEntry.deltas[serieId] !== 0) {
                        // old value at given time
                        seriesPoints[serieId].push([timelineEntry.time, timelineEntry.values[serieId] - timelineEntry.deltas[serieId]]);
                        seriesData[serieId].push({});
                    }

                    // new value at given time
                    seriesPoints[serieId].push([timelineEntry.time, timelineEntry.values[serieId]]);
                    seriesData[serieId].push(timelineEntry);
                });

            }
        });

        // Add a point for now if the sprint hasn't been completed yet
        if (!data.completeTime) {
            var lastTimelineEntry = _.last(timeline);
            _.each(seriesIds, function (serieId) {
                seriesPoints[serieId].push([Math.max(data.startTime, data.now), lastTimelineEntry.values[serieId]]);
            });
        }

        // put in the series
        if (BurndownChartModel.isTimeTracking()) {
            series.push({
                id: 'timeSpent',
                data: seriesPoints['timeSpent'],
                color: '#14892c',
                label: AJS.I18n.getText('gh.issue.time')
            });
        }
        series.push({
            id: 'estimate',
            data: seriesPoints['estimate'],
            color: '#d04437',
            label: AJS.I18n.getText('gh.rapid.chart.burndown.actual.label')
        });

        var nonWorkingColour;
        if (BurndownChartView.wallboardMode) {
            nonWorkingColour = ChartColors.nonWorkingDaysWallboard;
        } else {
            nonWorkingColour = ChartColors.nonWorkingDays;
        }
        // Add an empty series for non-working days to populate legend
        series.push({
            id: 'markings',
            color: nonWorkingColour,
            data: [],
            label: AJS.I18n.getText('gh.rapid.chart.burndown.nonworkingdays.label')
        });

        // set up the y-axis data
        var maxValue = 0;
        _.each(seriesIds, function (serieId) {
            maxValue = Math.max(maxValue, data.maxValues[serieId] || 0);
        });
        BurndownChartModel.calculateYAxis(maxValue);

        // done filtering the issues
        BurndownChartModel.series = series;
        BurndownChartModel.seriesData = seriesData;
    };

    /**
     * Create a y-axis object (suitable for flot) to best represent this chart
     */
    BurndownChartModel.calculateYAxis = function (maxStatisticValue) {
        BurndownChartModel.yAxisData = FlotChartUtils.calculateYAxis(maxStatisticValue, BurndownChartModel.getStatistic());
    };


    /**
     * Get the data table content to display alongside the series
     */
    BurndownChartModel.getDataTable = function () {
        return BurndownChartModel.dataTable;
    };

    /**
     * Get the series to be displayed in the chart
     */
    BurndownChartModel.getSeries = function () {
        return BurndownChartModel.series;
    };

    /**
     * Get the Y axis
     */
    BurndownChartModel.getYAxis = function () {
        return BurndownChartModel.yAxisData;
    };

    /**
     * Get the data for a series given the series id
     */
    BurndownChartModel.getDataBySeries = function (seriesId) {
        return BurndownChartModel.seriesData[seriesId];
    };

    /**
     * Get the data entry for a specific index inside a series
     */
    BurndownChartModel.getDataBySeriesAtIndex = function (seriesId, index) {
        if (BurndownChartModel.seriesData[seriesId]) {
            return BurndownChartModel.seriesData[seriesId][index];
        }
    };

    /**
     * Find a statistic description given the seriesId.
     */
    BurndownChartModel.getStatisticDefinitionForSeries = function (seriesId) {
        var definitions = BurndownChartModel.getStatisticDefinitions();

        var match = _.find(definitions, function (definition) {
            return (seriesId === definition.valueId);
        });
        return match;
    };

    BurndownChartModel.getStatisticDefinitions = function () {
        return BurndownChartModel.dataTable.statistics;
    };

    /**
     * Get warning.
     */
    BurndownChartModel.getWarning = function () {
        return BurndownChartModel.rawData.warningMessage;
    };

    /**
     * Check if there are any data to show on a chart.
     */
    BurndownChartModel.isEmpty = function () {
        return Object.keys(BurndownChartModel.rawData.changes).length === 0;
    };

    /**
     * Renders all values in text form according to the given statistics
     */
    BurndownChartModel.renderDataTableValues = function (dataTable) {
        // render the dates, where the date is different
        var lastDate = '';
        _.each(dataTable.rows, function (row) {
            // render the date string
            var dateString = ChartUtils.renderUTCMillisAsDate(row.time);
            var recordedDateString;
            if (row.recordedTime) {
                recordedDateString = ChartUtils.renderUTCMillisAsDate(row.recordedTime);
            }

            // only show the date in the row if it is different from the previous row
            if (dateString !== lastDate || recordedDateString) {
                row.dateString = dateString;
                row.recordedDateString = recordedDateString;
                lastDate = dateString;
            }

            // process string versions of statistic values
            row.valuesText = {};
            _.each(_.keys(row.values), function (key) {
                row.valuesText[key] = BurndownChartModel.renderStatisticText(row.values[key]);
            });

            // for initial and complete rows, calculate statistic text for each issue
            if (row.initial || row.complete) {
                _.each(row.issues, function (issue) {
                    issue.valuesText = {};
                    _.each(_.keys(issue.values), function (key) {
                        issue.valuesText[key] = BurndownChartModel.renderStatisticText(issue.values[key]);
                    });

                    // add parent issue key
                    issue.parentKey = BurndownChartModel.getParentIssueKey(issue.key);
                });

                // sort issues to ensure sub tasks appear near their parent
                row.issues = _.sortBy(row.issues, function (issue) {
                    return issue.parentKey || issue.key;
                });
            } else if (!row.openClose) {
                row.deltasType = {};
                row.deltasText = {};
                _.each(_.keys(row.deltas), function (key) {
                    row.deltasText[key] = BurndownChartModel.renderStatisticText(row.deltas[key], true);
                    row.deltasType[key] = row.deltas[key] < 0 ? 'dec' : 'inc';
                });

                // add parent issue key
                row.parentKey = BurndownChartModel.getParentIssueKey(row.key);
            }
        });
    };

    /**
     * Renders a statistic value for use on tooltip
     */
    BurndownChartModel.renderStatisticText = function (statisticValue, abs) {
        return ChartUtils.renderStatisticText(statisticValue, abs, BurndownChartModel.getStatistic().renderer);
    };

    /**
     * Renders the tooltip for a given seriesId and dataPoint
     */
    BurndownChartModel.getTooltipData = function (seriesId, dataPoint) {
        // dataPoint is in reality a timeline entry
        var timelineEntry = dataPoint;

        var tooltipData = {};
        var statistic = BurndownChartModel.getStatisticDefinitionForSeries(seriesId);
        tooltipData.statistic = statistic;

        // Render an appropriate tooltip for sprint start/end
        if (timelineEntry.initial || timelineEntry.complete) {
            if (timelineEntry.initial) {
                tooltipData.initial = true;
                tooltipData.event = BurndownTableProducer.getSprintStartChangeEvent();
            } else {
                tooltipData.complete = true;
                tooltipData.event = BurndownTableProducer.getSprintCompleteChangeEvent();
            }
            tooltipData.issueCount = _.size(timelineEntry.issues);
            tooltipData.statisticName = statistic.name;
            tooltipData.statisticValue = BurndownChartModel.renderStatisticText(timelineEntry.values[statistic.valueId]);

        }

        // render a change tooltip
        else {
            tooltipData.changes = [];
            _.each(timelineEntry.issues, function (issue) {
                var issueChange = {};
                issueChange.key = issue.key;
                issueChange.statisticName = statistic.name;
                issueChange.statisticDelta = BurndownChartModel.renderStatisticText(issue.deltas[statistic.valueId]);
                issueChange.event = BurndownChartModel.tableHandler.getChangeEvent(issue);
                tooltipData.changes.push(issueChange);
            });
        }
        tooltipData.dateText = ChartUtils.renderUTCMillisAsDate(timelineEntry.time);
        return tooltipData;
    };

    BurndownChartModel.hasAnyWorkingDays = function () {
        return BurndownRate.getWorkingDays().length > 0;
    };

    return BurndownChartModel;
});
