/* global GH */
/**
 * Burndown Chart Controller
 * @module jira-agile/rapid/ui/chart/burndown-chart-controller
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/chart/burndown-chart-view
 * @requires module:jira-agile/rapid/ui/chart/burndown-report-chart-controller
 * @requires module:jira-agile/rapid/ui/chart/chart
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 * @requires module:jira-agile/rapid/ui/component/chart-picker
 * @requires module:jira-agile/rapid/ui/component/estimate-picker
 */
define('jira-agile/rapid/ui/chart/burndown-chart-controller', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var $ = require('jquery');
    var _ = require('underscore');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var BurndownChartView = require('jira-agile/rapid/ui/chart/burndown-chart-view');
    var BurndownReportChartController = require('jira-agile/rapid/ui/chart/burndown-report-chart-controller');
    var Chart = require('jira-agile/rapid/ui/chart/chart');
    var ChartPicker = require('jira-agile/rapid/ui/component/chart-picker');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var EstimatePicker = require('jira-agile/rapid/ui/component/estimate-picker');

    // GLOBALS... FIX ME
    var Ajax = GH.Ajax;
    var log = GH.log;
    var Logger = GH.Logger;
    var RapidBoardState = GH.RapidBoard.State;
    var RapidBoardUrlState = GH.RapidBoard.UrlState;

    /**
     * Chart view component.
     *
     * Loads chart data, then passes contorl to chart implementation for actual rendering
     */
    var BurndownChartController = {};

    BurndownChartController.id = 'burndownChart';

    BurndownChartController.displayed = false;
    BurndownChartController.rapidViewId = undefined;

    BurndownChartController.analytics = new AnalyticsTracker('gh.report.burndownChart');

    BurndownChartController.setRapidView = function (rapidViewData) {
        BurndownChartController.rapidViewId = rapidViewData.id;
        BurndownReportChartController.setRapidView(rapidViewData.id);
    };

    // Begin URL stuff

    BurndownChartController.getNormalizedFromInternal = function () {
        return {
            sprint: BurndownChartController.getSprintIdForUrl() || null,
            estimate: BurndownChartController.getEstimateIdForUrl() || null
        };
    };

    /**
     * Get the selected sprint id - used for display in the url
     */
    BurndownChartController.getSprintIdForUrl = function () {
        // check whether we got a selected sprint id
        var sprint = BurndownChartController.sprintPicker.getSelectedItem();
        if (sprint) {
            return sprint.id;
        }

        // fall back to what is stored
        var sprintId = BurndownChartController.sprintPicker.getStoredSelectedItemId();
        return sprintId !== -1 ? sprintId : null;
    };

    /**
     * Get the selected estimate id - used for display in the url
     */
    BurndownChartController.getEstimateIdForUrl = function () {
        // we only place the value in the URL if it has been explicitly stored.
        var estimateId = BurndownChartController.estimatePicker.getStoredSelectedItemId();
        return estimateId !== -1 ? estimateId : null;
    };

    BurndownChartController.toUrl = function (params) {
        var obj = {};
        if (params.sprint !== null) {
            obj.sprint = params.sprint;
        }
        if (params.estimate !== null) {
            obj.estimate = params.estimate;
        }
        return obj;
    };

    /**
     * Update state from normalized state
     * @param params
     */
    BurndownChartController.updateInternalFromNormalized = function (params) {
        BurndownChartController.setSelectedSprintFromUrlState(params.sprint);
        BurndownChartController.setSelectedEstimateFromUrlState(params.estimate);
    };

    /**
     * Set the selected sprint id as provided in the url
     */
    BurndownChartController.setSelectedSprintFromUrlState = function (sprint) {
        if (sprint) {
            BurndownChartController.sprintPicker.storeSelectedItemId(sprint);
        }
    };

    /**
     * Set the selected estimate id as provided in the url
     */
    BurndownChartController.setSelectedEstimateFromUrlState = function (estimate) {
        if (estimate) {
            BurndownChartController.estimatePicker.storeSelectedItemId(estimate);
        }
    };

    /**
     * Get normalized state for this chart type
     */
    BurndownChartController.getNormalizedFromUrl = function (params) {
        return {
            sprint: RapidBoardUrlState.normalizeNumberParam(params.sprint, null),
            estimate: RapidBoardUrlState.normalizeStringParam(params.estimate, null)
        };
    };

    // End url stuff


    BurndownChartController.isApplicable = function (rapidViewData) {
        return rapidViewData.sprintSupportEnabled;
    };

    BurndownChartController.init = function () {
        // create a new sprint picker
        BurndownChartController.sprintPicker = new ChartPicker('burndown.selectedSprintId');
        BurndownChartController.estimatePicker = new EstimatePicker('burndown.selectedEstimate', true);
    };

    BurndownChartController.show = function () {
        log('BurndownChartController.show', Logger.Contexts.ui);
        BurndownChartController.displayed = true;
        BurndownReportChartController.show();

        // stop here if haven't got a view
        if (!BurndownChartController.rapidViewId) {
            return;
        }

        // reset some data we collected on the way when the chart is being redrawn on window resize. Flot doesn't know
        // about resizing and we collect the data on every draw operation.
        var chartIdentifier = '.' + BurndownChartController.id;
        $(GH).unbind(chartIdentifier);
        $(GH).bind(Chart.EVENT_RESIZE_REDRAW + chartIdentifier, BurndownChartController.handleResizeEvent);

        // render the view
        BurndownChartController.loadChart();

        BurndownChartController.analytics.trigger('show');
    };

    BurndownChartController.hide = function () {
        log('BurndownChartController.hide', Logger.Contexts.ui);
        BurndownReportChartController.hide();

        BurndownChartController.displayed = false;

        // remove resize listener
        $(GH).unbind('.' + BurndownChartController.id);
        Chart.destroy(BurndownChartView.id);

        // unbind listeners
        $(GH).unbind('.' + BurndownChartController.id);
    };

    /**
     * Loads the chart data for a given chart type
     */
    BurndownChartController.loadChart = function () {
        BurndownChartView.renderChartLoading();

        // and listen to selection changes
        BurndownChartController.sprintPicker.bindToSelectedItemChanged(BurndownChartController.handleSprintChanged);

        BurndownReportChartController.setNonWorkingDaysShown(RapidBoardState.getNonWorkingDaysShown('burndown'));

        var rapidViewId = BurndownChartController.rapidViewId;

        BurndownChartController.loadSprintOptions(rapidViewId).done(function (data) {
            BurndownChartController.processSprintsData(data);
            if (data.sprints && data.sprints.length > 0) {
                BurndownChartController.estimatePicker.load(BurndownChartController.rapidViewId, BurndownChartController.handleEstimateChanged).done(function () {
                    // fetch the selected sprint
                    var selectedSprint = BurndownChartController.sprintPicker.getSelectedItem();
                    var estimateStatistic = BurndownChartController.estimatePicker.getSelectedItem();

                    // and start drawing the report for the selectedSprint
                    if (selectedSprint) {
                        BurndownReportChartController.setSprintId(selectedSprint.id);
                        BurndownReportChartController.setEstimationStatistic(estimateStatistic.id);
                        BurndownReportChartController.loadChartData();
                    } else {
                        // hide spinner, nothing to show
                        ChartView.hideSpinner();
                    }
                });
            }
        });

        // load the board config then process the data, then..
    };

    /**
     * Loads the sprints available to choose from and then calls off to process them
     */
    BurndownChartController.loadSprintOptions = function (rapidViewId) {
        // then issue the request
        return Ajax.get({
            url: '/sprintquery/' + rapidViewId,
            data: {
                includeHistoricSprints: false,
                includeFutureSprints: false
            }
        }, 'sprintOptions');
    };

    /**
     * Processes the returned sprints data then calls the next load data method.
     */
    BurndownChartController.processSprintsData = function (data) {
        // ignore if we are not displayed anymore
        if (!BurndownChartController.displayed) {
            return;
        }

        // organise the sprints
        data.sprints.reverse();

        // set the sprints onto the picker
        BurndownChartController.sprintPicker.setItems(data.sprints);

        // determine the default sprint
        BurndownChartController.sprintPicker.setDefaultItem(BurndownChartController.getDefaultSprintForPicker(data.sprints));

        // render the dropdown
        BurndownChartView.renderAndShowSprintPicker();

        if (!data.sprints || data.sprints.length === 0) {
            BurndownChartController.estimatePicker.shouldRender = false;
        }
        // update the url based on the selected sprint
        RapidBoardState.replaceState();
    };

    /******************************
     * Events
     ***************************/

    /**
     * Called whenever the sprint picker changed
     */
    BurndownChartController.handleSprintChanged = function () {
        if (!BurndownChartController.displayed) {
            return;
        }

        log('BurndownChartController.handleSprintChanged', Logger.Contexts.event);

        // push the changed sprint into the url,
        RapidBoardState.pushState();

        // update the selected id
        BurndownReportChartController.setSprintId(BurndownChartController.sprintPicker.getSelectedItem().id);

        // and load the report data again
        BurndownReportChartController.loadChartData();
    };

    BurndownChartController.getDefaultSprintForPicker = function (sprints) {
        // for a burndown chart, we want the first active sprint as the default
        var activeSprints = _.filter(sprints, function (sprint) {
            return !sprint.closed;
        });

        if (activeSprints.length > 0) {
            return _.first(activeSprints);
        }
        return _.first(sprints);
    };

    /**
     * Called whenever the sprint picker changed
     */
    BurndownChartController.handleEstimateChanged = function (selectedId) {
        if (!BurndownChartController.displayed) {
            return;
        }

        log('BurndownChartController.handleEstimateChanged', Logger.Contexts.event);

        //// push the changed sprint into the url
        RapidBoardState.pushState();

        BurndownReportChartController.setEstimationStatistic(selectedId);
        BurndownReportChartController.loadChartData();
    };

    BurndownChartController.handleResizeEvent = function (event, data) {
        if (data.id !== BurndownChartController.id) {
            return;
        }

        // reset real points
        Logger.log('Updating BurndownChart after window resize event', Logger.Contexts.ui);
        data.chart.realPointsPerSeries = {};

        // add non-working days to the legend - force to happen after chart redraws
        setTimeout(BurndownChartView.addNonWorkingDaysToLegend, 0);
    };

    return BurndownChartController;
});