/**
 * View Controller
 *
 * Keeps track of the current view stage and adapts the column accordingly.
 */

GH.RapidBoard.ViewController = new GH.Events();

(function () {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.RapidBoard.ViewController.analytics = new AnalyticsTracker('gh.assignedWork.tooltip');
})();

GH.RapidBoard.ViewController.hasSideBar = false;

GH.RapidBoard.ViewController.hasSubnavigator = false;

GH.RapidBoard.ViewController.shouldShowSubnavigator = function () {
    //If sidebar is not enabled OR the installed jira-projects-plugin does not contain the subnavigator then we should the old reports navigation.
    //If, however, sidebar is enabled AND the installed jira-projects-plugin contains the subnavigator, we hide the old reports navigation and show the subnavigator instead.
    return GH.RapidBoard.ViewController.hasSideBar && GH.RapidBoard.ViewController.hasSubnavigator;
};

/**
 * Initializes the view controller.
 */
GH.RapidBoard.ViewController.init = function () {
    // Check if Project-centric sidebar is active and AUI sidebar is available
    GH.RapidBoard.ViewController.hasSideBar = AJS.sidebar && AJS.Meta.get('sidebar-enabled');

    // Check if jira-projects-plugin is installed and the installed version contains the Subnavigator
    GH.RapidBoard.ViewController.hasSubnavigator = JIRA.Projects && typeof JIRA.Projects.Subnavigator === 'function';

    // find view dialog
    GH.Dialog.registerDialog('.js-view-find', GH.Dialog.FindView.openFindExistingViewDialog);

    // hide the top level elements managed by the view controller
    AJS.$('#ghx-rabid,#ghx-view-modes,#ghx-notify').css({ 'display': 'none' });

    // render the working modes
    AJS.$('#ghx-operations').append(GH.tpl.viewcontroller.renderOperationsBar());

    // Enable tooltips
    GH.Tooltip.tipsify({
        selector: '.ghx-avatar-img[data-tooltip]',
        context: document,
        html: true,
        className: 'ghx-tipsy-left ghx-avatar-tipsy'
    }).on('show', function (tipsy) {
        // Trigger analytics event for sprint assigned work only
        if (tipsy.$element.closest('.ghx-assigned-work-stats').length) {
            GH.RapidBoard.ViewController.analytics.trigger('show');
        }
    });

    var extraFieldTipsy = GH.Tooltip.tipsify({
        selector: '.js-issue:not(.ui-sortable-helper) .ghx-extra-field',
        context: document,
        html: true,
        className: 'ghx-extra-field-tipsy ghx-tipsy-left'
    });

    extraFieldTipsy.on('show', detectTipsyTruncation);

    function detectTipsyTruncation(e) {
        var $tipsy = e.$tip;
        var tipsyInner = $tipsy.children('.tipsy-inner').get(0);
        if (tipsyInner.scrollHeight > tipsyInner.clientHeight) {
            $tipsy.addClass('ghx-extra-field-tipsy-truncated');
        }
    }

    GH.RapidBoard.ViewController.sidebarController = new GH.RapidBoard.SidebarController();

    GH.RapidBoard.ViewController.initModes(GH.RapidBoard.ViewController.hasSideBar);

    AJS.$(GH).bind(GH.RapidBoard.UrlState.HISTORY_CHANGED, GH.RapidBoard.ViewController.updateModeUrls);

    // initialize keyboard shortcuts
    GH.IssueOperationShortcuts.init();

    // initialise detail view
    GH.DetailsView.init();

    // initialise presentation mode
    GH.PresentationMode.init();

    if (GH.RapidBoard.State.isKanbanBoard()) {
        var KanPlanAcknowledge = require("jira-agile/rapid/ui/kanplan/kanplan-acknowledge");
        KanPlanAcknowledge.init();
    }

    // resize handling
    AJS.$(GH).bind(GH.Layout.EVENT_DELAYED_RESIZE, GH.RapidBoard.ViewController.handleResizeEvent);
    AJS.$(document).on('aui-message-close', GH.RapidBoard.ViewController.handleResizeEvent);
    AJS.$(window).on('confluence.header-resized', GH.RapidBoard.ViewController.handleResizeEvent);

    // initialize Manage Sprint Permission help link handler
    AJS.$(document).on('click', '.manage-sprint-permission-help', function (e) {
        e.preventDefault();
        window.open(GH.HelpPaths.getHelpPath('manage.sprint.permission').url, '_blank');
    });
};

GH.RapidBoard.ViewController.initModes = function (hasSideBar) {
    if (!hasSideBar) {
        AJS.$('#ghx-view-modes').html(GH.tpl.viewcontroller.renderModes());

        // Enable tooltips
        GH.Tooltip.tipsify({ selector: '#plan-toggle' });
        GH.Tooltip.tipsify({ selector: '#work-toggle' });
        GH.Tooltip.tipsify({ selector: '#report-toggle' });

        // listener for mode toggle buttons
        AJS.$('#plan-toggle,#work-toggle,#report-toggle').bind('click', function (event) {
            // if they are trying to open the view in a separate tab, allow the native event handling
            if (!GH.MouseUtils.openInSameTab(event)) {
                return;
            }
            // Prevent default. We are going to do a pushState
            event.preventDefault();

            // ignore if disabled
            if (AJS.$(this).is(':disabled')) {
                return;
            }

            // ignore if already active
            if (AJS.$(this).hasClass('active')) {
                return;
            }

            var id = AJS.$(this).attr('id');
            var mode = id.substring(0, id.length - '-toggle'.length);
            GH.RapidBoard.ViewController.setMode(mode);
        });
    } else {
        GH.RapidBoard.ViewController.sidebarController.interceptSidebarItems();
        GH.RapidBoard.ViewController.sidebarController.reactToSidebarSizeChange();
    }
};

/**
 * Called whenever the selected view changes
 */
GH.RapidBoard.ViewController.update = function (rapidViewData) {
    GH.SprintRetrospectiveView.clearDialog();

    if (rapidViewData.invalidView) {
        // if errors, show errors
        GH.RapidBoard.ViewController.showInvalidViewError();
        AJS.$('#ghx-rabid,#ghx-view-modes').css({ 'display': 'none' });
        AJS.$('#ghx-notify').css({ 'display': '' });
    } else if (rapidViewData.id) {
        // if valid view, show it
        GH.RapidBoard.ViewController.showRapidView(rapidViewData);
        AJS.$('#ghx-rabid,#ghx-view-modes').css({ 'display': '' });
        AJS.$('#ghx-notify').css({ 'display': 'none' }).empty();
    } else {
        // else show welcome
        GH.RapidBoard.ViewController.welcome();
        AJS.$('#ghx-rabid,#ghx-view-modes').css({ 'display': 'none' });
        AJS.$('#ghx-notify').css({ 'display': '' });
    }
};

/**
 * Shows an error on page load
 */
GH.RapidBoard.ViewController.showInvalidViewError = function () {
    var messages = [{
        message: AJS.I18n.getText('gh.rapid.view.error.noview')
    }];
    if (GH.UserData.hasUser()) {
        messages.push({ message: AJS.I18n.getText('gh.rapid.view.error.noview.known', '<a class="js-view-find">', '</a>', '<a class="js-view-action-create">', '</a>') });
    } else {
        messages.push({ message: AJS.I18n.getText('gh.rapid.view.error.noview.anon', '<a href="' + GH.Navigation.getLoginUrl() + '">', '</a>') });
    }

    AJS.$('#ghx-notify').html(GH.tpl.viewcontroller.renderErrors({ errors: messages }));
};

/**
 * Renders a Welcome Screen
 */
GH.RapidBoard.ViewController.welcome = function () {
    // No Views template
    if (GH.UserData.hasUser()) {
        AJS.$('#ghx-notify').html(GH.tpl.board.x.renderNoViews({}));
    } else {
        AJS.$('#ghx-notify').html(GH.tpl.board.x.renderNoViewsAnonymous({ loginUrl: GH.Navigation.getLoginUrl() }));
    }
};

GH.RapidBoard.ViewController.showRapidView = function (rapidViewData) {
    AJS.$('#ghx-errors').empty();
    GH.RapidBoard.ViewController.updateModeUrls();

    // update the tab selector
    GH.RapidBoard.ViewController.updateTabSelector(rapidViewData);

    // tell all controllers about the new view
    GH.PlanController.setRapidView(rapidViewData);
    GH.WorkController.setRapidView(rapidViewData);
    GH.ReportController.setRapidView(rapidViewData);

    // set the mode, to ensure each controller knows whether it is displayed
    GH.RapidBoard.ViewController.setMode(GH.RapidBoard.ViewController.getMode(), true);

    // show the welcome message if available
    GH.PersistentMessages.WelcomeMessage.renderMessage();
    GH.PersistentMessages.CreateViewMessage.renderMessage();
};

//
// Tabs support
//
GH.RapidBoard.ViewController.validModes = { 'plan': true, 'work': true, 'report': true };
GH.RapidBoard.ViewController.wrmContext = { 'plan': 'gh-rapid-plan', 'work': 'gh-rapid-work', 'report': 'gh-rapid-report' };
GH.RapidBoard.ViewController.displayedMode = undefined;

GH.RapidBoard.ViewController.updateTabSelector = function (rapidViewData) {
    var isScrumBoard = rapidViewData.sprintSupportEnabled;
    var labels = GH.RapidBoard.ViewController.getTabLabels(isScrumBoard);

    AJS.$('#plan-toggle').attr({
        'data-tooltip': labels.plan.tooltip,
        'disabled': !isScrumBoard
    }).text(labels.plan.label);

    AJS.$('#work-toggle').attr({
        'data-tooltip': labels.work.tooltip
    }).text(labels.work.label);
};

/**
 * Get the appropriate labels for Agile mode switchers based on board type.
 *
 * @param isScrumBoard Is this a scrum board?
 * @returns {Object}
 */
GH.RapidBoard.ViewController.getTabLabels = function (isScrumBoard) {
    return {
        plan: {
            label: isScrumBoard ? AJS.I18n.getText('gh.rapid.planning.link.scrum.label') : AJS.I18n.getText('gh.rapid.planning.link.kanban.label'),
            tooltip: isScrumBoard ? AJS.I18n.getText('gh.rapid.planning.link.scrum.tip') : AJS.I18n.getText('gh.rapid.planning.link.kanban.tip')
        },
        work: {
            label: isScrumBoard ? AJS.I18n.getText('gh.rapid.pool.link.scrum.label') : AJS.I18n.getText('gh.rapid.pool.link.kanban.label'),
            tooltip: isScrumBoard ? AJS.I18n.getText('gh.rapid.pool.link.scrum.tip') : AJS.I18n.getText('gh.rapid.pool.link.kanban.tip')
        }
    };
};

/**
 * A mode that a rapid baord can be viewed in.
 * @typedef {("plan"|"work"|"report")} GH.Mode
 */

/**
 * @returns {GH.Mode}
 */
GH.RapidBoard.ViewController.getMode = function () {
    return GH.RapidBoard.State.getMode();
};

/**
 * Event indicating that a navigation is about to happen (or has _just_ started).
 *
 * @event GH.RapidBoard.ViewController#before:navigate
 * @property {GH.Mode} mode - The mode which is navigated to.
 */

/**
 * @param {GH.Mode} mode
 * @param {boolean} [replaceState=false] True if a history item _should not_ be added. This happens for full page loads
 *     and when the browser does a "back" navigation.
 * @fires GH.RapidBoard.ViewController#before:navigate
 */
GH.RapidBoard.ViewController.setMode = function setMode(mode, replaceState) {
    replaceState = !!replaceState;

    GH.SprintRetrospectiveView.clearDialog();
    // ensure valid mode
    if (!GH.RapidBoard.ViewController.validModes[mode]) {
        return;
    }

    var resources = WRM.require('wrc!' + GH.RapidBoard.ViewController.wrmContext[mode]);

    // only flip if we aren't already in given mode
    if (GH.RapidBoard.ViewController.displayedMode !== mode) {
        GH.RapidBoard.ViewController.trigger('before:navigate', { mode: mode });

        // set the mode
        GH.RapidBoard.State.setMode(mode);

        // update selector
        GH.RapidBoard.ViewController.updateSelector();

        // clear any updated messages
        GH.Notification.clearBoardUpdatedMessage();

        // clear current linked pages dialog
        GH.LinkedPagesController.hideCurrentDialog();

        // update the sidebar
        if (GH.RapidBoard.ViewController.hasSideBar) {
            GH.RapidBoard.ViewController.sidebarController.setSidebarState(mode);
            if (!replaceState) {
                // update the badges only if it is not a full page reload
                GH.RapidBoard.ViewController.sidebarController.updateBadges();
            }
        }

        resources.then(_showView);
        _pushState();

        AJS.$(GH).trigger('pool-column-updated');
    }

    function _showView() {
        // show corresponding view
        switch (mode) {
            case 'plan':
                GH.PlanController.init();
                GH.WorkController.hide();
                GH.ReportController.hide();
                AJS.$('#ghx-work,#ghx-controls-work,.ghx-sprint-meta').hide();
                AJS.$('#ghx-report,#ghx-controls-report').hide();
                AJS.$('#ghx-plan,#ghx-controls-plan').show();
                GH.PlanController.show();
                break;
            case 'report':
                GH.ReportController.init();
                GH.PlanController.hide();
                GH.WorkController.hide();
                AJS.$('#ghx-plan,#ghx-controls-plan').hide();
                AJS.$('#ghx-work,#ghx-controls-work,.ghx-sprint-meta').hide();
                AJS.$('#ghx-report,#ghx-controls-report').show();
                GH.ReportController.show();
                break;
            case 'work':
            /* falls through */
            default:
                GH.WorkController.init();
                GH.PlanController.hide();
                GH.ReportController.hide();
                AJS.$('#ghx-plan,#ghx-controls-plan').hide();
                AJS.$('#ghx-report,#ghx-controls-report').hide();
                AJS.$('#ghx-work,#ghx-controls-work,.ghx-sprint-meta').show();
                GH.WorkController.show();
                break;
        }
    }

    function _pushState() {
        // push state
        if (replaceState) {
            GH.RapidBoard.State.replaceState();
        } else {
            GH.RapidBoard.State.pushState();
        }
    }
};

GH.RapidBoard.ViewController.preload = function (viewId) {
    switch (GH.RapidBoard.ViewController.getMode()) {
        case GH.RapidBoard.State.VIEW_MODE.PLAN:
            // GH.RapidBoard.PlanController.fetchData(viewId);
            break;
        case GH.RapidBoard.State.VIEW_MODE.REPORT:
            // GH.RapidBoard.ReportController.fetchData(viewId);
            break;
        default:
            GH.WorkController.preload(viewId);
            break;
    }
};

GH.RapidBoard.ViewController.updateSelector = function () {
    AJS.$(".js-mode-selector .aui-button").removeClass('active');
    var mode = GH.RapidBoard.State.getMode();
    AJS.$('#' + mode + '-toggle').addClass('active');
};

GH.RapidBoard.ViewController.viewByMode = {
    'work': 'detail',
    'plan': 'planning',
    'report': 'reporting'
};

GH.RapidBoard.ViewController.updateModeUrls = function () {
    var rapidViewId = GH.RapidBoard.State.getRapidViewId();
    _.forEach(GH.RapidBoard.ViewController.validModes, function (alwaysTrue, modeName) {
        var url = GH.RapidBoard.getRapidBoardPageUrl(rapidViewId, GH.RapidBoard.ViewController.viewByMode[modeName]);
        AJS.$('#' + modeName + '-toggle').attr('href', url);
    });
};

GH.RapidBoard.ViewController.updateContentContainer = function () {
    GH.RapidBoard.ViewController.handleResizeEvent();
};

/**
 * Handles content area resizing
 */
GH.RapidBoard.ViewController.handleResizeEvent = function () {

    // TODO: this should be handled differently, but its far to big a mess to move it elsewhere atm.
    GH.PlanView.updateFixedElementPositioning();
    GH.WorkView.updateFixedElementPositioning();

    if (GH.RapidBoard.ViewController.hasSideBar) {
        AJS.sidebar('.aui-sidebar').reflow();
    }
};

/**
 *
 * @returns {jQuery} the visible content containers on the rapid board
 */
GH.RapidBoard.ViewController.contentContainers = function () {
    var $containers = AJS.$('#ghx-work, #ghx-plan, #ghx-report, #ghx-plan-group');
    return $containers.filter(function () {
        return this.style.display !== 'none';
    });
};

GH.RapidBoard.ViewController.collectPrintablePlanIssues = function () {
    return GH.PlanController.collectPrintableIssues();
};

GH.RapidBoard.ViewController.collectPrintableWorkIssues = function () {
    return GH.WorkController.collectPrintableIssues();
};