GH.RapidBoard.SidebarController = (function(
    _,
    ViewController
){
    'use strict';

    function SidebarController() {}

     // This map is used to associate a linkId (key) with a Agile mode (value)
    var linkToMode = {
        'com.pyxis.greenhopper.jira:project-sidebar-work-scrum': 'work',
        'com.pyxis.greenhopper.jira:project-sidebar-work-kanban': 'work',
        'com.pyxis.greenhopper.jira:global-sidebar-work-scrum': 'work',
        'com.pyxis.greenhopper.jira:global-sidebar-work-kanban': 'work',

        'com.pyxis.greenhopper.jira:project-sidebar-plan-scrum': 'plan',
        'com.pyxis.greenhopper.jira:project-sidebar-plan-kanban': 'plan',
        'com.pyxis.greenhopper.jira:global-sidebar-plan-scrum': 'plan',
        'com.pyxis.greenhopper.jira:global-sidebar-plan-kanban': 'plan',

        'com.pyxis.greenhopper.jira:global-sidebar-report': 'report',
        'com.atlassian.jira.jira-projects-plugin:report-page': 'report'
    };

    // This map is used to associate an Agile mode (key) with a set of possible links to highlight (values)
    var modeToLink = {
        'plan': [
            'com.pyxis.greenhopper.jira:project-sidebar-plan-scrum',
            'com.pyxis.greenhopper.jira:global-sidebar-plan-scrum',
            'com.pyxis.greenhopper.jira:project-sidebar-plan-kanban',
            'com.pyxis.greenhopper.jira:global-sidebar-plan-kanban'
        ],
        'work': [
            'com.pyxis.greenhopper.jira:project-sidebar-work-scrum',
            'com.pyxis.greenhopper.jira:project-sidebar-work-kanban',
            'com.pyxis.greenhopper.jira:global-sidebar-work-scrum',
            'com.pyxis.greenhopper.jira:global-sidebar-work-kanban'
        ],
        'report': [
            'com.atlassian.jira.jira-projects-plugin:report-page',
            'com.pyxis.greenhopper.jira:global-sidebar-report'
        ]
    };

    function sidebarItemPointsToAgileReport(sidebarItem) {
        return sidebarItem.find('a.aui-nav-item').attr('href').indexOf('RapidBoard.jspa') > -1;
    }

    function shouldUpdateAgileMode(mode, sidebarItem) {
        switch(mode) {
            case 'plan':
            case 'work':
                return true;
            case 'report':
                return sidebarItemPointsToAgileReport(sidebarItem);
            default:
                return false;
        }
    }

    function isSidebarAPIPresent() {
        return JIRA.API && JIRA.API.getSidebar;
    }

    function isSidebarFeatureDiscoveryPresent() {
        return JIRA.Projects && JIRA.Projects.Sidebar && JIRA.Projects.Sidebar.FeatureDiscovery;
    }

    SidebarController.prototype = {
        constructor: SidebarController,

        /**
         * Intercept the Sidebar navigation to load the Agile mode in client-side, avoiding
         * a page pop.
         */
        interceptSidebarItems: function() {
            // JIRA < 6.4 does not have a Sidebar
            if (!isSidebarAPIPresent()) {
                return;
            }

            JIRA.API.getSidebar().done(function (sidebarApi) {
                var navigationPanel = sidebarApi.getGroup('sidebar-navigation-panel');
                if (!navigationPanel) {
                    return;
                }

                navigationPanel.on('before:navigate', function(ev) {
                    var mode = linkToMode[ev.emitter.id];
                    if(shouldUpdateAgileMode(mode, ev.emitter.$el)) {
                        ViewController.setMode(mode);
                        ev.preventDefault();
                    }
                });
            });
        },

        /**
         * Sets the sidebar state to reflect the current mode.
         *
         * @param {string} mode 'plan', 'work' or 'report'
         */
        setSidebarState: function(mode) {
            // JIRA < 6.4 does not have a Sidebar
            if (!isSidebarAPIPresent()) {
                return;
            }

            JIRA.API.getSidebar().done(function (sidebarApi) {
                var navigationPanel = sidebarApi.getGroup('sidebar-navigation-panel');
                if (!navigationPanel) {
                    return;
                }

                var candidateLinkIds = modeToLink[mode];
                if (!candidateLinkIds) {
                    return;
                }

                // Selects the first itemID that exists in the sidebar
                _.any(candidateLinkIds, function (id) {
                    var item = navigationPanel.getItem(id);
                    if (item) {
                        item.select();
                        return true;
                    }
                });
            });
        },

        reactToSidebarSizeChange: function() {
            // JIRA < 6.4 does not have a Sidebar
            if (!isSidebarAPIPresent()) {
                return;
            }
            JIRA.API.getSidebar().done(function (sidebarApi) {
                var AUISidebar = sidebarApi.getAUISidebar();
                AUISidebar.on('collapse-end expand-end', function () {
                    GH.Layout.fireDelayedWindowResize();
                });
            });
        },

        updateBadges: function() {
            // JIRA < 6.4 does not have a Sidebar
            if (!isSidebarAPIPresent() || !isSidebarFeatureDiscoveryPresent()) {
                return;
            }

            JIRA.API.getSidebar().done(function () {
                JIRA.Projects.Sidebar.FeatureDiscovery.updateBadges();
            });
        }
    };

    return SidebarController;
}(
    _,
    GH.RapidBoard.ViewController
));