define("jira-agile/rapid/state/state", ["jquery"], function ($) {
    "use strict";

    var State = {};

    /**
     * Add a browser history entry for the given viewState. The viewState object
     * will be treated as a map and all properties will be added to the URL, like so:
     *
     * viewState = { key : value, key2 : value 2}
     * => URL?key=value&key2=value2
     *
     * Keys and values will be URI encoded, please do not do that upfront.
     *
     * The viewState is stored against the browser history and can be read by using
     * State.getViewState()
     *
     * The title parameter is optional.
     */

    State.push = function (viewState, title) {
        var uri = State.toUri(viewState);
        if (typeof title === 'undefined') {
            title = document.title;
        }
        History.pushState(viewState, title, uri);
        $(document).trigger("gh.virtualpageview");
    };

    /**
     * Replaces the current state with the provided one.
     * Use this function when you don't want the old state to be available anymore.
     */
    State.replace = function (viewState, title) {
        var uri = State.toUri(viewState);
        if (typeof title === 'undefined') {
            title = document.title;
        }
        History.replaceState(viewState, title, uri);
    };

    /**
     * Get the current viewState stored against the history entry.
     */
    State.getViewState = function () {
        return History.getState().data;
    };

    /**
     * Return the hash parameters in the URL as an Object, or false if there are none. A page can,
     * based on this, decide if it needs to update the data model again before rendering (for HTML4
     * browsers).
     *
     * HTML5-capable browsers should not use hash params in the first place, but pushState, so we
     * avoid a browser switch here.
     */
    State.getHashParams = function () {
        var url = window.location.href;
        return url.indexOf('#') < 0 ? false : State.toViewState(url);
    };

    /**
     * Transforms a URI back into a viewState object
     */
    State.toViewState = function (uri) {
        // save the work if we haven't got any parameters
        if (uri.lastIndexOf('?') < 0) {
            return {};
        }

        //Create the uri object and copy its query params into a new object
        var viewState = State.getStateFromUri(uri);

        // ensure we don't keep the _suid
        delete viewState['_suid'];

        return viewState;
    };

    /**
     * Convert the viewState object into a URI snippet
     */
    State.toUri = function (viewState) {
        var uriObject = AJS.parseUri(document.location.href);
        uriObject.anchor = "";
        uriObject.queryKey = $.extend({}, viewState);

        return uriObject.toString();
    };

    /**
     * Register a history state change listener
     */
    State.registerStateChangeListener = function (listener) {
        History.Adapter.bind(window, 'statechange', listener);
    };

    /**
     * Extract the state from the URI (or URL, for that matter). Basically, this needs to do two different
     * things, based on if we have HTML5 url (with state in query string) or an HTML4 url (with state in
     * the anchor). If we have state in both parts, we want the anchor state to dominate (as this means
     * we've taken a html5 url, fed it into a html4 browser, changed state and so the state in the query
     * string is stale)
     */
    State.getStateFromUri = function (uri) {
        var uriObject = AJS.parseUri(uri);
        // if there's state in the anchor, everything before it is basically useless
        if (uriObject.anchor) {
            // Parse the url after the anchor, which should give us the real query parameters
            uriObject = AJS.parseUri(uriObject.anchor);
        }
        //regardless of what uri was, we've got the right state in uriObject.queryKey
        return uriObject.queryKey;
    };
    return State;
});

AJS.namespace("GH.State", null, require("jira-agile/rapid/state/state"));