
/* globals GH, _ */

/**
 * Handles the selection issues inside a list of issues.
 *
 * The view specific functionality is provided by a provider
 */

/** Issue model this manager works with. */
GH.IssueSelectionManagerModel = {
    /** Is the given key "valid", as in visible */
    isIssueValid: function(issueKey) { return false; },
    /** Get the index for a given issue. */
    getIssueIndex: function(issueKey) { return -1; },
    /** Returns true if the passed issueKey can be added to the selection. */
    canAddToSelection: function(selectedIssueKeys, issueKey) { return false; },
    /** Get an issue range. The issues returned should be in the order as specified by from and to. */
    getIssueRange: function(fromKey, toKey) { return [fromKey, toKey]; }
};


/**
 * Create a new selection manager
 *
 * @param model the model used by this manager
 * @param selectedIssuesStateKey the key to use to store the selected issues against
 */
GH.IssueSelectionManager = function(model, selectedIssuesStateKey) {
    this.model = model;
    this.selectedIssuesStateKey = selectedIssuesStateKey;

    /** Holds the issues that got clicked on without the shift key being pressed. */
    this.lastNonShiftSelectedKey = undefined;

    /** Holds the issues that got clicked on with the shift key being pressed. */
    this.lastShiftSelectedKey = undefined;
};


/**
 * Validate the current issue selection, removing any invalid issue if set
 */
GH.IssueSelectionManager.prototype.validateCurrentSelection = function () {
    var selectedIssueStack = this.getSelectedIssueKeys();

    // we want to check for validity as well as whether the issues can be selected together.
    // do it in reverse order to keep the primary key if possible
    var stackInReverse = selectedIssueStack.slice(0).reverse();
    var validKeys = [];
    _.each(stackInReverse, function(issueKey) {
        if (this.model.isIssueValid(issueKey)) {
            if (validKeys.length === 0 || this.model.canAddToSelection(validKeys, issueKey)) {
                validKeys.push(issueKey);
            }
        }
    }, this);
    validKeys.reverse();

    // check whether things have changed
    if (_.isEqual(selectedIssueStack, validKeys)) {
        return false;
    }

    // update stack and signal change
    this.storeSelectedIssueStack(validKeys);
    return true;
};

/**
 * Select the given issue
 */
GH.IssueSelectionManager.prototype.selectIssue = function selectIssue(issueKey) {
    var issueKeys = issueKey ? [issueKey] : [];
    this._replaceSelectedIssueKeys(issueKeys);
    this._setLastNonShiftSelectedKey(issueKey);
};

/**
 * Get the selected issues in form of an array of issue keys. The order is according to the insertion order.
 */
GH.IssueSelectionManager.prototype.getSelectedIssueKeys = function() {
    // we keep the selected issues in the session storage, avoids duplicating data in several places
    return GH.BoardState.getPerViewValue(this.selectedIssuesStateKey, []);
};

GH.IssueSelectionManager.prototype.storeSelectedIssueStack = function(selectedIssueStack) {
    GH.BoardState.setPerViewValue(this.selectedIssuesStateKey, selectedIssueStack);
};

/**
 * Get the selected issues in the order defined by the provider
 */
GH.IssueSelectionManager.prototype.getSelectedIssuesInOrder = function() {
    // sort by index, which is returned directly by getIssueIndexByKey
    var selectedIssueStack = this.getSelectedIssueKeys();
    return _.sortBy(selectedIssueStack, this.model.getIssueIndex.bind(this.model));
};

/**
 * Is the passed issue key currently in the selection
 */
GH.IssueSelectionManager.prototype.isInSelection = function(issueKey) {
    // TODO FIX ME: _.contains has terrible performance, Use a Set or an OrderedSet instead (Collection.js)
    var selectedIssueStack = this.getSelectedIssueKeys();
    return _.contains(selectedIssueStack, issueKey);
};

/**
 * Are any issues currently selected
 */
GH.IssueSelectionManager.prototype.hasSelection = function() {
    var selectedIssueStack = this.getSelectedIssueKeys();
    return !_.isEmpty(selectedIssueStack);
};

/**
 * Add an issue key to the selection
 */
GH.IssueSelectionManager.prototype._addToSelection = function(issueKey) {
    // only add if we are not in the selection already
    if(!this.isInSelection(issueKey)) {
        var selectedIssueStack = this.getSelectedIssueKeys();
        selectedIssueStack.push(issueKey);
        this.storeSelectedIssueStack(selectedIssueStack);
    }
};

/**
 * Prepend an issue key to the selection
 */
GH.IssueSelectionManager.prototype._prependToSelection = function(issueKey) {
    // only add if we are not in the selection already
    if(!this.isInSelection(issueKey)) {
        var selectedIssueStack = this.getSelectedIssueKeys();
        selectedIssueStack.unshift(issueKey);
        this.storeSelectedIssueStack(selectedIssueStack);
    }
};

/**
 * Adds a list of issues to the selection
 */
GH.IssueSelectionManager.prototype._addSelectedIssueKeys = function(issueKeys) {
    var manager = this;
    _.each(issueKeys, function(issueKey){
        manager._addToSelection(issueKey);
    });
};

/**
 * Prepends a list of issues to the selection
 */
GH.IssueSelectionManager.prototype._prependSelectedIssueKeys = function(issueKeys) {
    var manager = this;
    _.each(issueKeys, function(issueKey){
        manager._prependToSelection(issueKey);
    });
};

/**
 * Remove an issue key from the selection
 */
GH.IssueSelectionManager.prototype._removeFromSelection = function(issueKey) {
    var selectedIssueStack = this.getSelectedIssueKeys();
    selectedIssueStack = _.without(selectedIssueStack, issueKey);
    this.storeSelectedIssueStack(selectedIssueStack);
};

/**
 * Removes a list of issues from the selection
 */
GH.IssueSelectionManager.prototype.removeSelectedIssueKeys = function(issueKeys) {
    // remove each issue
    var manager = this;
    _.each(issueKeys, function(issueKey){
        manager._removeFromSelection(issueKey);
    });
};

/**
 * Sets the list of issues in the current selection.
 */
GH.IssueSelectionManager.prototype._replaceSelectedIssueKeys = function(issueKeys) {
    this._clearSelectedIssueKeys();
    this._addSelectedIssueKeys(issueKeys);
};

/**
 * Clears the list of currently selected issues
 */
GH.IssueSelectionManager.prototype._clearSelectedIssueKeys = function() {
    this.storeSelectedIssueStack([]);
};

/**
 * Get the selected issue key
 */
GH.IssueSelectionManager.prototype.getSelectedIssueKey = function() {
    // the last issue inside the selected list is regarded as the selected one
    var selectedIssueStack = this.getSelectedIssueKeys();
    var selectedKey = selectedIssueStack[selectedIssueStack.length - 1];
    return selectedKey || null;
};

/**
 * Get the last non shift issue key.
 */
GH.IssueSelectionManager.prototype._getLastNonShiftSelectedKey = function() {
    if (!this.lastNonShiftSelectedKey) {
        // see whether we got a selection, use that key instead
        var key = this.getSelectedIssueKey();
        if (key && this.model.isIssueValid(key)) {
            this._setLastNonShiftSelectedKey(key);
            return key;
        }
    }
    return this.lastNonShiftSelectedKey;
};

/**
 * Set the last non shift selected issue key
 */
GH.IssueSelectionManager.prototype._setLastNonShiftSelectedKey = function(issueKey) {
    this.lastNonShiftSelectedKey = issueKey;
    this.lastShiftSelectedKey = undefined;
};

GH.IssueSelectionManager.prototype._getLastShiftSelectedKey = function() {
    return this.lastShiftSelectedKey;
};

GH.IssueSelectionManager.prototype._setLastShiftSelectedKey = function(issueKey) {
    this.lastShiftSelectedKey = issueKey;
};

/**
 * Handles clicking on an issue card in order to select it.
 */
GH.IssueSelectionManager.prototype.handleIssueClick = function(metaKeys, issueKey) {
    var metaDown = GH.MouseUtils.isMetaPressed(metaKeys);

    var multiSelectType = GH.IssueSelectionManager.getMultiSelectType(metaKeys);

    // ensure the issue is valid
    if (!this.model.isIssueValid(issueKey)) {
        return false;
    }

    // ensure the issue can be added to the selection (shift and/or meta key)
    if (multiSelectType) {
        if (!this.model.canAddToSelection(this.getSelectedIssueKeys(), issueKey)) {
            return false;
        }
    }

    // easiest case, neither shift nor ctrl down
    if (!multiSelectType) {
        this.selectIssue(issueKey);
    }

    // slightly more complicated, ctrl without shift
    else if (multiSelectType === GH.IssueSelectionManager._META_MULTI_SELECT) {
        // remove issue from selection if already selected
        if(this.isInSelection(issueKey)) {
            this.removeSelectedIssueKeys([issueKey]);
        }
        // add to selection otherwise
        else {
            this._addSelectedIssueKeys([issueKey]);
        }
        this._setLastNonShiftSelectedKey(issueKey);
    }

    // shift, mother of all selections
    else {
        // fetch the last selected key that was a non-shift selection
        var lastClickedIssueKey = this._getLastNonShiftSelectedKey();

        // if we have no selected issues, or we don't have a last non-shift key simply select the issue
        var selectedIssueStack = this.getSelectedIssueKeys();
        if (selectedIssueStack.length === 0 || !lastClickedIssueKey || !this.model.isIssueValid(lastClickedIssueKey)) {
            this.selectIssue(issueKey);
        }

        else {
            // calculate the issues between the current and the recent selection
            var issueRangeToAdd = this.model.getIssueRange(lastClickedIssueKey, issueKey);
            if(metaDown) {
                // also calculate the old range, as we have to remove issues added in the previous shift selection
                var issueRangeToRemove = [];
                var lastShiftSelectedKey = this._getLastShiftSelectedKey();
                if(lastShiftSelectedKey) {
                    issueRangeToRemove = this.model.getIssueRange(lastClickedIssueKey, lastShiftSelectedKey) || [];

                    // ensure no new issues are contained in the issues to remove
                    issueRangeToRemove = _.without(issueRangeToRemove, issueRangeToAdd);
                }

                // remove the old range and add the new range to the selection
                this.removeSelectedIssueKeys(issueRangeToRemove);
                this._addSelectedIssueKeys(issueRangeToAdd);
            } else {
                // replace whatever is currently selected with the new selection
                this._replaceSelectedIssueKeys(issueRangeToAdd);
            }
        }
        this._setLastShiftSelectedKey(issueKey);
    }

    // signal change
    return true;
};

GH.IssueSelectionManager._NO_MULTI_SELECT = 0;
GH.IssueSelectionManager._META_MULTI_SELECT = 1;
GH.IssueSelectionManager._SHIFT_MULTI_SELECT = 2;

GH.IssueSelectionManager.getMultiSelectType = function(metaKeys) {
    var isMetaDown = GH.MouseUtils.isMetaPressed(metaKeys);
    var isShiftDown = GH.MouseUtils.isShiftPressed(metaKeys);
    var multiSelectType = GH.IssueSelectionManager._NO_MULTI_SELECT;

    if(isShiftDown) {
        multiSelectType = GH.IssueSelectionManager._SHIFT_MULTI_SELECT;
    } else if(isMetaDown && !isShiftDown) {
        multiSelectType = GH.IssueSelectionManager._META_MULTI_SELECT;
    }

    return multiSelectType;
}
