/**
 * Manages issue operation shortcuts
 */
GH.IssueOperationShortcuts = {};

/** Selected issue overwrite */
GH.IssueOperationShortcuts.overrideSelectedIssue = undefined;

(function() {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.IssueOperationShortcuts.analytics = function(categorySuffix) {
        return new AnalyticsTracker('gh.issueaction.' + categorySuffix);
    };
})();

/**
 * Registers the "return" handler that navigates to the view issue page
 */
GH.IssueOperationShortcuts.init = function() {

    GH.IssueOperationShortcuts.bindEnter();

    // analytics for keyboard shortcuts we care about
    AJS.$(document).delegate('.js-issueaction-container-dot .js-issueaction', 'click', function(e) {
        var actionId = AJS.$(e.currentTarget).attr('id');
        GH.IssueOperationShortcuts.trackKeyboardShortcuts(actionId);
    });

    // analytics for dot dialog actions
    AJS.$(document).delegate('#issueactions-suggestions a.aui-list-item-link', 'click', function(e) {
        var aClass = AJS.$(e.currentTarget).attr('class');
        if (aClass.indexOf('issueaction-') >= 0) {
            // note: workflows do not have distinct 'actionIds' -- they are all 'workflow-transition'
            var actionId = aClass.replace(/.*issueaction-([a-z0-9_-]+).*/, "$1");
            GH.IssueOperationShortcuts.trackDotDialogAction(actionId);
        }
    });

    AJS.bind("remove.dialog", function() {
        // ensure keyboards shortcuts are rebound when dialogs are closed
        // we remove them for the wizard
        GH.IssueOperationShortcuts.bindEnter();
    });
};

GH.IssueOperationShortcuts.bindEnter = function() {
    /**
     * Handles the return key, which should act like the o key (view issue)/
     */
    AJS.$(document).bind("keydown", function(e) {
        if (e.keyCode == '13') {
            var target = e.target;
            // On different browser the originalTarget is different, but all of these are impossible for a user to trigger.
            if (target === undefined || target.nodeName === "HTML" || target.nodeName === "BODY" || target == document) {
                GH.IssueOperationShortcuts.viewSelectedIssue();
            }
        }
    });
};

/**
 * Overrides the selected issue temporarily.
 * @param object issue of form { id: 12345, key: ='PRO-1'}
 */
GH.IssueOperationShortcuts.setOverrideSelectedIssue = function(issue) {
    GH.IssueOperationShortcuts.overrideSelectedIssue = issue;
};

/**
 * Clears the selected issue override.
 */
GH.IssueOperationShortcuts.clearOverrideSelectedIssue = function() {
    GH.IssueOperationShortcuts.overrideSelectedIssue = undefined;
};

/**
 * Is the override currently active.
 */
GH.IssueOperationShortcuts.isOverrideSelectedIssueActive = function() {
    return !!GH.IssueOperationShortcuts.overrideSelectedIssue;
};

/**
 * Returns the issue key of the selected issue
 */
GH.IssueOperationShortcuts.getSelectedIssueKey = function() {
    if (GH.IssueOperationShortcuts.overrideSelectedIssue) {
        return GH.IssueOperationShortcuts.overrideSelectedIssue.key;
    } else if(GH.WorkController.isActive()) {
        return GH.WorkSelectionController.getSelectedIssueKey();
    } else if (GH.PlanController.isActive()) {
        return GH.BacklogSelectionController.getSelectedIssueKey();
    }
};

/**
 * Returns the issue id of the selected issue
 */
GH.IssueOperationShortcuts.getSelectedIssueId = function() {
    if (GH.IssueOperationShortcuts.overrideSelectedIssue) {
        return GH.IssueOperationShortcuts.overrideSelectedIssue.id;
    } else if(GH.WorkController.isActive()) {
        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();
        if (issueKey) {
            return GH.GridDataController.getModel().getIssueIdForKey(issueKey);
        }
    } else if (GH.PlanController.isActive()) {
        var issueKey = GH.BacklogSelectionController.getSelectedIssueKey();
        if (issueKey) {
            return GH.BacklogModel.getIssueIdForKey(issueKey);
        }
    }
};

/**
 * View issue action
 */
GH.IssueOperationShortcuts.viewSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (issueKey) {
        // explicitly track this event
        GH.IssueOperationShortcuts.trackKeyboardShortcuts('open-issue');

        var url = GH.Ajax.CONTEXT_PATH+'/browse/'+issueKey;
        window.location.href = url;
    }
};

/**
 * Edit issue action
 */
GH.IssueOperationShortcuts.editSelectedIssue = function() {
    var issueId = GH.IssueOperationShortcuts.getSelectedIssueId();
    if (issueId) {
        AJS.$('a.issueaction-edit-issue').trigger('click');
    }
};

/**
 * Assign issue action
 */
GH.IssueOperationShortcuts.assignSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (issueKey) {
        AJS.$('.js-issueaction-container-dot a.issueaction-assign-issue').trigger('click');
    }
};

/**
 * Assign to me issue action
 */
GH.IssueOperationShortcuts.assignSelectedIssueToMe = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (issueKey) {
        AJS.$('.js-issueaction-container-dot a.issueaction-assign-to-me').trigger('click');
    }
};

/**
 * Comment action
 */
GH.IssueOperationShortcuts.commentSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (issueKey) {
        var $detailViewAddCommentAnchor = AJS.$('#ghx-detail-view #addcomment a[name=add-comment]');

        if ($detailViewAddCommentAnchor.length > 0) {
            $detailViewAddCommentAnchor.trigger('click');
        } else {
            AJS.$('.js-issueaction-container-dot a.issueaction-comment-issue').trigger('click');
        }
    }
};

/**
 * Attach Files action
 */
GH.IssueOperationShortcuts.attachFilesSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (issueKey) {
        AJS.$('.js-issueaction-container-dot a.issueaction-attach-file').trigger('click');
    }
};

/**
 * Edit labels action
 */
GH.IssueOperationShortcuts.editSelectedIssueLabels = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (issueKey) {
        AJS.$('.js-issueaction-container-dot a.issueaction-edit-labels').trigger('click');
    }
};

/**
 * Delete issue action
 */
GH.IssueOperationShortcuts.deleteSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueId();
    if (issueKey) {
        AJS.$('.js-issueaction-container-dot a.issueaction-delete-issue').trigger('click');
    }
};


/**
 * Archive issue action
 */
GH.IssueOperationShortcuts.archiveSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueId();
    if (issueKey) {
        AJS.$('a.issueaction-archive-issue').trigger('click');
    }
};

/**
 * Link issue action
 */
GH.IssueOperationShortcuts.linkSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueId();
    if (issueKey) {
        AJS.$('.js-issueaction-container-dot a.issueaction-link-issue').trigger('click');
    }
};

/**
 * Open the create subtask dialog
 */
GH.IssueOperationShortcuts.createSubtask = function() {
    var xsrfToken = AJS.$('#atlassian-token').attr('content');
    var url = GH.Ajax.CONTEXT_PATH+'/secure/CreateSubTaskIssue!default.jspa?parentIssueId={0}'
        + '&atl_token='
        + xsrfToken
        + '&returnUrl='; // not used in our case
    var link = '<a href="'+url+'" class="issueaction-create-subtask" style="display:none"></a>';
    var $link = AJS.$(link);
    AJS.$('#gh').append($link);
    $link.click();
    $link.remove();
};

/**
 * Log work issue action
 */
GH.IssueOperationShortcuts.logWorkOnSelectedIssue = function() {
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (issueKey) {
        AJS.$('.js-issueaction-container-dot a.issueaction-log-work').trigger('click');
    }
};

/**
 * Show operations dialog
 */
GH.IssueOperationShortcuts.showOperationsDialog = function() {
    // ensure an issue is selected
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();
    if (!issueKey) {
        return;
    }

    JIRA.Dialogs.issueActions.show();
};

GH.IssueOperationShortcuts.trackKeyboardShortcuts = function(actionId) {
    GH.IssueOperationShortcuts.analytics('keyboardshortcuts').trigger(actionId); // SAFE
};

GH.IssueOperationShortcuts.trackDotDialogAction = function(actionId) {
    GH.IssueOperationShortcuts.analytics('dotdialog').trigger(actionId); // SAFE
};
